-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   GuiDoubleListbox Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

-- this implements the double-listbox-window

-- needed: an init-function
--         an operate-function

-- programming TclHaskell:
--   init composes all elements together
--   operate reads and writes messages to the CHD

module CHD.GuiDoubleListbox
  (
    initDoubleListbox,
    operateDoubleListbox
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import CHD.GuiMsgChan
import CHD.DebugMsgChan
import CHD.Environment
import CHD.GuiWindow
import CHD.BaseTypes
import CHD.BaseFunctions

import Tcl


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

initDoubleListbox :: GuiWindow -> GUI () -> GUI ()
initDoubleListbox confWin windowFinalizer = do
-- new Window
  myWindow <- window []
  title myWindow ("CHD - " ++ (show confWin))
  bind myWindow "<Destroy>" (do
				interactGui (CloseGuiWin confWin)
				windowFinalizer)

-- new Grid
  windowGrid <- frame myWindow []

-- create Elements
  enableLabel <- case confWin of
    VisualizationConfigWindow -> label myWindow [text "Visualize:"]
    ActionBreakConfigWindow -> label myWindow [text "Continue:"]
    _ -> label myWindow [text "Enable:"]
  disableLabel <- case confWin of
    VisualizationConfigWindow -> label myWindow [text "Hide:"]
    ActionBreakConfigWindow -> label myWindow [text "Break:"]
    _ -> label myWindow [text "Disable:"]
  enableListbox <- listbox myWindow [ takefocus True, 
				      background "LightGoldenRod"]
  disableListbox <- listbox myWindow [takefocus True]
  enableScroll <- vscroll enableListbox []
  disableScroll <- vscroll disableListbox []
  let listboxes = (enableListbox, disableListbox)
  enableButton <- button myWindow [text "<-",
			 command (enableMove confWin listboxes)]
  disableButton <- button myWindow [text "->", 
			  command (disableMove confWin listboxes)]

-- pack elements in grid
  gridAdd enableLabel (0,0) [widthX 2, gfillX, ginFrame windowGrid]
  gridAdd enableListbox (0,1) [heightY 5, gfillXY, ginFrame windowGrid]
  gridAdd enableScroll (1,1) [heightY 5, gfillY, ginFrame windowGrid]
  gridAdd enableButton (2,4) [ginFrame windowGrid]
  gridAdd disableButton (2,2) [ginFrame windowGrid]
  gridAdd disableLabel (4,0) [widthX 2, gfillX, ginFrame windowGrid]
  gridAdd disableListbox (4,1) [heightY 5, gfillXY, ginFrame windowGrid]
  gridAdd disableScroll (5,1) [heightY 5, gfillY, ginFrame windowGrid]

-- pack grid in window
  packAdd windowGrid [packV, fillXY]

-- get defaultColor
  defaultColor <- cget disableListbox background

-- focus on enableListbox
  focus enableListbox

-- bindings to move Focus
  let moveFocus = (\unfocusLB focusLB -> do
	focus focusLB
	cset unfocusLB [background defaultColor]
	cset focusLB [background "LightGoldenRod"])
      moveFocusEnable = moveFocus disableListbox enableListbox
      moveFocusDisable = moveFocus enableListbox disableListbox
  bind enableListbox "<Key-Tab>" moveFocusDisable
  bind disableListbox "<1>" moveFocusDisable
  bind disableLabel "<1>" moveFocusDisable
  bind disableListbox "<Key-Tab>" moveFocusEnable
  bind enableListbox "<1>" moveFocusEnable
  bind enableLabel "<1>" moveFocusEnable

-- key-bindings: arrows left/right moves element
  let moveToEnable = enableMove confWin listboxes
      moveToDisable = disableMove confWin listboxes
  bind myWindow "<Key-Left>" moveToEnable
  bind disableListbox "<Double-1>" moveToEnable
  bind myWindow "<Key-Right>" moveToDisable
  bind enableListbox "<Double-1>" moveToDisable

-- telling Debugger & Gui
  interactGui (OpenGuiWin confWin (operateDoubleListbox listboxes))
  proc $ interactCHD (GetInfoGuiWin confWin)

  return ()  


enableMove :: GuiWindow -> (Listbox, Listbox) -> GUI ()
enableMove confWin (eLB,dLB) =
  let message = case confWin of
        VisualizationConfigWindow -> 
	  (\name -> interactCHD $ Visualize (parseObject name) True)
	ActionBreakConfigWindow ->
	  (\name -> interactCHD $ BreakAction (read name) NoTime)
	_ -> (\_ -> return ())
      in
  transfer dLB eLB message


disableMove :: GuiWindow -> (Listbox, Listbox) -> GUI ()
disableMove confWin (eLB,dLB) =
  let message = case confWin of
        VisualizationConfigWindow -> 
	  (\name -> interactCHD $ Visualize (parseObject name) False)
	ActionBreakConfigWindow ->
	  (\name -> interactCHD $ BreakAction (read name) EveryTime)
	_ -> (\_ -> return ())
      in
  transfer eLB dLB message


transfer :: Listbox -> Listbox -> (String -> IO ()) -> GUI ()
transfer fromLB toLB messageConstructor = do
  items <- getListboxSelection fromLB
  list <- foldM
    (\result num -> do 
      let pos = LIndex num
      elems <- getListboxEntries fromLB pos pos
      return (elems ++ result)) 
    [] 
    items 
  mapM_ (removeListboxElement fromLB) list
  if (list == [])
    then return ()
    else do
      let (item:_) = items
      selectListboxElement fromLB (LIndex item)
      mapM_ (\elem -> proc $ messageConstructor elem) list
      
 

operateDoubleListbox :: (Listbox, Listbox) -> GuiWinMsg -> GUI ()
operateDoubleListbox (enableLB,disableLB) msg = 
  case msg of
    InsertListbox key list -> do
      case key of
        EnableListbox -> mapM_ (insertListboxElement enableLB) list
	DisableListbox -> mapM_ (insertListboxElement disableLB) list
	_ -> return ()
    RemoveListbox key list -> do
      case key of
        EnableListbox -> mapM_ (removeListboxElement enableLB) list
	DisableListbox -> mapM_ (removeListboxElement disableLB) list
	_ -> return ()
    _ -> return ()



selectListboxElement :: Listbox -> LIndex -> GUI ()
selectListboxElement listbox pos = do
  listboxMoveToSee listbox pos
  clearListboxSelection listbox (LIndex 0) LIndexEnd
  addListboxSelection listbox pos pos
--  setListboxSelectionAnchor listbox pos



insertListboxElement :: Listbox -> String -> GUI LIndex
insertListboxElement listbox elem = do
  list <- getListboxEntries listbox (LIndex 0) LIndexEnd
  let breaked = break (>= elem) list
      pos = LIndex (length (fst breaked))
      tail = snd breaked
  case tail of
    (elemx:_) | elemx == elem -> proc $ chdMessage (show tail)
    _ -> insertListbox listbox pos [elem]
  return pos
  

removeListboxElement :: Listbox -> String -> GUI ()
removeListboxElement listbox elem = do
  list <- getListboxEntries listbox (LIndex 0) LIndexEnd
  listboxLength <- getListboxSize listbox
  let pos = length (fst (break (== elem) list))
  let lbPos = LIndex pos
  if (pos < listboxLength)
    then deleteListbox listbox lbPos lbPos
    else return ()

