-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   GuiMain Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

module CHD.GuiMain
  where


-------------------------------------------------------------------------------
-- IMPORTS
------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import Data.FiniteMap
import Tcl

import CHD.GuiMsgChan
import CHD.GuiState
import CHD.GuiWindow
import CHD.GuiDoubleListbox
import CHD.GuiSelectDisplayWindow
import CHD.GuiGraphicWindow
import CHD.GuiGraphicWindowState
import CHD.DebugMsgChan
import CHD.BaseTypes
import CHD.BaseFunctions
import CHD.Environment
import CHD.PriorDoubleChannel


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

-- what should be done with which message

getMsg :: GuiState -> (Either GuiInteract GuiMsg) -> GUI ()
getMsg guiState (Left msg) = do
  case msg of
    InitGuiWin guiWin message -> do
      execGuiWindow guiState guiWin message
      guiReceive guiState
    OpenGuiWin win operate -> do
      guiReceive $ stateAddGuiWindow guiState win operate
    CloseGuiWin win -> do
      guiReceive $ stateRemGuiWindow guiState win
    ContinueAll -> do
      mapM_ 
	(\object -> case object of
	  ThreadObj tNo -> proc $ interactCHD (Continue tNo)
	  _ -> return ())
	(keysFM (objectDB guiState))
      guiReceive guiState
    ContinueSuspended object -> do
      let continueMsg = (\obj -> case obj of
	    ThreadObj tNo -> proc $ interactCHD (Continue tNo)
	    _ -> return () )
      mapM_ (\(ArrowObj (obj1,obj2)) -> do
	      continueMsg obj1
	      continueMsg obj2 )
	    (arrows (lookupObjectState guiState object))
      guiReceive guiState
    PopupMenu object list position -> do
      chdPopupMenu guiState object list position
      guiReceive guiState
    MoveUp object1 -> do 
      let objectState1 = lookupObjectState guiState object1
      newState <- maybe (return guiState)
	    (\object2 -> do
	      let objectState2 = lookupObjectState guiState object2
		  state = linkBefore guiState object1 (prevCO objectState2)
	      guiMoveObject state object1 (coordinate objectState2) )
	    (prevCO objectState1)
      guiReceive newState 

    MoveDown object1 -> do 
      let objectState1 = lookupObjectState guiState object1
      newState <- maybe (return guiState)
	    (\object2 -> do
	      let state = linkBefore guiState object2 (prevCO objectState1)
	      guiMoveObject state object2 (coordinate objectState1) )
	    (nextCO objectState1)
      guiReceive newState 
    GuiQuit -> do
      proc $ C.yield
      destroy (windowMain guiState)
      proc $ chdMessage "GUI: exit"
    GuiRemove object identifier -> do
      state <- removeCanvasObject guiState object identifier
      guiReceive state
    _ -> guiReceive guiState
getMsg guiState (Right msg) = do
  case msg of
    GuiWinMessage guiWin message -> do 
      execGuiWindow guiState guiWin message
      guiReceive guiState
    GuiTitle name -> do
      if (name == "")
	then title (windowMain guiState) "Concurrent Haskell Debugger"
	else title (windowMain guiState) ("CHD - " ++ name)
      guiReceive guiState
    GuiAdd object params -> do
      state <- stateAddObject guiState object
      getMsg state (Right (GuiPar object params))
    GuiPar object params -> do
      state <- foldM
	         (\state param -> stateParameterObject state object param)
		 guiState
		 params
      guiReceive state
    GuiDel object -> do
      state <- stateRemoveObject guiState object
      guiReceive state

-- checking if visualized Thread can be autocontinued....
    ThreadReady tNo -> do
      proc $ interactCHD (Continue tNo)
      guiReceive guiState
    _ -> guiReceive guiState



chdPopupMenu :: GuiState -> Object -> [String] -> Coord -> GUI ()
chdPopupMenu state object list position = do
  popupMenu <- menu (windowMain state) [ tearoff False ] 

  mbutton popupMenu
	  [ wgt_label (show object),
	    foreground $ colorFct object,
	    active_state Disabled ]
  
  mapM (\string -> mbutton popupMenu 
	  [ wgt_label string, 
	    foreground $ colorFct object, 
	    active_state Disabled ]) list
     
  separator popupMenu

  mbutton popupMenu $
    case object of
      ThreadObj tNo -> 
	[ wgt_label "Continue", 
	  command $ proc $ interactCHD $ Continue tNo ]
      _ -> 
        [ wgt_label "Continue Suspended",
	  command $ interactGui $ ContinueSuspended object ]

  mbutton popupMenu 
	  [ wgt_label "Hide",
	    command (do
		      interactGui $ InitGuiWin VisualizationConfigWindow 
				    $ RemoveListbox EnableListbox [show object]
		      proc $ interactCHD $ Visualize object False) ]

  mbutton popupMenu 
	  [ wgt_label "Move Up",
	    command (do interactGui $ MoveUp object) ]
  mbutton popupMenu 
	  [ wgt_label "Move Down",
	    command (do interactGui $ MoveDown object) ]

  separator popupMenu
  
  popup popupMenu position


chdCanvasMain :: GuiState -> GUI GuiState
chdCanvasMain guiState = do
  let mywindow = windowMain guiState
  myframe <- frame mywindow []
  mycanvas <- canvas mywindow [ background "white", takefocus True,
				scrollregion ((0,0),(0,0)) ]
  myhscroll <- hscroll mycanvas [takefocus False]
  myvscroll <- vscroll mycanvas [takefocus False]
  packAdd mycanvas [packH, inFrame myframe, expand True, fillXY]
  packAdd myvscroll [packH, inFrame myframe, fillY]
  packAdd myframe [packV, expand True, fillXY]
  packAdd myhscroll [packV, fillX]
  bind mycanvas "<Key-Up>" (yScroll mycanvas (ScrollUnits (-1)))
  bind mycanvas "<Key-Down>" (yScroll mycanvas (ScrollUnits 1))
  bind mycanvas "<Key-Left>" (xScroll mycanvas (ScrollUnits (-1)))
  bind mycanvas "<Key-Right>" (xScroll mycanvas (ScrollUnits 1))
  bind mycanvas "<Key-Home>" (do
    xMoveTo mycanvas 0
    yMoveTo mycanvas 0)
  bind mycanvas "<Key-Next>" (yScroll mycanvas (ScrollPages 1))
  bind mycanvas "<Key-Prior>" (yScroll mycanvas (ScrollPages (-1)))
  scanMark mycanvas 0 0
--  focus mycanvas
  return (guiState { canvasMain = mycanvas })
	 

quitGui :: GUI ()
quitGui = do
  proc $ interactCHD CHDQuit
  interactGui GuiQuit


-------------------------------------------------------------------------------
-- Window Menu
-------------------------------------------------------------------------------

chdMenuMain :: GuiState -> GUI GuiState
chdMenuMain guiState = do
  frame1 <- frame (windowMain guiState) []
  frame2 <- frame (windowMain guiState) []

-----------------
-- Menu Help
  item9 <- menubutton (windowMain guiState) [text "Help", underline 0]
  menu9 <- menu item9 [tearoff False]
  cset item9 [use_menu menu9]
-- Button Legend
  mbutton menu9 [wgt_label "Legend"]
-- Button Messages
  mbutton menu9 [wgt_label "Messages"]
-- Button CHD-Functions
  mbutton menu9 [wgt_label "CHD-Functions"]
  separator menu9
-- Button About
  mbutton menu9 [wgt_label "About ..."]


-----------------
-- Menu View
  item3 <- menubutton (windowMain guiState) [text "View", underline 0]
  menu3 <- menu item3 [tearoff False]
  cset item3 [use_menu menu3] 
-- Button Source Code
  butSouCod <- mbutton menu3 [ wgt_label "Source Code" ]
  cset butSouCod
       [ command (do
	   cset butSouCod [ active_state Disabled ]
	   initSelectDisplayWindow SourceCodeViewWindow
	     (cset butSouCod [ active_state Active ])) ]
-- Button Progress
--  butProg <- mbutton menu3 [ wgt_label "Progress" ]
--  cset butProg
--      [ command (do
--	   cset butProg [ active_state Disabled ]
--	   initGraphicWindow ProgressViewWindow
--	     (cset butProg [ active_state Active ]))]
-- Button Show Root Window
  mbutton menu3 [ wgt_label "Show Root Window", 
 	          command (do
		    rootwin <- rootWin
		    showWindow rootwin) ]

-----------------
-- Menu Config
  item2 <- menubutton (windowMain guiState) [text "Config", underline 0]
  menu2 <- menu item2 [tearoff False]
  cset item2 [use_menu menu2]
-- Button Visualize Elements
  butVisEle <- mbutton menu2 [ wgt_label "Visualize Elements" ]
  cset butVisEle 
       [ command (do
	   cset butVisEle [ active_state Disabled ]
	   initDoubleListbox VisualizationConfigWindow 
	     (cset butVisEle [ active_state Active ])) ]
--  separator menu2
-- Button Break Action
  butBreAct <- mbutton menu2 [ wgt_label "Break Action" ]
  cset butBreAct
       [ command (do
	   cset butBreAct [ active_state Disabled ]
	   initDoubleListbox ActionBreakConfigWindow
	     (cset butBreAct [ active_state Active ])) ]


-----------------
-- Menu Debugger
  item1 <- menubutton (windowMain guiState) [text "Debugger", underline 0]
  menu1 <- menu item1 [tearoff False]
  cset item1 [use_menu menu1]
  mbutton menu1 [wgt_label "Continue All", command $ interactGui ContinueAll]
  separator menu1
-- Button Quit
  mbutton menu1 [wgt_label "Quit", command quitGui] --butCHDQui
  

-----------------
-- Button Continue All
  button1 <- button (windowMain guiState) 
    [ text "Continue",
      command $ interactGui ContinueAll ]

-----------------
-- placing in panels
  packAdd item1 [packH, inFrame frame1]
  packAdd item2 [packH, inFrame frame1]
  packAdd item3 [packH, inFrame frame1]
  packAdd item9 [packAnchor E, inFrame frame1]

  packAdd button1 [packH, inFrame frame2]

  packAdd frame1 [packV, fillX, packAnchor W]
  packAdd frame2 [packV, fillX, packAnchor W]
  return guiState


-------------------------------------------------------------------------------
-- Main Window
-------------------------------------------------------------------------------

chdWinMain :: GuiState -> (GuiState -> GUI ()) -> GUI ()
chdWinMain guiState mainGuiFunction = do
  rootwin <- rootWin
  addFinaliserW rootwin quitGui
  hideWindow rootwin

  window <- window []
  title window "Concurrent Haskell Debugger"
  geometry window (WinSz (500,600))
  bind window "<Destroy>" quitGui	     

  newState <- chdMenuMain (guiState {windowMain = window})
{-  bind window "<Key-p>" (do
    protocol <- getMCheck (checkbuttonProtocolWindow newState)
    setMCheck (checkbuttonProtocolWindow newState) (not protocol) 
--    interactGui ProtocolView
    ) -}

  newState2 <- chdCanvasMain newState
  
  focus window
  mainGuiFunction newState2


guiReceive :: GuiState -> GUI ()
guiReceive guiState = do
  empty <- proc $ isEmptyDblChan guiMsgChan
  if empty
    then do
      proc $ C.threadDelay 100000           -- this is strongly recommended!
      after 1 $ guiReceive guiState
      return ()
    else do
      message <- proc $ readDblChan guiMsgChan
      getMsg guiState message


startGui :: (GuiState -> GUI ()) -> IO () 
startGui mainGuiFunction = do
  start (chdWinMain initGuiState mainGuiFunction)


