-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   GuiMsgChan Module
--     by Thomas Boettcher <thomas.boettcher@gmx.de>
-------------------------------------------------------------------------------

-- implements the msg-Chan for the GUI

module CHD.GuiMsgChan 
  (
  GuiInteract(..),
  GuiMsg(..),
  Parameter(..),
  parseGuiMsg,
  interactGui,
  guiMsgChan,
  priorGui
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import qualified Control.Concurrent as C
import System.IO.Unsafe
import Tcl

import CHD.GuiWindow
import CHD.BaseTypes
import CHD.BaseFunctions
import CHD.PriorDoubleChannel
import CHD.Environment


-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------

type GuiMsgChan = DblChan GuiInteract GuiMsg


data GuiInteract = 
    GuiRemove Object Int	-- Remove Object from Display
  | GuiQuit
  | InitGuiWin GuiWindow GuiWinMsg
  | OpenGuiWin GuiWindow OperateGuiWin
  | CloseGuiWin GuiWindow
  | ContinueAll
  | ContinueSuspended Object
  | PopupMenu Object [String] Coord
  | MoveUp Object
  | MoveDown Object
  deriving Show  
 

data GuiMsg = 
    GuiAdd Object [Parameter] -- Add Object if not added and set para
  | GuiPar Object [Parameter] -- just Parameter an exisiting Obj
  | GuiDel Object	      -- Delete Object in Time from Display
  | GuiTitle String	      -- Set Title of Window
  | ThreadReady ThreadNo      -- Return ready Message to CHD
  | GuiWinMessage GuiWindow GuiWinMsg
  deriving (Show,Eq)


-- Parameter for GuiMsg. presenting Display-Options
data Parameter = Name String 
	       | FillColor String
	       | OutlineColor String
	       | Parent ThreadNo
	       | Elements [String]
	       | Thickness Int
	       | Action String
  deriving (Show,Eq)


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

parseParameter :: [String] -> [Parameter]
parseParameter [] = [] 
parseParameter (parameter:arg:params) = 
  let pm = case parameter of
		"Name"	       -> Name arg 
		"FillColor"    -> FillColor arg
		"OutlineColor" -> OutlineColor arg
		"Parent"       -> Parent (ThreadNo (read arg))
		"Elements"     -> Elements (parseElements arg)
		"Thickness"    -> Thickness (read arg)
		"Action"       -> Action arg in
  pm:(parseParameter params)


parseElements :: String -> [String]
parseElements [] = []
parseElements list = 
  let (first,tail) = breakCut (==',') list in
  (first:(parseElements tail))


parseGuiMsg :: [String] -> GuiMsg
parseGuiMsg (comand:obj:params) = 
  case comand of
       "Add" -> GuiAdd (parseObject obj) 
		       (parseParameter params)  
       "Par" -> GuiPar (parseObject obj)
		       (parseParameter params)
       "Del" -> GuiDel (parseObject obj)
       "GuiWinMessage" -> GuiWinMessage (parseGuiWindow obj) 
					(parseGuiWinMsg params)
       "Title" -> GuiTitle obj
       "ThreadReady" -> ThreadReady (ThreadNo (read obj))


-- message-sender for prio-channel

priorGui :: GuiInteract -> IO ()
priorGui message = 
  writeDblChanPrio guiMsgChan message


interactGui :: GuiInteract -> GUI ()
interactGui message = 
  proc $ priorGui message


guiMsgChan :: GuiMsgChan
guiMsgChan = unsafePerformIO ( do
  guiChan <- newDblChan
  return guiChan
  )



