-------------------------------------------------------------------------------
-- Concurrent Haskell Debugger 
--   GuiSelectDisplayWindow Module
--
-------------------------------------------------------------------------------

-- this implements the double-listbox-window

-- needed: an init-function
--         an operate-function

-- programming TclHaskell:
--   init composes all elements together
--   operate reads and writes messages to the CHD

module CHD.GuiSelectDisplayWindow
  (
    initSelectDisplayWindow,
    operateSelectDisplayWindow
  )
  where


-------------------------------------------------------------------------------
-- IMPORTS
-------------------------------------------------------------------------------

import CHD.GuiMsgChan
import CHD.DebugMsgChan
import CHD.Environment
import CHD.GuiWindow
import CHD.BaseTypes
import CHD.BaseFunctions

import Tcl

import Debug.Trace

-------------------------------------------------------------------------------
-- TYPES
-------------------------------------------------------------------------------


-------------------------------------------------------------------------------
-- FUNCTIONS
-------------------------------------------------------------------------------

initSelectDisplayWindow :: GuiWindow -> GUI () -> GUI ()
initSelectDisplayWindow viewWin windowFinalizer = do
-- new Window
  myWindow <- window []
  title myWindow ("CHD - " ++ (show viewWin))
  bind myWindow "<Destroy>" (do interactGui (CloseGuiWin viewWin)
				windowFinalizer)

-- new Grid
  windowGrid <- frame myWindow []

-- create Elements
  selectLabel <- case viewWin of
    SourceCodeViewWindow -> label myWindow [text "SourceCodeFiles:"]
    _ -> label myWindow [text "Selection:"]
  displayLabel <- case viewWin of
    SourceCodeViewWindow -> label myWindow [text "SourceCode:"]
    _ -> label myWindow [text "Display:"]
  selectLB <- listbox myWindow [takefocus True, 
				     background "LightGoldenRod"]
  displayEdit <- edit myWindow [active_state Disabled,
    font "-adobe-helvetica-medium-r-normal--10-100-75-75-p-56-iso8859-1",
    background "white"]
  selectScroll <- vscroll selectLB []
  displayScroll <- vscroll displayEdit []
  let widgets = (selectLB, displayEdit)
  displayButton <- button myWindow [text "->",command (do display widgets)]

-- pack elements in grid
  gridAdd selectLabel (0,0) [widthX 2, gfillX, ginFrame windowGrid]
  gridAdd selectLB (0,1) [heightY 5, gfillXY, ginFrame windowGrid]
  gridAdd selectScroll (1,1) [heightY 5, gfillY, ginFrame windowGrid]
  gridAdd displayButton (2,3) [ginFrame windowGrid]
  gridAdd displayLabel (4,0) [widthX 2, gfillX, ginFrame windowGrid]
  gridAdd displayEdit (4,1) [heightY 5, gfillXY, ginFrame windowGrid]
  gridAdd displayScroll (5,1) [heightY 5, gfillY, ginFrame windowGrid]

-- pack grid in window
  packAdd windowGrid [packV, fillXY]

-- get defaultColor
  defaultColor <- cget displayEdit background

-- focus on enableListbox
  focus selectLB

-- key-bindings: arrows left/right moves element
  bind selectLB "<Double-1>" (display widgets)
  bind myWindow "<Key-Right>" (display widgets)
  
-- telling Debugger & Gui
  interactGui (OpenGuiWin viewWin (operateSelectDisplayWindow widgets))
  proc $ interactCHD (GetInfoGuiWin viewWin)

  return ()


display :: (Listbox, Edit) -> GUI ()
display (selectLB,edit) = do
  items <- getListboxSelection selectLB
  list <- foldM
    (\result num -> do
      let pos = LIndex num
      elems <- getListboxEntries selectLB pos pos
      return (elems ++ result)) 
    [] 
    items
  if list==[]
     then return ()
     else do clearListboxSelection selectLB (LIndex 0) LIndexEnd
	     proc $ interactCHD $ DisplayPosition (last list)


markLine :: Edit -> Int -> String -> String -> GUI ()
markLine edit line bgcolor fgcolor = do
  tag edit [TIndex line 0,TIndex (line+1) 0] 
	   [background bgcolor,foreground fgcolor]
  return ()


unMarkLine :: Edit -> Int -> GUI ()
unMarkLine edit line = do
  --oldTags <- getAllTags edit
  --map (tagId -> TIndexTagFirst tagId) oldTags
  return ()


operateSelectDisplayWindow :: (Listbox, Edit) -> GuiWinMsg -> GUI ()
operateSelectDisplayWindow (selectLB,edit) msg = 
  case msg of
    InsertListbox SelectListbox list ->
            mapM_ (insertListboxElement (selectLB,edit)) list
    RemoveListbox SelectListbox list ->
            mapM_ (removeListboxElement (selectLB,edit)) list	    
    Select SelectListbox filePath threaName ->
	    select (selectLB,edit) filePath threaName  
    MarkEdit DisplayEdit line bgcolor fgcolor -> 
	    markLine edit line bgcolor fgcolor
    UnmarkEdit DisplayEdit line -> 
	    markLine edit line "white" "black"
    _ -> return ()


removeListboxElement :: (Listbox, Edit) -> String -> GUI ()
removeListboxElement (selectLB,edit) threadName = do
  list <- getListboxEntries selectLB (LIndex 0) LIndexEnd
  let breaked = break (>= threadName) list
  if (head (snd breaked)) == threadName
     then let start = LIndex (length (fst breaked))
	      end = LIndex ((length (fst breaked))+1) in
	  deleteListbox selectLB start start
     else return ()


insertListboxElement :: (Listbox, Edit) -> String -> GUI ()
insertListboxElement (selectLB,edit) threadName = do
  list <- getListboxEntries selectLB (LIndex 0) LIndexEnd
  let breaked = break (>= threadName) list
      pos = LIndex (length (fst breaked))
  insertListbox selectLB pos [threadName]


select :: (Listbox, Edit) -> String -> String -> GUI ()
select (selectLB,edit) filePath threadName = do
  list <- getListboxEntries selectLB (LIndex 0) LIndexEnd
  selected <- getListboxSelection selectLB
  let breaked = break (>= threadName) list
      first = fst breaked
      second = snd breaked
      pos = LIndex (length first)
  if (second /= []) && ((head second) == threadName)
     then if (selected /= []) &&  ((head selected) == (length first))
	     then return ()
	     else loadEdit edit filePath
     else error $ threadName ++  " not in Listbox"
  clearListboxSelection selectLB (LIndex 0) LIndexEnd
  addListboxSelection selectLB pos pos