#!/usr/bin/env python
# coding: utf-8
# Basismodul für MultiTest 0.3
# Ruthard Baudach
# 13.05.2007

"""
Base classes and definitions/configuration.

_base_def is a dictionary containing the configuration.
The classes cRadioBox, cTextInput and cTextDisplay provide controls for the testdisplay
cTest is a notebook with two pages showing the actual test and the history of former tests.
cTestDisplay is the actual test, cTestHistory the history

Concerning test definitions see module mtSample.

If the tests to be defined are none-standard with multiple subtests, multiple scores, none-standard evaluation or whatever the affected classes and attributes have to be overridden in the test module

"""

import wx
import csv
import wx.grid

_base_def = {
	# the modules
	'index_modules': [
		'mtProcam',
		'mtDSFB'
		],
	# here should be the real path and file with patient import data. TODO: integrate os.path
	'importfile': 'import.dat',
	# index_import defines the field order in the import csv file
	'index_import': ['Nr','NN','VN','NZ','Geb', 'LDL', 'HDL', 'TG', 'BG', 'BP', 'AGE', 'Sex', 'DM', 'Smoke', 'FH', 'Medis'  ],
	'database': 'multitest.csv',	# here should be the real path and file of the csv-file used as database
	'exportfile': 'export.dat'	# exportfile back to emr
	}

# current Patient - dictionary with imported patient data including personal and medical data
currentPatient = {}
# ---------------------------------------------------------------------------------------------
# ---------------------------------------------------------------------------------------------


class cRadioBox(wx.RadioBox):
	""" definiert eine Radiobox """

	def __init__(self,parent,test,item):
		wx.RadioBox.__init__(
		self,
		parent=parent,
		id=-1,
		label=parent.template[test][item]["prompt"],
		pos=wx.DefaultPosition,
		size=wx.DefaultSize,
		choices=parent.template[test][item]["choices"],
		majorDimension=parent.template[test][item]["col"],
		style=parent.template[test][item]["style"]
		)
		self.SetSelection(parent.template[test][item]['default'])

	def GetValue(self):
		return self.GetSelection()

	def SetValue(self,val):
		self.SetSelection(int(val))


class cTextInput(wx.BoxSizer):
	""" definiert ein Textfeld mit Beschriftung """
	def __init__(self,parent,test,item):
		wx.BoxSizer.__init__(self,wx.HORIZONTAL)
		sb = wx.StaticBox(parent,-1,parent.template[test][item]['prompt'])
		sbs = wx.StaticBoxSizer(sb,wx.VERTICAL)
		self.tc = wx.TextCtrl(parent,-1,'',size=parent.template[test][item]['size'],style=wx.TE_PROCESS_ENTER)

		sbs.Add(self.tc,0,wx.ALL,5)
		self.Add(sbs,0,wx.ALL,5)

	def GetId(self):
		return self.tc.GetId()
		
	def SetValue(self,val):
		self.tc.SetValue(str(val))

	def GetValue(self):
		return self.tc.GetValue()
	

class cTextDisplay(wx.StaticText):
	"""definiert ein Textdisplay, z.B. zur Beschriftung"""
	def __init__(self,parent,test,item):
		wx.StaticText.__init__(self,parent,-1,parent.template[test][item]['text'],size=parent.template[test][item]['size'])

	def SetValue(self,val):
		self.SetLabel(val)


# -----------------------------------------------------------------------------------------
# Haupklassen
# -----------------------------------------------------------------------------------------

class cTest(wx.Notebook):
	"""
	A Notebook showing the test and the history of tests.

	template is the configuration of the test as layed out in the testmodule._test_def
	"""
	def __init__(self,parent,template,test):
		"""
		The following lines have to be added in derived classes in the test modules in order to set the derived test pages and not the basic ones:
		If there are multiple subtests, all subtests have to be added, of course.)
		self.testdisplay = cTestDisplay(self,self.template,test)
		self.testhistory = cTestHistory(self,self.template,test)
		self.AddPage(self.testdisplay,self.test)
		self.AddPage(self.testhistory,'Verlauf')
		"""
		wx.Notebook.__init__(self,parent,-1,style=wx.NB_BOTTOM)
		self.template = template
		self.test = test

		self.Bind(wx.EVT_NOTEBOOK_PAGE_CHANGED,self._on_notebook_page_changed,self)

	def _on_notebook_page_changed(self,evt):
		"""
		Refresh the current page in order to show the current values.
		"""
		currentPage = self.GetCurrentPage()
		currentPage._refresh_score(evt)

	
	def _store_data(self):
		"""
		Store the data of the current test.
		"""
		row = [str(currentPatient['Nr']),self.test,str(wx.DateTime.Today().FormatDate())]
		for item in self.template[self.test]['index_score'] + self.template[self.test]['index_result']:
			row = row + [str(self.testdisplay.controls[item].GetValue())]
		file = open(_base_def['database'],'a')
		writer = csv.writer(file)
		writer.writerow(row)
		file.close()

	def _export(self):
		"""
		Export the result of the current test to exportfile.

		Can be used to be imported in EMR
		"""
		for notice,score in self.template[self.test]['index_back']:
			backnotice =  self.template[self.test]['backnotice'] + str(self.testdisplay.controls[score].GetValue())
		file = open(_base_def['exportfile'],'w')
		file.write(backnotice)
		file.close()

	def _on_new_patient(self):
		self.testdisplay._set_imported_values()
		self.testdisplay._refresh_score(self)
		self.testhistory._do_layout()
		self.testhistory._refresh_score(self)

		
# -----------------------------------------------------------------------------------------

class cTestDisplay(wx.Panel):
	
	"""
	definiertaktueller Test ein Panel mit Darstellung aller benötigte Werte, des Ergebnisses
	und automatischer Aktualisierung des Ergebnisses bei Änderung der Werte.
	Standardauswertung ist ein einfacher Score mit Addition der sich aus den
	einzelnen Testwerten ergebenden Punktzahlen entsprechend der Testdefinition in _test_def.
	Bei anderen Auswertungen muss eine eigene Klasse für diesen Test abgeleitet werden,
	und die Funktionen dementsprechend neu definiert werden.
	"""
	
	def __init__(self,parent,template,test):
		"""
		template = _test_def of current test module
		test = current test
		"""
		wx.Panel.__init__(self,parent,-1)
		self.template = template
		self.test = test
		#self.currentPatient = parent.currentPatient
		self.controls = {}  #dictionary with controls

		self._make_questionaire()
		self._make_result_display()
		self._do_layout()
		self._set_imported_values()
		self._refresh_score(self)

	# -----------------------------------------------------------------------------
	def _make_questionaire(self):
		"""
		Makes questionaire for the test.

		Has to be overridden if it does not suit the test..
		"""
		# GridSizer nimmt einzelne Fragen auf
		row, col = self.template[self.test]['size_questionaire']
		self.gs = wx.FlexGridSizer(row,col,5,5)
		# Wörterbuch mit Fragen erstellen und in Gridsizer aufnehmen
		

		for item in self.template[self.test]['index_layout']:

			try:
				width,hight = self.template[self.test][item]['size']
			except KeyError:
				pass

			controltype = self.template[self.test][item]['type']

			if controltype  == 'RadioBox':
				self.controls[item] = cRadioBox(self,self.test,item)
				self.Bind(wx.EVT_RADIOBOX,self._refresh_score,self.controls[item])
			elif controltype == 'TextInput':
				self.controls[item] = cTextInput(self,self.test,item,)
				self.Bind(wx.EVT_TEXT_ENTER,self._refresh_score,self.controls[item])
			elif controltype == 'Space':
				self.controls[item] = wx.Size(width,hight)
			elif controltype == 'Line':
				self.controls[item] = wx.StaticLine(self,-1,size=self.template[self.test][item]['size'])
			elif controltype == 'TextDisplay':
				self.controls[item] = cTextDisplay(self,self.test,item)
			else:
				wx.MessageBox('Beim Layout des Fragebogens trat ein Fehler auf:\n\nAngefordertes Element nicht definiert','Fehler')
			self.gs.Add(self.controls[item],0,wx.ALIGN_CENTER,5)
		
	# -----------------------------------------------------------

	def _make_result_display(self):
		
		"""
		Returns sizer with result and comments.

		Has to be overridden if it does not suit the test.
		"""
		row, col = self.template[self.test]['size_resultdisplay']
		self.result = wx.FlexGridSizer(row,col,5,5)
		for item in self.template[self.test]['index_result_layout']:
			
			try:
				width,hight = self.template[self.test][item]['size']
			except KeyError:
				pass

			controltype = self.template[self.test][item]['type']

			if controltype == 'TextInput':
				self.controls[item] = cTextInput(self,self.test,item)
			elif controltype == 'Space':
				self.controls[item] = wx.Size(width,hight)
			elif controltype == 'Line':
				self.controls[item] = wx.StaticLine(self,-1,size=self.template[self.test][item]['size'])
			elif controltype == 'TextDisplay':
				self.controls[item] = cTextDisplay(self,self.test,item)
			else:
				wx.MessageBox('Beim Layout der Resultatanzeige trat ein Fehler auf:\n\nAngefordertes Element nicht definiert','Fehler')
			self.result.Add(self.controls[item],0,wx.ALIGN_CENTRE,5)

	# ------------------------------------------------------------------------------
	def _do_layout(self):
		"""
		Layout.

		Has to be overridden, if standard layout does not suit the test.
		"""
		caption = wx.StaticText(self,-1,self.template[self.test]['caption'])
		line1 = wx.StaticLine(self,-1,size=(500,0))
		line2 = wx.StaticLine(self,-1,size=(500,0))
		
		border = wx.BoxSizer(wx.VERTICAL)
		border.Add(caption,0,wx.ALL|wx.CENTRE,5)
		border.Add(line1,0,wx.ALL|wx.CENTRE,5)
		border.Add(self.gs,0,wx.ALL|wx.CENTRE,5)
		border.Add(line2,0,wx.ALL|wx.CENTRE,5)
		border.Add(self.result,0,wx.ALL|wx.CENTRE,5)
		
		self.SetSizerAndFit(border)
		
	# ------------------------------------------------------------------------

	def _set_imported_values(self):
		"""
		Try to set imported value, if any.
		"""
		for item in self.template[self.test]['index_layout']:
			try:
				self.controls[item].SetValue(currentPatient[item])
			except (KeyError, AttributeError, ValueError):
				pass
	# ------------------------------------------------------------------------
	
	def _refresh_score(self,evt):
		"""
		Get score and display it.

		Has to be overridden if it does not suit the test.
		"""
		score = self._get_score(self.template[self.test]['index_score'])
		self.controls['score'].SetValue(score)

	# -------------------------------------------------------------------

	def _get_score(self,index):
		"""
		Standard evaluation.
		"""
		score = 0
		for item in index:
			# sort keys numerically (not according to alphabeth!)
			tablekeys = []
			for i in self.template[self.test][item]['table'].keys():
				tablekeys.append(int(i))
				tablekeys.sort()
			#look up score in table
			k = self.template[self.test][item]['table'][tablekeys[0]] # just in case the value is lower than the first entry in the table
			for i in tablekeys:
				try:
					if float(self.controls[item].GetValue()) >= float(i):
						k = self.template[self.test][item]['table'][i]
						continue
				except ValueError:
					score = 'nicht berechenbar'
					return score
			score = score + k
		return score


# ---------------------------------------------------------------------
# 
#----------------------------------------------------------------------

class cTestHistory(wx.grid.Grid):
	"""
	Displays all stored tests chronologically in a table.
	"""
	def __init__(self,parent,template,test):
		wx.grid.Grid.__init__(self,parent,-1)
		self.template = template
		self.test = test
		self.parent = parent
		self._do_layout()

	def _do_layout(self):
		"""
		Layout and fill the table with the result of former, saved tests.
		"""
		# rowindex = index of the stored data
		self.rowindex = self.template[self.test]['index_score'] + self.template[self.test]['index_result']
		
		self.CreateGrid(len(self.rowindex),1)
		# label today's column
		self.SetColLabelValue(0,wx.DateTime.Today().FormatDate())
		# label rows
		for i in range(len(self.rowindex)):
			self.SetRowLabelValue(i,self.rowindex[i])
			if self.rowindex[i] == 'score':
				hlt_score = wx.grid.GridCellAttr()
				hlt_score.SetBackgroundColour(wx.LIGHT_GREY)
				
				self.SetRowAttr(i,hlt_score)
		# read data
		data = self._get_stored_data()
		# reverse data to get the newest to the left
		data.reverse()
		# find data of this test
		col = 1
		for record in data:
			if record['test'] == self.test:
				self.AppendCols(1)
				self.SetColLabelValue(col,record['date'])
				i = 0
				for i in range(len(self.rowindex)):
					self.SetCellValue(i,col,str(record[self.rowindex[i]]))
				col = col + 1
					

	def _get_stored_data(self):
		"""
		Reads the stored data for this test.
		 
		A list of dictionaries, each containing one stored test, is returned.
		"""
		valuekeys = ['Nr','test','date']
		for i in range(len(self.rowindex)):
			valuekeys.append(self.rowindex[i])
		stored_data = []
		file = open(_base_def['database'],'r')
		reader = csv.DictReader(file,fieldnames=valuekeys)
		for row in reader:
			if row['Nr'] == currentPatient['Nr']:
				stored_data.append(row)
		file.close()
		return stored_data

	def _refresh_score(self,evt):
		"""
		Transfers data from testdisplay in today's column.
		"""
		for i in range(len(self.rowindex)):
			self.SetCellValue(i,0,str(self.parent.testdisplay.controls[self.rowindex[i]].GetValue()))

