/* NSSound+AU.m

   Methods for reading AU/SND files

   Copyright (C) 2009 Free Software Foundation, Inc.
   
   Written by:  Stefan Bidigaray <stefanbidi@gmail.com>
   Date: May 2009
   
   This file is part of the GNUstep GUI Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.
   If not, see <http://www.gnu.org/licenses/> or write to the 
   Free Software Foundation, 51 Franklin Street, Fifth Floor, 
   Boston, MA 02110-1301, USA.
*/

#include <Foundation/Foundation.h>
#include "AppKit/NSSound.h"

struct au_file_header
{
  unsigned int magicNumber;
  unsigned int dataOffset;
  unsigned int dataByteCount;
  unsigned int encoding;
  unsigned int sampleRate;
  unsigned int channelCount;
};

@implementation NSSound (AU)

+ (BOOL) _soundIsAU: (NSData *) soundData
{
  void *bytes;
  int  magicNumber;
  
  bytes = [soundData bytes];
  magicNumber = ((int *)bytes)[0];
  if (NSSwapBigIntToHost(magicNumber) == 0x2e736e64)
    {
      return YES;
    }
  return NO;
}

- (id) _initSoundWithAU: (NSData *) soundData
{
  struct au_file_header fileHeader;
  int format;
  NSRange dataRange;
  
  if (!(self = [super init]))
    {
      return nil;
    }
  
  [soundData getBytes: &fileHeader length: sizeof(struct au_file_header)];
  _sampleRate = NSSwapBigIntToHost (fileHeader.sampleRate);
  _channelCount = NSSwapBigIntToHost (fileHeader.channelCount);
  
  /* I only understand PCM, uLaw and aLaw data */
  switch (NSSwapBigIntToHost (fileHeader.encoding))
    {
      case 1: format = GSSoundFormatULaw;
              break;
      case 2: format = GSSoundFormatPCM8;
              break;
      case 3: format = GSSoundFormatPCM16;
              break;
      case 5: format = GSSoundFormatPCM32;
              break;
      case 27: format = GSSoundFormatALaw;
               break;
      default: RELEASE(self);
               return nil;
    }
  
  dataRange = NSMakeRange (NSSwapBigIntToHost(fileHeader.dataOffset),
                  NSSwapBigIntToHost(fileHeader.dataSize));
  
  /* AU/SND files are always big endian */
  return [self _initWithData: soundData
                       range: dataRange
                      format: format
                   byteOrder: NS_BigEndian];
}

@end

