/* NSSound+WAV.m

   Methods for reading AU/SND files

   Copyright (C) 2009 Free Software Foundation, Inc.
   
   Written by:  Stefan Bidigaray <stefanbidi@gmail.com>
   Date: May 2009
   
   This file is part of the GNUstep GUI Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.
   If not, see <http://www.gnu.org/licenses/> or write to the 
   Free Software Foundation, 51 Franklin Street, Fifth Floor, 
   Boston, MA 02110-1301, USA.
*/

#include <Foundation/Foundation.h>
#include "AppKit/NSSound.h"

#if !defined(HAVE_LIBSNDFILE)

struct wav_file_header
{
  unsigned int magicNumber;
  unsigned int chunkSize;
  unsigned int format;
};

struct wav_fmt_chunk
{
  unsigned int chunkID;
  unsigned int chunkSize;
  unsigned short audioFormat;
  unsigned short channelCount;
  unsigned int sampleRate;
  unsigned int byteRate;
  unsigned short blockAlign;
  unsigned short bytesPerSample;
};

struct wav_data_chunk
{
  unsigned int chunkID;
  unsigned int dataSize;
};

@implementation NSSound (WAV)

+ (BOOL) _soundIsWAV: (NSData *)soundData
{
  struct wav_file_header fileHeader;
  
  [soundData getBytes: &fileHeader length: sizeof(struct wav_file_header)];
  if (NSSwapBigIntToHost (fileHeader.magicNumber) == 0x52494646 &&
      NSSwapBigIntToHost (fileHeader.format) == 0x57415645)
    {
      return YES;
    }
  return NO;
}

- (id) _initSoundWithWAV: (NSData *)soundData
{
  struct wav_fmt_chunk fmtChunk;
  struct wav_data_chunk dataChunk;
  int format;
  NSRange fmtChunkRange;
  NSRange dataChunkRange;
  NSRange dataRange;
  
  if (!(self = [super init]))
    {
      return nil;
    }
  
  fmtChunkRange = NSMakeRange (sizeof(struct wav_file_header), sizeof(struct wav_fmt_chunk));
  
  [soundData getBytes: &fmtChunk range: fmtChunkRange];
  
  _sampleRate = NSSwapLittleIntToHost (fmtChunk.sampleRate);
  _channelCount = NSSwapLittleIntToHost (fmtChunk.channelCount);
  
  /* I only read PCM, uLaw and aLaw data for WAV files */
  switch (NSSwapLittleIntToHost (fmtChunk.audioFormat))
    {
      case 1: switch (NSSwapLittleIntToHost (fmtChunk.bytesPerSample))
                {
                  case 8: format = GSSoundFormatPCM8;
                          break;
                  case 16: format = GSSoundFormatPCM16;
                           break;
                  case 32: format = GSSoundFormatPCM32;
                           break;
                  default: RELEASE(self);
                           return nil;
                }
              break;
      case 6: if (NSSwapLittleIntToHost (fmtChunk.bytesPerSample) != 8)
                {
                  RELEASE(self);
                  return nil;
                }
              format = GSSoundFormatALaw;
              break;
      case 7: if (NSSwapLittleIntToHost (fmtChunk.bytesPerSample) != 8)
                {
                  RELEASE(self);
                  return nil;
                }
              format = GSSoundFormatULaw;
              break;
      default: RELEASE(self);
               return nil;
    }
  else if (NSSwap
  
  [soundData getBytes: &dataChunk range: dataChunkRange];
  dataChunkRange = NSMakeRange (sizeof(struct wav_file_header) + fmtChunkRange->length,
                     NSSwapLittleIntToHost (dataChunk.chunkSize));
  dataRange = NSMakeRange (sizeof(struct wav_file_header) + fmtChunkRange->length + dataChunkRange->length,
                NSSwapLittleIntToHost(dataChunk.dataSize));
  
  /* WAV files are always in little endian format */
  return [self _initWithData: soundData
                       range: dataRange
                      format: format
                   byteOrder: NS_LittleEndian];
}

@end

#else

@implementation NSSound (WAV)

+ (BOOL) _soundIsWAV: (NSData *)soundData
{
  return NO;
}

- (id) _initSoundWithWAV: (NSData *)data
{
  RELEASE(self);
  return nil;
}

#endif
