/* 
   NSSound.h

   Load, manipulate and play sounds

   Copyright (C) 2002 Free Software Foundation, Inc.

   Written by:  Enrico Sersale <enrico@imago.ro>
   Date: Jul 2002
   
   This file is part of the GNUstep GUI Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.
   If not, see <http://www.gnu.org/licenses/> or write to the 
   Free Software Foundation, 51 Franklin Street, Fifth Floor, 
   Boston, MA 02110-1301, USA.
*/ 

#ifndef _GNUstep_H_NSSound
#define _GNUstep_H_NSSound

#include <Foundation/NSObject.h>
#include <Foundation/NSBundle.h>

@class NSArray;
@class NSData;
@class NSMutableData;
@class NSPasteboard;
@class NSString;
@class NSURL;

#if OS_API_VERSION(GS_API_NONE, GS_API_NONE)
enum
{
  GSSoundFormatUnknown = 0,
  GSSoundFormatPCM8    = 1,
  GSSoundFormatPCM16   = 2,
  GSSoundFormatPCM24   = 3,
  GSSoundFormatPCM32   = 4,
  GSSoundFormatFloat32 = 5,
  GSSoundFormatFloat64 = 6,
  
  GSSoundFormatULaw    = 10,
  GSSoundFormatALaw    = 11
};
#endif

@interface NSSound : NSObject <NSCoding, NSCopying>
{		
  NSString *_name;
  NSString *_playbackDeviceIdentifier;
  NSArray  *_channelMap; /* Currently unused */
  BOOL _onlyReference;
  id _delegate;
  
  NSTimeInterval _duration;
  
  short *_data;
  long  _dataSize;
  long  _frameCount;
  int   _sampleRate;
  int   _channelCount;
  
  /* Used for OpenAL */
  unsigned int _sndSource;
  unsigned int _sndBuffer;
}

//
// Creating an NSSound 
//
/** Initalizes the receiver object with the contents of file located at path.
 *  If byRef is set to YES only the name of the NSSound is encoded with -encodeWithCoder:;
 *  if set to NO the data is encoded.
 *  <p>See Also:</p>
 *  <list>
 *    <item>-initWithContentsOfURL:byReference:</item>
 *    <item>-initWithData:</item>
 *  </list>
 */
- (id)initWithContentsOfFile:(NSString *)path byReference:(BOOL)byRef;
/** Initializes the receiver object with the contents of the data located in url.
 *  If byRef is set to YES only the name of the NSSound is encoded with -encodeWithCoder:;
 *  if set to NO the data is encoded.
 *  <p>See Also:</p>
 *  <list>
 *    <item>-initWithContentsOfFile:byReference:</item>
 *    <item>-initWithData:</item>
 *  </list>
 */
- (id)initWithContentsOfURL:(NSURL *)url byReference:(BOOL)byRef;
/** Initializes the receiver object with the contents of data.
 *  If byRef is set to YES only the name of the NSSound is encoded with -encodeWithCoder:;
 *  if set to NO the data is encoded.
 *  <p>See Also:</p>
 *  <list>
 *    <item>-initWithContentsOfFile:byReference:</item>
 *    <item>-initWithContentsOfURL:byReference:</item>
 *  </list>
 */
- (id)initWithData:(NSData *)data;
- (id)initWithPasteboard:(NSPasteboard *)pasteboard;

//
// Playing
//
/** Pauses audio playback.
 *  Returns YES on success and NO if sound could not be paused.
 */
- (BOOL)pause;
- (BOOL)play; 
- (BOOL)resume;
- (BOOL)stop;
- (BOOL)isPlaying;

//
// Working with pasteboards 
//
+ (BOOL)canInitWithPasteboard:(NSPasteboard *)pasteboard;
/** Provides an array of file types that NSSound can understand.  The returning
 *  array may be directly passed to [NSOpenPanel -runModalForTypes:].
 *  <p>Built with libsndfile:</p>
 *    wav, aiff, aifc, aif, au, snd, mat, mat4, mat5, paf, sf, voc, w64,
 *    xi, caf, sd2, flac, ogg, oga
 *  <p>Built without libsndfile:</p>
 *    wav, aiff, aifc, aif, au, snd
 */
+ (NSArray *)soundUnfilteredPasteboardTypes;
- (void)writeToPasteboard:(NSPasteboard *)pasteboard;

//
// Working with delegates 
//
/** Returns the receiver's delegate
 */
- (id)delegate;
/** Sets the receiver's delegate
 */
- (void)setDelegate:(id)aDelegate;

//
// Sound Information
//
+ (id)soundNamed:(NSString *)name;
+ (NSArray *)soundUnfilteredFileTypes;
- (NSString *)name;
- (BOOL)setName:(NSString *)aName;

#if OS_API_VERSION(MAC_OS_X_VERSION_10_5, GS_API_LATEST)
/** NOT IMPLEMENTED
 */
+ (NSArray *) soundUnfilteredTypes;
/** Returns the length, in seconds, of the receiver.
 */
- (NSTimeInterval) duration;
/** Returns the volume of the receiver.
 *  Volume will be between 0.0 and 1.0.
 */
- (float) volume;
/** Sets the volume of the receiver.
 *  Volume must be between 0.0 and 1.0.
 */
- (void) setVolume: (float) volume;
/** Returns the current position of the audio playback.
 */
- (NSTimeInterval) currentTime;
/** Sets the current time of the audio playback.
 */
- (void) setCurrentTime: (NSTimeInterval) currentTime;
/** Returns the current loop property of the receiver.
 *  YES indicates this NSSound will restart once it reaches the end, otherwise NO.
 */
- (BOOL) loops;
/** Sets the loop property of the receiver.
 *  YES indicates this NSSound will restart once it reaches the end, otherwise NO.
 */
- (void) setLoops: (BOOL) loops;

- (NSString *) playbackDeviceIdentifier;
- (void) setPlaybackDeviceIdentifier: (NSString *) playbackDeviceIdentifier;
- (NSArray *) channelMapping;
- (void) setChannelMapping: (NSArray *) channelMapping;
#endif

@end

#if OS_API_VERSION(GS_API_NONE, GS_API_NONE)
@interface NSSound (GNUstepExtensions)
/** <p>Retrieve an NSData object with the data in the specified format and
 *  fileType headers.  This method is useful to convert between different
 *  sound file types and data formats.</p>
 *  <p>The fileType parameter needs to be one of the NSStrings returned by
 *  -soundUnfilteredPasteboardTypes.</p>
 *  Acceptable formats are:
 *  <list>
 *    <item>GSSoundFormatUnknown</item>
 *    <item>GSSoundFormatPCM8</item>
 *    <item>GSSoundFormatPCM16</item>
 *    <item>GSSoundFormatPCM32</item>
 *    <item>GSSoundFormatULaw</item>
 *    <item>GSSoundFormatALaw</item>
 *  </list>
 */
- (id) dataWithFormat: (int) format fileType: (NSString *) fileType
@end
#endif

//
// Methods Implemented by the Delegate 
//
@interface NSObject (NSSoundDelegate)

/** Method called when sound has finished playing.  Currently this method is not called.
 */
- (void)sound:(NSSound *)sound didFinishPlaying:(BOOL)aBool;

@end


@interface NSBundle (NSSoundAdditions)

- (NSString *)pathForSoundResource:(NSString *)name;

@end

#endif // _GNUstep_H_NSSound

