/** <title>NSSound</title>

   <abstract>Load, manipulate and play sounds</abstract>

   Copyright (C) 2002 Free Software Foundation, Inc.
   
   Author: Enrico Sersale <enrico@imago.ro>
   Date: Jul 2002

   This file is part of the GNUstep GUI Library.
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; see the file COPYING.LIB.
   If not, see <http://www.gnu.org/licenses/> or write to the 
   Free Software Foundation, 51 Franklin Street, Fifth Floor, 
   Boston, MA 02110-1301, USA.
*/ 

#include "config.h"
#include <Foundation/Foundation.h>
#include <GNUstepBase/NSTask+GS.h>
#include "AppKit/NSPasteboard.h"
#include "AppKit/NSSound.h"

#include "NSSound+AU.h"
#include "NSSound+WAV.h"

#if defined(HAVE_SNDFILE_H)
#include <sndfile.h>
#endif

#if defined(HAVE_AL_AL_H)
#include <AL/al.h>
#include <AL/alc.h>

static ALCdevice *device = NULL;
static ALCcontext *context = NULL;
#endif

/* Class variables and functions for class methods */
static NSMutableDictionary *nameDict = nil;
static NSDictionary *nsmapping = nil;

/* Some helper macros to convert to signed short */
#define UCHAR_TO_SHORT(x) ((((signed char)x) - 0x80)<<8)
#define SCHAR_TO_SHORT(x) ((x)<<8)
#define LE_INT_TO_INT(x) (int)NSSwapLittleIntToHost (x)
#define BE_INT_TO_INT(x) (int)NSSwapBigIntToHost (x)
#define LE_INT_TO_SHORT(x) (short)((LE_INT_TO_INT(x))>>16)
#define BE_INT_TO_SHORT(x) (short)((BE_INT_TO_INT(x))>>16)

static short uLawDecompressionTable[256] =
{
     -32124,-31100,-30076,-29052,-28028,-27004,-25980,-24956,
     -23932,-22908,-21884,-20860,-19836,-18812,-17788,-16764,
     -15996,-15484,-14972,-14460,-13948,-13436,-12924,-12412,
     -11900,-11388,-10876,-10364, -9852, -9340, -8828, -8316,
      -7932, -7676, -7420, -7164, -6908, -6652, -6396, -6140,
      -5884, -5628, -5372, -5116, -4860, -4604, -4348, -4092,
      -3900, -3772, -3644, -3516, -3388, -3260, -3132, -3004,
      -2876, -2748, -2620, -2492, -2364, -2236, -2108, -1980,
      -1884, -1820, -1756, -1692, -1628, -1564, -1500, -1436,
      -1372, -1308, -1244, -1180, -1116, -1052,  -988,  -924,
       -876,  -844,  -812,  -780,  -748,  -716,  -684,  -652,
       -620,  -588,  -556,  -524,  -492,  -460,  -428,  -396,
       -372,  -356,  -340,  -324,  -308,  -292,  -276,  -260,
       -244,  -228,  -212,  -196,  -180,  -164,  -148,  -132,
       -120,  -112,  -104,   -96,   -88,   -80,   -72,   -64,
        -56,   -48,   -40,   -32,   -24,   -16,    -8,     0,
      32124, 31100, 30076, 29052, 28028, 27004, 25980, 24956,
      23932, 22908, 21884, 20860, 19836, 18812, 17788, 16764,
      15996, 15484, 14972, 14460, 13948, 13436, 12924, 12412,
      11900, 11388, 10876, 10364,  9852,  9340,  8828,  8316,
       7932,  7676,  7420,  7164,  6908,  6652,  6396,  6140,
       5884,  5628,  5372,  5116,  4860,  4604,  4348,  4092,
       3900,  3772,  3644,  3516,  3388,  3260,  3132,  3004,
       2876,  2748,  2620,  2492,  2364,  2236,  2108,  1980,
       1884,  1820,  1756,  1692,  1628,  1564,  1500,  1436,
       1372,  1308,  1244,  1180,  1116,  1052,   988,   924,
        876,   844,   812,   780,   748,   716,   684,   652,
        620,   588,   556,   524,   492,   460,   428,   396,
        372,   356,   340,   324,   308,   292,   276,   260,
        244,   228,   212,   196,   180,   164,   148,   132,
        120,   112,   104,    96,    88,    80,    72,    64,
         56,    48,    40,    32,    24,    16,     8,     0
};

/**************************************/
/* Functions used to convert formats. */
/**************************************/

static inline void ucharToShort (short *dest, unsigned char *src, int count)
{
  while (count-- > 0)
    {
      dest[count] = UCHAR_TO_SHORT(src[count]);
    }
}

static inline void scharToShort (short *dest, signed char *src, int count)
{
  while (count-- > 0)
    {
      dest[count] = SCHAR_TO_SHORT(src[count]);
    }
}

static inline void bigIntToShort (short *dest, int *src, int count)
{
  while (count-- > 0)
    {
      dest[count] = BE_INT_TO_SHORT(src[count]);
    }
}

static inline void littleIntToShort (short *dest, int *src, int count)
{
  while (count-- > 0)
    {
      dest[count] = LE_INT_TO_SHORT(src[count]);
    }
}

static inline void uLawToShort (short *dest, signed char *src, int count)
{
  while (count-- > 0)
    {
      dest[count] = uLawDecompressionTable[src[count]];
    }
}

static inline void aLawToShort (short *dest, signed char *src, int count)
{
  return;
}

static inline void byteSwapShort (short *ptr, int count)
{
  short temp;
  while (count-- > 0)
    {
      temp = ptr[count];
      ptr[count] = (short)GSSwapI16(temp);
    }
}

/**************************************/

@implementation NSBundle (NSSoundAdditions)

- (NSString *) pathForSoundResource: (NSString *)name
{
  NSString *ext = [name pathExtension];
  NSString *path = nil;

  if ((ext == nil) || [ext isEqualToString:@""])
    {
      NSArray	*types = [NSSound soundUnfilteredFileTypes];
      unsigned	c = [types count];
      unsigned	i;

      for (i = 0; path == nil && i < c; i++)
	{
	  ext = [types objectAtIndex: i];
	  path = [self pathForResource: name ofType: ext];
	}
    }
  else
    {
      name = [name stringByDeletingPathExtension];
      path = [self pathForResource: name ofType: ext];
    }
  return path;
}

@end 

@interface NSSound (PrivateMethods)

- (id) _initWithData: (NSData *)data range: (NSRange)range
         format: (int)format byteOrder: (NS_ByteOrder)byteOrder;
- (BOOL) _getContext;
- (BOOL) _bufferData;

@end

@implementation NSSound (PrivateMethods)

- (id) _initWithData: (NSData *)data range: (NSRange)range
         format: (int)format byteOrder: (NS_ByteOrder)byteOrder
{
  uint8_t *buffer;
  int currentLocation;
  
  if ([data length] < range.length);
    {
      RELEASE(self
  buffer = (uint8_t *)[data bytes];
  
  _framesCount = _
  
  /* Move buffer pointer to where the sound data start */
  currentLocation = range.location;
  buffer += currentLocation;
  switch (format)
    {
      case GSSoundFormatUnknown: RELEASE(self);
                                 return nil;
                                 break;
      case GSSoundFormatULaw:    _dataSize = range.length<<1; /* Multiply by 2 (sizeof short) */
                                 _frameCount = (_dataSize / _channelCount)>>1;
                                 _data = NSZoneMalloc(NSDefaultMallocZone(), _dataSize);
                                 uLawToShort (_data, buffer, range.length);
      default:                   RELEASE(self);
                                 return nil;
    }
  
  return self;
}

#if defined(HAVE_AL_AL_H)
- (BOOL) getContext
{
  if ((context = alcGetCurrentContext()) == NULL)
    {
      device = alcOpenDevice (NULL);
      if (device == NULL)
        {
          return NO;
        }
      context = alcCreateContext (device, NULL);
      alcMakeContextCurrent (context);
      if (alGetError() != AL_NO_ERROR)
        {
          return NO;
        }
    }
  device = alcGetContextsDevice (context);
  
  return YES;
}

- (BOOL) bufferData
{
  ALenum format;
  
  alGenSources (1, &_sndSource);
  if (alGetError() != AL_NO_ERROR)
    {
      return NO;
    }
  alSourcef (_sndSource, AL_MIN_GAIN, 0.0);
  alSourcef (_sndSource, AL_MAX_GAIN, 1.0);
  if (alGetError() != AL_NO_ERROR)
    {
      return NO;
    }
  
  alGenBuffers (1, &_sndBuffer);
  if (alGetError() != AL_NO_ERROR)
    {
      return NO;
    }
  
  if (_channelCount == 1)
    {
      format = AL_FORMAT_MONO16;
    }
  else if (_channelCount == 2)
    {
      format = AL_FORMAT_STEREO16;
    }
  else
    {
      format = 0;
    }
  alBufferData (_sndBuffer, format, _data, (ALsizei)_dataSize, (ALsizei)_sampleRate);
  if (alGetError() != AL_NO_ERROR)
    {
      NSLog (@"Could not buffer");
      return NO;
    }
  
  alSourcei (_sndSource, AL_BUFFER, _sndBuffer);
  return YES;
}
#else
- (BOOL) createContext
{
  NSLog(@"NSSound: No sound software installed, cannot create context");
  return NO;
}

- (BOOL) bufferData
{
  NSLog(@"NSSound: No sound software installed, cannot buffer sound");
  return NO;
}
#endif

@end

@implementation	NSSound

+ (void) initialize
{
  if (self == [NSSound class]) 
    {
      NSString *path = [NSBundle pathForLibraryResource: @"nsmapping"
	                        			                 ofType: @"strings"
				                                    inDirectory: @"Sounds"];
      [self setVersion: 1];

      nameDict = [[NSMutableDictionary alloc] initWithCapacity: 10];
      
      if (path) 
        {
          nsmapping = RETAIN([[NSString stringWithContentsOfFile: path]
                  propertyListFromStringsFileFormat]);
        }
    }
}

- (void) dealloc
{
  TEST_RELEASE (_data);
  if ((_name != nil) && self == [nameDict objectForKey: _name]) 
    { 
      [nameDict removeObjectForKey: _name];
    }
#if defined(HAVE_AL_AL_H)
  alDeleteSources (1, &_sndSource);
  alDeleteBuffers (1, &_sndBuffer);
#endif
  TEST_RELEASE (_name);
  TEST_RELEASE (_playbackDeviceIdentifier);
  TEST_RELEASE (_channelMap);
  NSZoneFree(NSDefaultMallocZone(), _data);
  [super dealloc];
}

//
// Creating an NSSound 
//
- (id) initWithContentsOfFile: (NSString *)path byReference:(BOOL)byRef
{
  self = [super init];
	
  if (!self) 
    {
      return nil;
    }
  
  _onlyReference = byRef;			
  ASSIGN (_name, [path lastPathComponent]);
  _playbackDeviceIdentifier = nil;
  _channelMap = nil;
  
#if defined (HAVE_SNDFILE_H)
  {
    SNDFILE *file;
    SF_INFO fileInfo;
    long dataRead;

    fileInfo.format = 0;
    if ((file = sf_open([path fileSystemRepresentation], SFM_READ, &fileInfo)) == NULL)
      {
        NSLog (@"Could not open file %@ for reading!". path);
        DESTROY(self);
        return nil;
      }
    
    _channelCount = (int) fileInfo.channels;
    _sampleRate = (int) fileInfo.samplerate;
    _frameCount = (long) fileInfo.frames;
    _duration = ((double) _frameCount) / ((double) _sampleRate);
		
		/* FIXME: Need more error checking around here. */
    _data = NSZoneMalloc(NSDefaultMallocZone(), (_frameCount * _channelCount * sizeof(short)));
    dataRead = sf_read_short (file, _data, (_frameCount * _channelCount));
    _dataSize = dataRead * (long)sizeof(short);
    
    sf_close (file);
  }
#else
  {
    NSData *fileData;
    
    fileData = [NSData dataWithContentsOfFile: path];
    if (!fileData)
      {
        NSLog (@"Could not get sound data from: %@", path);
        DESTROY(self);
        return nil;
      }
    
    [self initWithData: fileData];
    RELEASE(fileData);
  }
#endif  
  
  if ([self getContext] == NO)
    {
      NSLog (@"Could not get context");
      DESTROY (self);
      return nil;
    }
  if ([self bufferData] == NO)
    {
      NSLog (@"Could not buffer data!");
      DESTROY (self);
      return nil;
    }
	
  return self;
}

- (id) initWithContentsOfURL: (NSURL *)url byReference:(BOOL)byRef
{
  _onlyReference = byRef;	
  return [self initWithData: [NSData dataWithContentsOfURL: url]];
}

- (id) initWithData: (NSData *)data
{
  /* FIXME: If libsndfile is available use it to load data instead of fall back code */
  if ([isa _soundIsAU: data])
    {
      return [self _initSoundWithAU: data];
    }
  if ([isa _soundIsWAV: data])
    {
      return [self _initSoundWithWAV: data];
    }
  if ([isa _soundIsAIFF: data])
    {
      return [self _initSoundWithAIFF: data];
    }
  RELEASE(self);
  return nil;
}

- (id) initWithPasteboard: (NSPasteboard *)pasteboard
{
  if ([NSSound canInitWithPasteboard: pasteboard] == YES) 
    {	
      NSData *d = [pasteboard dataForType: @"NSGeneralPboardType"];	
      return [self initWithData: d];	
    }
  return nil;
}

//
// Playing and Information
//
#if defined(HAVE_AL_AL_H)

- (BOOL) pause 
{
  alSourcePause (_sndSource);
  return (alGetError() == AL_NO_ERROR ? YES : NO);
}

- (BOOL) play
{
  alSourcePlay (_sndSource);
  return (alGetError() == AL_NO_ERROR ? YES : NO);
}

- (BOOL) resume
{
  ALint value;
  alGetSourcei (_sndSource, AL_SOURCE_STATE, &value);
  if (value == AL_PLAYING)
    {
      return YES;
    }
  alSourcePlay (_sndSource);
  return (alGetError() == AL_NO_ERROR ? YES : NO);
}

- (BOOL) stop
{
  alSourceStop (_sndSource);
  return (alGetError() == AL_NO_ERROR ? YES : NO);
}

- (BOOL) isPlaying
{
  ALint value;
  alGetSourcei (_sndSource, AL_SOURCE_STATE, &value);
  return (value == AL_PLAYING ? YES : NO);
}

- (float) volume
{
  ALfloat value;
  alGetSourcef (_sndSource, AL_GAIN, &value);
  return (float)value;
}

- (void) setVolume: (float) volume
{
  alSourcef (_sndSource, AL_GAIN, (ALfloat)volume);
}
  
- (NSTimeInterval) currentTime
{
  ALfloat value;
  alGetSourcef (_sndSource, AL_SEC_OFFSET, &value);
  return (NSTimeInterval)value;
}

- (void) setCurrentTime: (NSTimeInterval) currentTime
{
  alSourcef (_sndSource, AL_SEC_OFFSET, (ALfloat)currentTime);
}

- (BOOL) loops
{
  ALint value;
  alGetSourcei (_sndSource, AL_LOOPING, &value);
  return (value == AL_TRUE ? YES : NO);
}

- (void) setLoops: (BOOL) loops
{
  ALint value;
  (loops == YES) ? (value = AL_TRUE) : (value = AL_FALSE);
  alSourcei (_sndSource, AL_LOOPING, value);
}

#else

- (BOOL) pause 
{
  return NO;
}

- (BOOL) play
{
  return NO;
}

- (BOOL) resume
{
  return NO;
}

- (BOOL) stop
{
  return NO;
}

- (BOOL) isPlaying
{
  return NO;
}

- (float) volume
{
  return 0.0;
}

- (void) setVolume: (float) volume
{
  return;
}

- (NSTimeInterval) currentTime;
{
  return 0.0;
}

- (void) setCurrentTime: (NSTimeInterval) currentTime
{
  return;
}

- (BOOL) loops;
{
  return NO;
}

- (void) setLoops: (BOOL) loops
{
  return;
}

#endif

- (NSTimeInterval) duration
{
  return _duration;
}

//
// Working with pasteboards 
//
+ (BOOL) canInitWithPasteboard: (NSPasteboard *)pasteboard
{
  NSArray *pbTypes = [pasteboard types];
  NSArray *myTypes = [NSSound soundUnfilteredPasteboardTypes];

  return ([pbTypes firstObjectCommonWithArray: myTypes] != nil);
}

+ (NSArray *) soundUnfilteredPasteboardTypes
{
  return [NSArray arrayWithObjects: @"NSGeneralPboardType", nil];
}

- (void) writeToPasteboard: (NSPasteboard *)pasteboard
{
  NSData *d = [NSArchiver archivedDataWithRootObject: self];

  if (d != nil) {
    [pasteboard declareTypes: [NSSound soundUnfilteredPasteboardTypes] 
		owner: nil];
    [pasteboard setData: d forType: @"NSGeneralPboardType"];
  }
}

//
// Working with delegates 
//
- (id) delegate
{
  return _delegate;
}

- (void) setDelegate: (id)aDelegate
{
  _delegate = aDelegate;
}

//
// Naming Sounds 
//
+ (id) soundNamed: (NSString*)name
{
  NSString	*realName = [nsmapping objectForKey: name];
  NSSound	*sound;

  if (realName) 
    {
      name = realName;
    }
	
  sound = (NSSound *)[nameDict objectForKey: name];
 
  if (sound == nil) 
    {
      NSString	*extension;
      NSString	*path = nil;
      NSBundle	*main_bundle;
      NSArray	*array;
      NSString	*the_name = name;

      // FIXME: This should use [NSBundle pathForSoundResource], but this will 
      // only allow soundUnfilteredFileTypes.
      /* If there is no sound with that name, search in the main bundle */
			
      main_bundle = [NSBundle mainBundle];
      extension = [name pathExtension];
		
      if (extension != nil && [extension length] == 0) 
	{
	  extension = nil;
	}

      /* Check if extension is one of the sound types */
      array = [NSSound soundUnfilteredFileTypes];
	
      if ([array indexOfObject: extension] != NSNotFound) 
	{
	  /* Extension is one of the sound types
	     So remove from the name */
	  the_name = [name stringByDeletingPathExtension];
	} 
      else 
	{
	  /* Otherwise extension is not an sound type
	     So leave it alone */
	  the_name = name;
	  extension = nil;
	}

      /* First search locally */
      if (extension) 
	{
	  path = [main_bundle pathForResource: the_name ofType: extension];
	} 
      else 
	{
	  id o, e;

	  e = [array objectEnumerator];
	  while ((o = [e nextObject])) 
	    {
	      path = [main_bundle pathForResource: the_name ofType: o];
	      if (path != nil && [path length] != 0) 
		{
		  break;
		}
	    }
	}

      /* If not found then search in system */
      if (!path) 
	{
	  if (extension) 
	    {
	      path = [NSBundle pathForLibraryResource: the_name
				               ofType: extension
				          inDirectory: @"Sounds"];
	    } 
	  else 
	    {
	      id o, e;

	      e = [array objectEnumerator];
	      while ((o = [e nextObject])) {
	      path = [NSBundle pathForLibraryResource: the_name
				               ofType: o
				          inDirectory: @"Sounds"];
		if (path != nil && [path length] != 0) 
		  {
		    break;
		  }
	      }
	    }
	}

      if ([path length] != 0) 
	{
	  sound = [[self allocWithZone: NSDefaultMallocZone()]
		    initWithContentsOfFile: path byReference: NO];

	  if (sound != nil) 
	    {
	      [sound setName: name];
	      RELEASE(sound);	
	      sound->_onlyReference = YES;
	    }

	  return sound;
	}
    }  
	
  return sound;
}

+ (NSArray *) soundUnfilteredFileTypes
{
  return [NSArray arrayWithObjects: @"wav", @"aiff", @"aifc", @"aif", @"au", @"snd",
#if defined(HAVE_SNDFILE_H)
  @"mat", @"mat4", @"mat5", @"paf", @"sf", @"voc", @"w64", @"xi", @"caf", @"sd2", @"flac", @"ogg", @"oga",
#endif
  nil];
}

+ (NSArray *) soundUnfilteredTypes
{
  return [NSArray arrayWithObjects: nil];
}

- (NSString *) name
{
  return _name;
}

- (BOOL) setName: (NSString *)aName
{
  BOOL retained = NO;
  
  if (!aName || [nameDict objectForKey: aName]) 
    {
      return NO;
    }
	
  if ((_name != nil) && self == [nameDict objectForKey: _name]) 
    {
      /* We retain self in case removing from the dictionary releases
	 us */
      RETAIN (self);
      retained = YES;
      [nameDict removeObjectForKey: _name];
    }
  
  ASSIGN(_name, aName);
  
  [nameDict setObject: self forKey: _name];
  if (retained) 
    {
      RELEASE (self);
    }
  
  return YES;
}

- (NSString *) playbackDeviceIdentifier
{
  [self notImplemented: _cmd];
  return nil;
}

- (void) setPlaybackDeviceIdentifier: (NSString *) playbackDeviceIdentifier
{
  [self notImplemented: _cmd];
}

- (NSArray *) channelMapping
{
  [self notImplemented: _cmd];
  return nil;
}

- (void) setChannelMapping: (NSArray *) channelMapping
{
  [self notImplemented: _cmd];
}

//
// NSCoding 
//
- (void) encodeWithCoder: (NSCoder *)coder
{
  if ([coder allowsKeyedCoding])
    {
      // TODO_NIB: Determine keys for NSSound.
    }
  else
    {
      [coder encodeValueOfObjCType: @encode(BOOL) at: &_onlyReference];
      [coder encodeObject: _name];
      
      if (_onlyReference == YES) 
      	{
	        return;
      	}
      [coder encodeConditionalObject: _delegate];
      [coder encodeValueOfObjCType: @encode(NSTimeInterval) at: &_duration];
      [coder encodeValueOfObjCType: @encode(long) at: &_dataSize];
      [coder encodeValueOfObjCType: @encode(long) at: &_frameCount];
      [coder encodeValueOfObjCType: @encode(int) at: &_sampleRate];
      [coder encodeValueOfObjCType: @encode(int) at: &_channelCount];
      [coder encodeValueOfObjCType: @encode(unsigned int) at: &_sndSource];
      [coder encodeValueOfObjCType: @encode(unsigned int) at: &_sndBuffer];
      /* FIXME: need to encode _data */
      
      [coder encodeObject: _playbackDeviceIdentifier];
    }
}

- (id) initWithCoder: (NSCoder*)decoder
{	
  if ([decoder allowsKeyedCoding])
    {
      // TODO_NIB: Determine keys for NSSound.
    }
  else
    {
      [decoder decodeValueOfObjCType: @encode(BOOL) at: &_onlyReference];
      
      if (_onlyReference == YES) 
	      {
	        NSString *theName = [decoder decodeObject];
	        
	        RELEASE (self);
	        self = RETAIN ([NSSound soundNamed: theName]);
	        [self setName: theName];	
	      } 
      else 
	      {
	        _name = TEST_RETAIN ([decoder decodeObject]);
	        [self setDelegate: [decoder decodeObject]];
	        
	        [decoder decodeValueOfObjCType: @encode(NSTimeInterval) at: &_duration];
          [decoder decodeValueOfObjCType: @encode(long) at: &_dataSize];
          [decoder decodeValueOfObjCType: @encode(long) at: &_frameCount];
          [decoder decodeValueOfObjCType: @encode(int) at: &_sampleRate];
          [decoder decodeValueOfObjCType: @encode(int) at: &_channelCount];
          [decoder decodeValueOfObjCType: @encode(unsigned int) at: &_sndSource];
          [decoder decodeValueOfObjCType: @encode(unsigned int) at: &_sndBuffer];
	        /* FIXME: need to decode _data */
	        
	        _playbackDeviceIdentifier = RETAIN([decoder decodeObject]);
	      }
    }
  return self;
}

- (id) awakeAfterUsingCoder: (NSCoder *)coder
{
  return self;
}

//
// NSCopying 
//
- (id) copyWithZone: (NSZone *)zone
{
  /* FIXME: need to copy _data and all the other stuff */
  NSSound *newSound = (NSSound *)NSCopyObject(self, 0, zone);
	
  newSound->_name = [_name copyWithZone: zone];
	
  return newSound;
}

@end
