#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <math.h>

#include <grass/gis.h>

#define PROGVERSION 1.0

int
main (int argc, char *argv[])
{
	struct GModule *module;
	
	struct
	{
		struct Option *input;
		struct Option *output;
		struct Option *radius;
	} parm;
	
	struct
	{
		struct Flag *absolute;		
		struct Flag *localnorm;
		struct Flag *rect;
		struct Flag *single;
		struct Flag *quiet;
	} flag;
			
	char *mapset;
	int i, j, radius, n, x, y;
	int a, b, dist;
	int nullvalue;
	double val;
	double centerval;
	double min, max;
	float ffrom, fto;
	double dfrom, dto;
	struct FPRange drange;
	int nrows, ncols;
	int fd, fd_out;
	int type_in, type_out;
	struct Cell_head region;
	struct Colors colors;
	DCELL *ddiskrow = NULL;
	DCELL *doutrow = NULL;
	FCELL *fdiskrow = NULL;
	FCELL *foutrow = NULL;


	module = G_define_module ();
	module->description = "Normalizes raster cell values globally (default) or within a neighborhood";
	/* setup some basic GIS stuff */
	G_gisinit (argv[0]);
	/* do not pause after a warning message was displayed */
	G_sleep_on_error (0);
		
	parm.input = G_define_standard_option(G_OPT_R_INPUT);
	parm.input->required = YES;
	parm.input->description = "Raster map to normalize";
	
	parm.output = G_define_standard_option(G_OPT_R_OUTPUT);
	parm.output->required = YES;
	parm.output->description = "Raster to store output";	
	
	parm.radius = G_define_option ();
	parm.radius->key = "radius";
	parm.radius->type = TYPE_INTEGER;
	parm.radius->required = YES;
	parm.radius->description = "Radius of neighborhood in map cells";

	flag.absolute = G_define_flag ();
	flag.absolute->key = 'a';
	flag.absolute->description = "Normalize all data to positive range [0;1]";

	flag.localnorm = G_define_flag ();
	flag.localnorm->key = 'l';
	flag.localnorm->description = "Perform only local data normalization";
	
	flag.rect = G_define_flag ();
	flag.rect->key = 'r';
	flag.rect->description = "Use square neighborhood (instead of circular) for local normalization";

	flag.single = G_define_flag ();
	flag.single->key = 's';
	flag.single->description = "Always produce single precision floating point output";

	flag.quiet = G_define_flag ();
	flag.quiet->key = 'q';
	flag.quiet->description = "Disable on-screen progress display";

		
	/* parse command line */
	if (G_parser (argc, argv))
	{
		exit (-1);
	}

	G_get_window (&region);
	nrows = G_window_rows ();
	ncols = G_window_cols ();
	
	/* check parameters for validity */
	radius = atoi (parm.radius->answer);
	if (( radius < 1 ) || ( radius > nrows/2 ) || ( radius > ncols/2 )) {
		G_fatal_error ("Radius must be > 1 and smaller than half the map's extent in both directions.");
	}
	mapset = G_calloc (512, sizeof (char));
	mapset = G_find_cell (parm.input->answer, "");
	if ( mapset == NULL) {
		G_fatal_error ("Input map does not exist in the current location.");
	}
	fd = G_open_cell_old (parm.input->answer, mapset);	
	if (fd < 0) {
		G_fatal_error ("Could not open input map for reading.\n");
	}	
	if (!G_legal_filename (parm.output->answer)) {
		G_fatal_error ("%s is not a legal filename for an output map.\n",
				parm.output->answer);
	}
        type_in = G_get_raster_map_type (fd);
	if ( type_in == CELL_TYPE ) {
		G_fatal_error ("Only floating point input maps are supported.\n");
	}
	if ( flag.single->answer ) {
		type_out = FCELL_TYPE;
	} else {
		type_out = type_in;
	}
	fd_out = G_open_raster_new (parm.output->answer, type_out);
	if ( fd_out < 0 ) {
		G_fatal_error ("Could not open output map for writing.\n");
	}
	
	if ( type_in == DCELL_TYPE ) {
		ddiskrow = G_allocate_d_raster_buf ();
	} else {
		fdiskrow = G_allocate_f_raster_buf ();		
	}
	if ( type_out == DCELL_TYPE ) {
		doutrow = G_allocate_d_raster_buf ();
	} else {
		foutrow = G_allocate_f_raster_buf ();
	}
	
	/* initialize statistics */
	n = 0;
	min = 0;
	max = 0;
	G_init_fp_range(&drange);
	
	if ( G_read_fp_range ( parm.input->answer, mapset, &drange ) < 0 ) {
		G_fatal_error ("%s does not contain valid range (min/max) data.\n",
				parm.output->answer);
	}
	min = (double) drange.min;
	max = (double) drange.max;

	/* GLOBAL NORMALIZATION (DEFAULT) */
	if (!flag.localnorm->answer) {
		if (!flag.quiet->answer) {
			fprintf (stdout,"\nNormalizing data by global map maximum:\n");
			fflush (stdout);
		}
		/* normalise data within map */
		for (y = 0; y < nrows; y ++) {
			if ( type_in = DCELL_TYPE ) {
				G_get_d_raster_row (fd, ddiskrow, y);
			} else {
				G_get_f_raster_row (fd, fdiskrow, y);
			}
			for (x = 0; x < ncols; x++) {
				/* write one cell to output map */
				if (	(type_in == DCELL_TYPE) && !G_is_d_null_value (&ddiskrow[x])
					|| 
					(type_in == FCELL_TYPE) && !G_is_f_null_value (&fdiskrow[x])) {
					if (flag.absolute->answer) {
						if ( type_in == DCELL_TYPE ) {
							val = (double) (ddiskrow [x] - min) / ( max - min );
						} else {
							val = (double) (fdiskrow [x] - min) / ( max - min );
						}
						if ( type_out == DCELL_TYPE ) {
							doutrow [x] = (ddiskrow [x] - min) / ( max - min );
						} else {
							foutrow [x] = (fdiskrow [x] - min) / ( max - min );
						}
					} else {
						if ( min < 0 ) {
							if ( type_in == DCELL_TYPE ) {
								if ( ddiskrow [x] < 0 ) {
									val = ddiskrow [x] / fabs (min);
								}
								if ( ddiskrow [x] > 0 ) {
									val = ddiskrow [x] / max;
								}
								if ( ddiskrow [x] == 0 ) {
  									val = 0.0;
								}
								
							} else {
								if ( fdiskrow [x] < 0 ) {
									val = fdiskrow [x] / fabs (min);
								}
								if ( fdiskrow [x] > 0 ) {
									val = fdiskrow [x] / max;
								}
								if ( fdiskrow [x] == 0 ) {
  									val = 0.0;
								}
								foutrow[x] = val;
							}
							if ( type_out == DCELL_TYPE ) {
								doutrow[x] = val;
							} else {
								foutrow[x] = val;
							}
						} else {
							if ( type_in == DCELL_TYPE ) {
								val = (ddiskrow [x] - min) / ( max - min ); 
							} else {
								val = (fdiskrow [x] - min) / ( max - min ); 
							}
							if ( type_out == DCELL_TYPE ) {
								doutrow [x] = (val - min) / ( max - min );
							} else {
								foutrow [x] = (val - min) / ( max - min );
							}
						}
					}
				} else {
					if ( type_out == DCELL_TYPE ) {
						G_set_d_null_value ( &doutrow[x],1 );
					} else {
						G_set_f_null_value ( &foutrow[x],1 );
					}
				}
			}
			/* write one row to output map on disk */
			if ( type_out == DCELL_TYPE ) {
				G_put_raster_row (fd_out, doutrow, DCELL_TYPE);
			} else {
				G_put_raster_row (fd_out, foutrow, FCELL_TYPE);
			}
		
			if (!flag.quiet->answer) {
				G_percent (y, nrows-1, 1);
				fflush (stdout);
			}
		}
		G_close_cell (fd);
		G_close_cell (fd_out);
	}


	/* LOCAL NORMALIZATION */
	if (flag.localnorm->answer) {
		if (!flag.quiet->answer) {
			fprintf (stdout,"\nNormalizing data by neighborhood maximum:\n");
			fflush (stdout);
		}		
		i = 0;
		j = 0;
		for (y = 0; y < nrows; y ++) {		
			for (x = 0; x < ncols; x++) {
				if ( type_in == DCELL_TYPE ) {
					G_get_d_raster_row (fd, ddiskrow, y);
				} else {
					G_get_f_raster_row (fd, fdiskrow, y);
				}
				if ( type_in == DCELL_TYPE ) {
					centerval = ddiskrow [x];
					min = ddiskrow [x];
					max = ddiskrow [x];
				} else {
					centerval = fdiskrow [x];
					min = fdiskrow [x];
					max = fdiskrow [x];
				}
				if (  	(type_in == DCELL_TYPE && G_is_d_null_value ( &ddiskrow[x] ))
					|| 
					(type_in == FCELL_TYPE && G_is_f_null_value ( &fdiskrow[x] )) ) {
					nullvalue = 1;
				} else {
					nullvalue = 0;
					for (j = y-radius; j <= y+radius; j++) {
						if ((j>0) && (j<nrows)) {
							if ( type_in == DCELL_TYPE ) {
								G_get_d_raster_row (fd, ddiskrow, j);
							} else {
								G_get_f_raster_row (fd, fdiskrow, j);
							}
							nullvalue = 0;
							for (i = x-radius; i <= x+radius; i++) {
								if ( ( (i!=x) || (j!=y)  )  && (i>0) && (i<ncols)) {
									if ( 	(type_in == DCELL_TYPE) && (!G_is_d_null_value (&ddiskrow[i])) 
										|| 
										(type_in == FCELL_TYPE) && (!G_is_f_null_value (&fdiskrow[i]))) {
										if (flag.rect->answer) {
											/* use rectangular neighborhood */
											if ( type_in == DCELL_TYPE ) {
												if ( ddiskrow [i] > max ) {
													max = ddiskrow [i];
												}
												if ( ddiskrow [i] < min ) {
													min = ddiskrow [i];
												}
											} else {
												if ( fdiskrow [i] > max ) {
													max = fdiskrow [i];
												}
												if ( fdiskrow [i] < min ) {
													min = fdiskrow [i];
												}
											}
										} else {
											/* use circular neighborhood */
											a = abs ( x - i );
											b = abs ( y - j );
											dist = (int) sqrt ( (double) (a*a) + (b*b) );
											if (dist <= radius) {
												if ( type_in == DCELL_TYPE ) {
													if ( ddiskrow[i] > max ) {
														max = ddiskrow [i];
													}
													if ( ddiskrow[i] < min ) {
														min = ddiskrow [i];
													}
												} else {
													if ( fdiskrow[i] > max ) {
														max = fdiskrow [i];
													}
													if ( fdiskrow[i] < min ) {
														min = fdiskrow [i];
													}
												}
											}
										}
									}
								}	
							}
						}
					}
				}
				/* write one cell to output map */
				if (!nullvalue) {
					if (flag.absolute->answer) {
						if ( type_out == DCELL_TYPE ) {
							doutrow [x] = (centerval - min) / ( max - min );
						} else {
							foutrow [x] = (centerval - min) / ( max - min );
						}
					} else {
						if ( min < 0 ) {
							if ( centerval < 0 ) {
								val = centerval / fabs (min);
							}
							if ( centerval > 0 ) {
								val = centerval / max;
							}
							if ( centerval == 0 ) {
								val = 0;
							}

							if ( type_out == DCELL_TYPE ) {
								doutrow[x] = val;
							} else {
								foutrow[x] = val;
							}

						} else {
							if ( type_out == DCELL_TYPE ) {
								doutrow [x] = (centerval - min) / ( max - min );
							} else {
								foutrow [x] = (centerval - min) / ( max - min );
							}
						}
					}
				} else {
					if ( type_out == DCELL_TYPE ) {
						G_set_d_null_value ( &doutrow[x],1 );
					} else {
						G_set_f_null_value ( &foutrow[x],1 );
					}
				}
				
			}
			/* write one row to output map on disk */
			if ( type_out == DCELL_TYPE ) {
				G_put_raster_row (fd_out, doutrow, DCELL_TYPE);
			} else {
				G_put_raster_row (fd_out, foutrow, FCELL_TYPE);
			}
		
			if (!flag.quiet->answer) {
				G_percent (y, nrows-1, 1);
				fflush (stdout);
			}
		}
		G_close_cell (fd);
		G_close_cell (fd_out);		
	}


	/* assign color scale */
	if ( flag.absolute->answer) {
		/* in -a mode, we only have positive values */
		min = 0;
		max = 1.0;
	} else {
		min = -1.0;
		max = 1.0;
	}

	G_init_colors (&colors);
	if ( type_out == DCELL_TYPE ) {
		if  ( (flag.absolute->answer) ) {
			/* write a colormap in grey shades: lowest value gets black, highest white color */
			dfrom = min;
			dto = max/2;
			G_add_raster_color_rule (&dfrom, 0, 0, 0, &dto, 127, 127, 127, &colors, type_out);
			dfrom = max/2;
			dto = max;
			G_add_raster_color_rule (&dfrom, 128, 128, 128, &dto, 255,  255, 255, &colors, type_out);
		} else {
			/* write a colormap from red (-1) thru black (0) to white (1) */
			if ( min < 0 ) {
				dfrom = min;
				dto = min/2;
				G_add_raster_color_rule (&dfrom, 255, 0, 0, &dto, 127,  0, 0, &colors, type_out);
				dfrom = min/2;
				dto = 0;
				G_add_raster_color_rule (&dfrom, 127, 0, 0, &dto, 0,  0, 0, &colors, type_out);
				dfrom = 0;
			}
			else {
				dfrom = min;
			}
			dto = max/2;
			G_add_raster_color_rule (&dfrom, 0, 0, 0, &dto, 127,  127, 127, &colors, type_out);
			dfrom = max/2;
			dto = max;
			G_add_raster_color_rule (&dfrom, 128, 128, 128, &dto, 255,  255, 255, &colors, type_out);
		}
	} else {
		if  ( (flag.absolute->answer) ) {
			/* write a colormap in grey shades: lowest value gets black, highest white color */
			ffrom = min;
			fto = max/2;
			G_add_raster_color_rule (&ffrom, 0, 0, 0, &fto, 127, 127, 127, &colors, type_out);
			ffrom = max/2;
			fto = max;
			G_add_raster_color_rule (&ffrom, 128, 128, 128, &fto, 255,  255, 255, &colors, type_out);
		} else {
			/* write a colormap from red (-1) thru black (0) to white (1) */
			if ( min < 0 ) {
				ffrom = min;
				fto = min/2;
				G_add_raster_color_rule (&ffrom, 255, 0, 0, &fto, 127,  0, 0, &colors, type_out);
				ffrom = min/2;
				fto = 0;
				G_add_raster_color_rule (&ffrom, 127, 0, 0, &fto, 0,  0, 0, &colors, type_out);
				ffrom = 0;
			}
			else {
				ffrom = min;
			}
			fto = max/2;
			G_add_raster_color_rule (&ffrom, 0, 0, 0, &fto, 127,  127, 127, &colors, type_out);
			ffrom = max/2;
			fto = max;
			G_add_raster_color_rule (&ffrom, 128, 128, 128, &fto, 255,  255, 255, &colors, type_out);
		}
	}

	G_write_colors (parm.output->answer, G_mapset(), &colors);
	G_free_colors (&colors);

	/* CLEAN UP */
	if ( type_in == DCELL_TYPE ) {
		G_free (ddiskrow);
	} else {
		G_free (fdiskrow);
	}
	if ( type_out == DCELL_TYPE ) {
		G_free (doutrow);
	} else {
		G_free (foutrow);
	}
	free (mapset);

	if (!flag.quiet->answer) {
		fprintf (stdout,"\nDone.\n");
		fflush (stdout);
	}
				
	return (EXIT_SUCCESS);
}
