/*      bit_vector.c
 *
 * Copyright (C) 2006, 2007, 2008 Ivo Alxneit
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <stdlib.h>

#include "bit_vector.h"
#include "bv.h"


struct bit_vector *
bit_vector_init ()
/*
 * returns a bit_vector initially holding 8*sizeof(int) bits. returns NULL upon
 * failure. all 8*sizeof(int) bits (0..8*sizeof(int)-1) are cleared upon
 * initialization.
 */
{
  struct bit_vector *bv;


  bv = (struct bit_vector *) malloc (sizeof (struct bit_vector));
  if (bv == NULL)
    return NULL;

  bv->v = (int *) malloc (sizeof (int));
  if (bv->v == NULL)
    {
      free (bv);
      return NULL;
    }

  bv->len = BITS_PER_CHUNK - 1;
  bv->v[0] = 0;
  bv->implied_bits = 0;

  return bv;

}


struct bit_vector *
bit_vector_init_all_set ()
/*
 * returns a bit_vector initially holding 8*sizeof(int) bits. returns NULL upon
 * failure. all bits are set upon initialization (including implied bits).
 */
{
  struct bit_vector *bv;


  bv = (struct bit_vector *) malloc (sizeof (struct bit_vector));
  if (bv == NULL)
    return NULL;

  bv->v = (int *) malloc (sizeof (int));
  if (bv->v == NULL)
    {
      free (bv);
      return NULL;
    }

  bv->len = BITS_PER_CHUNK - 1;
  bv->v[0] = ALL_BITS_SET;
  bv->implied_bits = 1;

  return bv;

}


void
bit_vector_free (struct bit_vector *bv)
/*
 * frees the bit_vector.
 */
{

  free (bv->v);
  free (bv);

  bv = NULL;
}


int
bit_vector_set_bit (struct bit_vector *bv, const size_t bit)
/*
 * sets bit number 'bit' in bit_vector. if necessary the length of the
 * vector is increased.
 * the function returns zero upon success or ENOMEM if memory allocation
 * failed.
 */
{
  size_t offset;
  size_t nr;

  int mask;


  get_addr (bit, &offset, &nr);

  if (bit > bv->len)
    {                           /* implied bit */

      if (bv->implied_bits == 1)        /* bit allready set */
        return 0;

      /*
       * we set an implicitly cleared bit.
       * allocated new memory
       */
      if (increase_length (bv, bit))
        return ENOMEM;

    }

  /* set bit */
  mask = (1 << nr);
  bv->v[offset] |= mask;

  if (bv->implied_bits)
    try_shrink (bv);

  return 0;

}


int
bit_vector_clear_bit (struct bit_vector *bv, const size_t bit)
/*
 * clears bit number 'bit' in bit_vector.
 * if we fail to allocate new memory (clearing an implicitly set bit)
 * we return ENOMEM and bv is not modified.
 */
{
  size_t offset;
  size_t nr;

  int mask;


  get_addr (bit, &offset, &nr);

  if (bit > bv->len)
    {                           /* implied bit */

      if (bv->implied_bits == 0)        /* bit is not set */
        return 0;

      /*
       * we have to clear an implicitly set bit. we have to allocate
       * new memory to keep bits len..bit in memory.
       */
      if (increase_length (bv, bit))
        return ENOMEM;

    }

  /* prepare mask */
  mask = ~(1 << nr);

  bv->v[offset] &= mask;        /* clear bit */

  if (!bv->implied_bits)
    try_shrink (bv);

  return 0;

}


int
bit_vector_test_bit (const struct bit_vector *bv, const size_t bit)
/*
 * the function test whether bit number 'bit' is set.
 * the function returns 1 if the bit is set and zero otherwise.
 */
{
  size_t offset;
  size_t nr;

  int mask;


  if (bit > bv->len)
    return bv->implied_bits;

  get_addr (bit, &offset, &nr);

  mask = (1 << nr);

  if (bv->v[offset] & mask)
    return 1;
  else
    return 0;

}
