/*      bit_vector_logical.c
 *
 * Copyright (C) 2007, 2008 Ivo Alxneit
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include "bit_vector_logical.h"
#include "bv.h"

int
bit_vector_and (const struct bit_vector *src, struct bit_vector *dest)
/*
 * this function performs a bit-wise AND of the two bit_vectors 'src' and
 * 'dest'. it resturns the result as 'dest' and keeps 'src' unchanged.
 * if the the number of explicitly allocated bits in 'dest' is different from
 * 'src' memory is adjusted.
 * return 0 upon sucess or ENOMEM (and dest is unchanged).
 */
{
  size_t n, i;

  int dest_grown = 0;

  (void) get_addr;              /* prevent warnings "defined but not used" */

  /* index of last allocated integer in smaller bit_vector */
  n = MIN (src->len, dest->len) / BITS_PER_CHUNK;

  /*
   * adjust dest:
   *
   *   dest->len < src->len:
   *            increase only if dest->implied_bits==1!
   *
   *            if dest->implied_bits==0 we have:
   *            src bit (x) AND implied dest bit (0) results in 0 and
   *            we treat these bits as implied 0 and do not grow dest.
   *
   *    dest->len > src->len:
   *            decrease only if src->implied_bits==0!
   *
   *            if src->implied_bits==1 we have:
   *            dest bit (x) AND implied src bit (1) results in x. thus
   *            we keep the original dest bits and do not shrink dest.
   */

  if ((dest->len < src->len) && (dest->implied_bits == 1))
    {

      if (increase_length (dest, src->len))
        return ENOMEM;

      dest_grown = 1;

    }
  else if ((dest->len > src->len) && (src->implied_bits == 0))
    shrink_to (dest, n + 1);

  /* perform AND on initially allocated bits (both in src and dest) */
  for (i = 0; i <= n; i++)
    dest->v[i] &= src->v[i];

  /*
   * if dest has grown we copy the rest of the src bits.
   * (resulting from AND with implied 1 of dest)
   */
  if (dest_grown)
    {

      for (; i <= dest->len / BITS_PER_CHUNK; i++)
        dest->v[i] = src->v[i];

    }

  /* AND implied bits of src and dest */
  dest->implied_bits &= src->implied_bits;

  try_shrink (dest);

  return 0;

}


int
bit_vector_or (const struct bit_vector *src, struct bit_vector *dest)
/*
 * this function performs a bit-wise OR of the two bit_vectors 'src' and
 * 'dest'. it returns the result (length adjusted) as 'dest' and keeps
 * 'src' unchanged.
 * ENOMEM is returned if allocation of additional memory has failed.
 */
{
  size_t n;
  size_t i;

  int dest_grown = 0;


  /* index of last allocated integer in smaller bit_vector */
  n = MIN (src->len, dest->len) / BITS_PER_CHUNK;

  /*
   * adjust dest:
   *
   *   dest->len < src->len:
   *            increase only if dest->implied_bits==0!
   *
   *            if dest->implied_bits==1 we have:
   *            src bit (x) OR implied dest bit (0) results in 1 and
   *            we treat these bits as implied 1 and do not grow dest.
   *
   *    dest->len > src->len:
   *            decrease only if src->implied_bits==1!
   *
   *            if src->implied_bits==0 we have:
   *            dest bit (x) OR implied src bit (0) results in x. thus
   *            we keep the original dest bits and do not shrink dest.
   */

  if ((dest->len < src->len) && (dest->implied_bits == 0))
    {

      if (increase_length (dest, src->len))
        return ENOMEM;

      dest_grown = 1;

    }
  else if ((dest->len > src->len) && (src->implied_bits == 1))
    shrink_to (dest, n + 1);

  /* perform OR on initially allocated bits (both in src and dest) */
  for (i = 0; i <= n; i++)
    dest->v[i] |= src->v[i];

  /*
   * if dest has grown we copy the rest of the src bits.
   * (resulting from OR with implied 0 of dest)
   */
  if (dest_grown)
    {

      for (; i <= dest->len / BITS_PER_CHUNK; i++)
        dest->v[i] = src->v[i];

    }

  /* OR implied bits of src and dest */
  dest->implied_bits |= src->implied_bits;

  return 0;

}


int
bit_vector_xor (const struct bit_vector *src, struct bit_vector *dest)
/*
 * this function performs a bit-wise XOR of the two bit_vectors 'src' and
 * 'dest'. it returns the result (length adjusted) as 'dest' and keeps
 * 'src' unchanged.
 * ENOMEM is returned if allocation of additional memory has failed.
 */
{
  size_t i;


  /* adjust dest */
  if (dest->len < src->len)
    {

      if (increase_length (dest, src->len))
        return ENOMEM;

    }

  /* XOR common explicit bits (after possible increase of dest */
  for (i = 0; i <= src->len / BITS_PER_CHUNK; i++)
    dest->v[i] ^= src->v[i];

  /* XOR possible remaining explicit bits with implicit bits of src */
  if (dest->len > src->len)
    {
      int mask = 0;

      /*
       * prepare mask:
       * we have to test all bits 1/0 depending on value of implied bits.
       */
      if (src->implied_bits)
        mask = 0xFFFFFFFF;

      for (; i <= dest->len / BITS_PER_CHUNK; i++)
        dest->v[i] ^= mask;

    }

  dest->implied_bits ^= src->implied_bits;

  try_shrink (dest);

  return 0;

}

extern void
bit_vector_not (struct bit_vector *bv)
/* 
 * performs logical NOT on bit_vector bv.
 */
{
  size_t i;


  for (i = 0; i <= bv->len / BITS_PER_CHUNK; i++)
    bv->v[i] = ~bv->v[i];

  if (bv->implied_bits)
    bv->implied_bits = 0;
  else
    bv->implied_bits = 1;

}
