/*      bit_vector_util.c
 *
 * Copyright (C) 2008 Ivo Alxneit
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
#include <stdlib.h>
#include <string.h>

#include "bit_vector_util.h"
#include "bv.h"

struct bit_vector *
bit_vector_copy (const struct bit_vector *src)
/*
 * return copy of bit_vector 'src' or NULL if allocation of
 * new bit_vector failed.
 */
{
  struct bit_vector *copy;

  const size_t n = (src->len + 1) / BITS_PER_CHUNK;

  (void) get_addr;              /* prevent warnings "defined but not used" */
  (void) try_shrink;
  (void) increase_length;
  /*
   * allocate copy. return NULL upon failure
   */
  copy = (struct bit_vector *) malloc (sizeof (struct bit_vector));
  if (copy == NULL)
    return NULL;

  copy->v = (int *) malloc (n * sizeof (int));
  if (copy->v == NULL)
    {
      free (copy);
      return NULL;
    }

  /*
   * copy 'src'
   */
  memcpy (copy->v, src->v, n * sizeof (int));
  copy->len = src->len;
  copy->implied_bits = src->implied_bits;

  return copy;

}

extern void
bit_vector_print (FILE * stream, const struct bit_vector *bv,
                  const size_t n_bytes, const size_t n_groups)
/*
 * pretty print bit_vector bv
 * groups of 8 bits (1 byte) separated by space
 * groups of 'n_bytes' byte separated by 4 spaces
 * maximum of 'n_groups' groups of n_bytes bytes per line
 */
{
  size_t bit = 0;

  const size_t nb = n_bytes * 8;
  const size_t ng = n_groups * nb;


  while (1)
    {

      if (bit_vector_test_bit (bv, bit))
        fprintf (stream, "1");
      else
        fprintf (stream, "0");

      bit++;
      if (bit > bv->len)
        break;

      if (bit % (ng) == 0)      /* end of line ? */
        fprintf (stream, "\n");

      else
        {                       /* no trailing spaces */

          if (bit % 8 == 0)     /* separator between chars */
            fprintf (stream, " ");

          if (bit % nb == 0)    /* separator between group of bytes */
            fprintf (stream, "   ");

        }

    }

  fprintf (stream, "    %d...\n", bv->implied_bits);

}

extern struct bit_vector *
bit_vector_fread (FILE * stream)
{
  struct bit_vector *bv;

  size_t n_items;
  size_t n_ints;

  bv = (struct bit_vector *) malloc (sizeof (struct bit_vector));
  if (bv == NULL)
    return NULL;

  n_items = fread (&bv->len, sizeof (size_t), 1, stream);

  if (n_items != 1 || bv->len == 0)
    {

      free (bv);
      return NULL;

    }

  n_ints = (bv->len + 1) / BITS_PER_CHUNK;

  bv->v = (int *) malloc (n_ints * sizeof (int));
  if (bv->v == NULL)
    {
      free (bv);
      return NULL;
    }

  n_items = fread (bv->v, sizeof (int), n_ints, stream);

  if (n_items != n_ints)
    {
      bit_vector_free (bv);
      return NULL;
    }

  n_items = fread (&bv->implied_bits, sizeof (int), 1, stream);

  if (n_items != 1 || bv->implied_bits < 0 || bv->implied_bits > 1)
    {
      bit_vector_free (bv);
      return NULL;
    }

  return bv;

}

extern int
bit_vector_fwrite (FILE * stream, const struct bit_vector *bv)
{
  size_t n_items;
  size_t n_ints;

  n_items = fwrite (&bv->len, sizeof (size_t), 1, stream);
  if (n_items != 1)
    return EIO;                 /* I/O error. */
  /* FIXME: return correct error code */

  n_ints = (bv->len + 1) / BITS_PER_CHUNK;

  n_items = fwrite (bv->v, sizeof (int), n_ints, stream);

  if (n_items != n_ints)
    return EIO;                 /* I/O error. */
  /* FIXME: return correct error code */

  n_items = fwrite (&bv->implied_bits, sizeof (int), 1, stream);

  if (n_items != 1)
    return EIO;                 /* I/O error. */
  /* FIXME: return correct error code */

  return 0;

}
