/*      bv.h
 *
 * Copyright (C) 2008 Ivo Alxneit
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#ifndef __BV_H__
#define __BV_H__

#include <errno.h>

#define MIN(X_, Y_)  ((X_) < (Y_) ? (X_) : (Y_))
#define BITS_PER_CHUNK (8 * sizeof(int))
#define ALL_BITS_SET -1

static inline void
get_addr (const size_t bit, size_t * offset, size_t * nr)
/*
 * determines offset 'offset' and position 'nr' that corresponds to bit number
 * 'bit'. first bit is number 0.
 */
{

  *offset = bit / BITS_PER_CHUNK;
  *nr = bit % BITS_PER_CHUNK;

}


static int
increase_length (struct bit_vector *bv, size_t n_bits)
/*
 * increase length of bit_vector to accomodate n_bits bits.
 */
{
  size_t i;
  size_t n_new;
  size_t n_curr;

  int mask = 0;

  void *tmp;


  /*
   * prepare mask:
   * we have to set all bits 1/0 depending on value of implied bits.
   */
  if (bv->implied_bits)
    mask = ALL_BITS_SET;

  /*
   * calulate number of allocated integers in bv before
   * and after increasing its length.
   */
  n_curr = (bv->len + 1) / BITS_PER_CHUNK;
  n_new = n_bits / BITS_PER_CHUNK + 1;

  /* get new storage */
  tmp = realloc (bv->v, n_new * sizeof (int));
  if (tmp == NULL)
    return ENOMEM;

  bv->v = (int *) tmp;          /* use new storage */

  /* clear/set new bits */
  for (i = n_curr; i < n_new; i++)
    bv->v[i] = mask;

  /* update length */
  bv->len = n_new * BITS_PER_CHUNK - 1;

  return 0;

}


static void
shrink_to (struct bit_vector *bv, size_t i)
/*
 * shrink bit_vector to contain i integers
 */
{
  void *tmp;


  tmp = realloc (bv->v, i * sizeof (int));
  bv->v = (int *) tmp;          /* use new storage */
  bv->len = i * BITS_PER_CHUNK - 1;

}


static void
try_shrink (struct bit_vector *bv)
/*
 * shrink (thus allways successful) memory if there are integers
 * allocated (for 'high' integers) that have no/all bits set. keep at least
 * one integer (for bit 0..sizeof(int)-1).
 */
{
  size_t n;
  size_t i;
  size_t stop;

  int mask = 0;


  /* get index of last integer allocated */
  n = bv->len / BITS_PER_CHUNK;
  i = n;
  stop = 0;

  /*
   * prepare mask:
   * we have to test all bits 1/0 depending on value of implied bits.
   */
  if (bv->implied_bits)
    mask = ALL_BITS_SET;

  while ((i >= 1) && (stop == 0))
    {

      /* test is all bits in this int are set/cleared */
      if (bv->v[i] != mask)
        stop = 1;
      else
        i--;

    }

  if (i < n)
    shrink_to (bv, i + 1);

}


#endif /* __BV_H__ */
