/*      test.c
 *
 * Copyright (C) 2008 Ivo Alxneit
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 */

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>

#include "bit_vector.h"
#include "bit_vector_util.h"
#include "bit_vector_logical.h"
#include "bit_vector_shift.h"


static void
list_bits (const size_t n, const size_t b[])
{
  size_t i;

  for (i = 0; i < n - 1; i++)
    fprintf (stdout, "%d, ", b[i]);
  fprintf (stdout, "%d", b[i]);

}



static void
print (const struct bit_vector *bv, const size_t n,
       const size_t b[], const char *desc)
{
  fprintf (stdout, "bits ");
  list_bits (n, b);
  fprintf (stdout, " %s\n", desc);
  bit_vector_print (stdout, bv, 4, 2);
  fprintf (stdout, "\n");

}



static struct bit_vector *
init_test_vector (const size_t n, const size_t b[])
{
  size_t i;
  struct bit_vector *bv = bit_vector_init ();

  for (i = 0; i < n; i++)
    bit_vector_set_bit (bv, b[i]);

  return bv;

}



static void
shift_print (const char *desc, const struct bit_vector *bv,
             const int shift,
             struct bit_vector *(*f) (const struct bit_vector *,
                                      const size_t))
{
/* 
 *  warning: passing argument 2 of 'f' as unsigned due to prototype
 *  warning: passing argument 4 of 'shift_print' from incompatible
 *           pointer type
 *
 *  warnings are because 'bit_vector_[lr]shift()' and 'bit_vector_shift()'
 *  have a slightly different prototype. the former two pass the second
 *  argument as 'const size_t' while the latter passes a 'const int'.
 */

  struct bit_vector *result = (f) (bv, shift);

  fprintf (stdout, "initial vector %s by %d\n", desc, shift);
  bit_vector_print (stdout, result, 4, 2);
  fprintf (stdout, "\n");

  bit_vector_free (result);

}



static void
logical (const char *desc,
         int (*f) (const struct bit_vector *,
                   struct bit_vector *),
         const struct bit_vector *bv0, const struct bit_vector *bv1)
{
  struct bit_vector *result = bit_vector_copy (bv1);

  (f) (bv0, result);

  fprintf (stdout, "vector0 %s vector1:\n", desc);
  bit_vector_print (stdout, result, 4, 2);
  fprintf (stdout, "\n");

  bit_vector_free (result);

}



static void
test_sc (void)
{
  const size_t b[6] = { 0, 1, 63, 64, 247, 433 };
  const size_t n = 6;

  size_t i;

  struct bit_vector *bv;

  bv = bit_vector_init ();
  fprintf (stdout, "initial vector\n");
  bit_vector_print (stdout, bv, 4, 2);
  fprintf (stdout, "\n");

  for (i = 0; i < n; i++)
    bit_vector_set_bit (bv, b[i]);

  print (bv, n, b, "set");

  for (i = 0; i < n; i++)
    bit_vector_clear_bit (bv, b[i]);

  print (bv, n, b, "cleared");

  bit_vector_free (bv);


  bv = bit_vector_init_all_set ();
  fprintf (stdout, "initial vector\n");
  bit_vector_print (stdout, bv, 4, 2);
  fprintf (stdout, "\n");

  for (i = 0; i < n; i++)
    bit_vector_clear_bit (bv, b[i]);

  print (bv, n, b, "cleared");

  for (i = 0; i < n; i++)
    bit_vector_set_bit (bv, b[i]);

  print (bv, n, b, "set");

  bit_vector_free (bv);

}



static void
test_copy (void)
{
  const size_t b[6] = { 0, 1, 63, 64, 247, 433 };
  const size_t n = 6;

  struct bit_vector *bv0, *bv1;

  bv0 = init_test_vector (n, b);
  fprintf (stdout, "initial vector: ");
  print (bv0, n, b, "set");

  bv1 = bit_vector_copy (bv0);
  fprintf (stdout, "copied vector:\n");
  bit_vector_print (stdout, bv1, 4, 2);
  fprintf (stdout, "\n");

  bit_vector_free (bv1);


  bit_vector_not (bv0);
  fprintf (stdout, "initial vector: ");
  print (bv0, n, b, "cleared");

  bv1 = bit_vector_copy (bv0);
  fprintf (stdout, "copied vector:\n");
  bit_vector_print (stdout, bv1, 4, 2);

  bit_vector_free (bv1);
  bit_vector_free (bv0);

}



static void
test_shift (void)
{
  const size_t b[6] = { 0, 1, 63, 64, 247, 433 };
  const size_t n = 6;

  const int shift = 37;

  struct bit_vector *bv;

  bv = init_test_vector (n, b);
  fprintf (stdout, "initial vector: ");
  print (bv, n, b, "set");

  shift_print ("left-shifted", bv, shift, &bit_vector_lshift);
  shift_print ("right-shifted", bv, shift, &bit_vector_rshift);
  shift_print ("shifted by", bv, -shift, &bit_vector_shift);
  shift_print ("shifted by", bv, shift, &bit_vector_shift);


  bit_vector_not (bv);
  fprintf (stdout, "initial vector: ");
  print (bv, n, b, "cleared");

  shift_print ("left-shifted", bv, shift, &bit_vector_lshift);
  shift_print ("right-shifted", bv, shift, &bit_vector_rshift);
  shift_print ("shifted by", bv, -shift, &bit_vector_shift);
  shift_print ("shifted by", bv, shift, &bit_vector_shift);

  bit_vector_free (bv);

}



static void
test_rw (void)
{
  const size_t b[6] = { 0, 1, 63, 64, 247, 433 };
  const size_t n = 6;
  const char fname[] = { "test.dat", };

  struct bit_vector *bv0, *bv1;
  FILE *fp;

  bv0 = init_test_vector (n, b);
  fprintf (stdout, "initial vector: ");
  print (bv0, n, b, "set");

  fp = fopen (fname, "w");
  bit_vector_fwrite (fp, bv0);
  fclose (fp);
  fprintf (stdout, "vector written to file ...");

  fp = fopen (fname, "r");
  bv1 = bit_vector_fread (fp);
  fclose (fp);
  fprintf (stdout, " and read again\n");

  bit_vector_print (stdout, bv1, 4, 2);
  fprintf (stdout, "\n");
  bit_vector_free (bv1);


  bit_vector_not (bv0);
  fprintf (stdout, "initial vector: ");
  print (bv0, n, b, "cleared");

  fp = fopen (fname, "w");
  bit_vector_fwrite (fp, bv0);
  fclose (fp);
  fprintf (stdout, "vector written to file ...");

  fp = fopen (fname, "r");
  bv1 = bit_vector_fread (fp);
  fclose (fp);
  fprintf (stdout, " and read again\n");

  bit_vector_print (stdout, bv1, 4, 2);
  fprintf (stdout, "\n");
  bit_vector_free (bv1);

  bit_vector_free (bv0);

}



static void
test_logical (void)
{
  const size_t b0[6] = { 0, 1, 63, 64, 247, 433 };
  const size_t n0 = 6;
  const size_t b1[9] = { 0, 3, 37, 63, 65, 247, 403, 476, 499 };
  const size_t n1 = 9;

  struct bit_vector *bv0, *bv1;

  bv0 = init_test_vector (n0, b0);
  fprintf (stdout, "initial vector 0: ");
  print (bv0, n0, b0, "set");

  bv1 = init_test_vector (n1, b1);
  fprintf (stdout, "initial vector 1: ");
  print (bv1, n1, b1, "set");

  logical ("and", &bit_vector_and, bv0, bv1);
  logical ("or", &bit_vector_or, bv0, bv1);
  logical ("xor", &bit_vector_xor, bv0, bv1);


  bit_vector_not (bv0);
  fprintf (stdout, "initial vector 0: ");
  print (bv0, n0, b0, "cleared");

  fprintf (stdout, "initial vector 1: ");
  print (bv1, n1, b1, "set");

  logical ("and", &bit_vector_and, bv0, bv1);
  logical ("or", &bit_vector_or, bv0, bv1);
  logical ("xor", &bit_vector_xor, bv0, bv1);


  fprintf (stdout, "initial vector 0: ");
  print (bv0, n0, b0, "cleared");

  bit_vector_not (bv1);
  fprintf (stdout, "initial vector 1: ");
  print (bv1, n1, b1, "cleared");

  logical ("and", &bit_vector_and, bv0, bv1);
  logical ("or", &bit_vector_or, bv0, bv1);
  logical ("xor", &bit_vector_xor, bv0, bv1);


  bit_vector_not (bv0);
  fprintf (stdout, "initial vector 0: ");
  print (bv0, n0, b0, "set");

  fprintf (stdout, "initial vector 1: ");
  print (bv1, n1, b1, "cleared");

  logical ("and", &bit_vector_and, bv0, bv1);
  logical ("or", &bit_vector_or, bv0, bv1);
  logical ("xor", &bit_vector_xor, bv0, bv1);


  bit_vector_free (bv0);
  bit_vector_free (bv1);

}



int
main (void)
{

  printf ("test set/clear:\n\n");
  test_sc ();
  printf ("\n\n");

  printf ("test copy:\n\n");
  test_copy ();
  printf ("\n\n");

  printf ("test shift:\n\n");
  test_shift ();
  printf ("\n\n");

  printf ("test read/write:\n\n");
  test_rw ();
  printf ("\n\n");

  printf ("test logical:\n\n");
  test_logical ();
  printf ("\n\n");

  return EXIT_SUCCESS;

}
