#ifndef GSL_MARRAY_IDX_H_INCLUDED
#define GSL_MARRAY_IDX_H_INCLUDED


/** This component supports the notion of a general affine
 *  indexing map used to compute offsets in a linear data
 *  structure, from a set of indices. The number of such
 *  indices is the "rank" of the indexing map.
 *
 *  The offset is computed from
 *    offset = sum(n, index[n] * stride[n]),
 *  where n is in [0,rank).
 *
 *  This is essentially trivial, but handles all the cases which
 *  correspond to "indexed, strided" data storage and therefore
 *  provides a general methodology.
 *
 *  Clients need not concern themselves with the details of the
 *  indexing maps, if they will only be using the higher-level
 *  multi-arrays, which manage both indexing and data.
 */


/** Describe one component of the affine indexing map used to
 *  compute an offset from a set of indices. A set of such
 *  components, of size equal to the rank of the indexing
 *  map, specifies the map completely.
 *
 *  The full indexing map for general rank itself is _not_ introduced
 *  as an explicit type, because this would raise annoying questions
 *  about the allocation/initialization of such maps. By avoiding an
 *  explicit type, we insulate ourselves from that question, and the
 *  low-level details of the library are made more flexible. It is
 *  convenient that an array of these serves our general purpose.
 */
typedef struct
{
  int  dimens; // dimension in this index place
  int  stride; // addressing stride for this index
}
gsl_marray_idx;


/** Calculate the offset for a given index vector,
 *  using a given affine indexing map.
 *
 *  Uses the most general (and therefore non-optimized) method.
 *  Clients of explicit-rank multi-arrays should use the
 *  specialized indexing for their type, rather than this
 *  general method.
 */
int
gsl_marray_idx_offset(const gsl_marray_idx p[], const int index[], int rank);


/** Compute an indexing map for the generic packed
 *  case, where the addressing is entirely contiguous,
 *  so that the map is specified completely by the
 *  set of dimensions. Returns an error code.
 */
int
gsl_marray_idx_compute_packed(gsl_marray_idx p[], const int dims[], int rank);


/** Compress an indexing map to a lower rank by finding
 *  and eliding any indices with dimension = 1. Returns
 *  the new rank, which is always less than or equal to
 *  the old rank.
 *  FIXME: this could be done in place... right ??
 */
int
gsl_marray_idx_compress(gsl_marray_idx pnew[], const gsl_marray_idx pold[], int rank_old);


/** Calculate a slicing index map which freezes one of
 *  the indices at a given value. This always reduces the
 *  rank of the map by one. Places the new indexing map
 *  in pnew[0]. Returns the offset of the new origin.
 */
int
gsl_marray_idx_slice_freeze(
  gsl_marray_idx       pnew[],
  const gsl_marray_idx pold[],
  int    rank_old,
  int    freeze_place,
  int    freeze_value
  );


/** Calculate a slicing index map for a rank one slice, which is
 *  a slice that represents a rank one index map, no matter the
 *  rank of the initial indexing map.
 *
 *  Returns the offset of the new origin. Places the new
 *  indexing map in pnew[0].
 *
 *  The main issue here is describing a general rank one slice.
 *  A rank one slice is completely specified by two integer
 *  vectors of size equal to the rank of the old scheme.
 *  The map defines the "old" index values in terms of
 *  the "new" index values:
 *
 *    i_old[n] = begin[n] + i_new * delta[n]
 *
 *    where i_old[n] in [0, dim_old[n]).
 */
int
gsl_marray_idx_slice_1(
  gsl_marray_idx       pnew[],
  const gsl_marray_idx pold[],
  int rank_old,
  const int begin[],
  const int delta[]
  );



/** A slice of a single index is a range of index values and a
 *  stride. The index values are inclusive at both ends; this
 *  is analogous to the fortran notion of slicing, and it
 *  turned out to be the most convenient way to do it.
 *
 *  The span of the slice is abs(index_hi - index_lo),
 *  and in normal usage the span should be divisible
 *  by the stride. Computations will work if this is
 *  violated but will probably be harder to understand.
 */
typedef struct
{
  int  index_hi;
  int  index_lo;  // FIXME: can  hi be less than lo ??? (sign of stride)
  int  stride;
}
gsl_marray_idx_slice;


/** Calculate a slicing index map which restricts
 *  indices to given ranges, with given strides.
 *  Returns the offset of the new origin.
 *
 *  The new map may be compressible, since some ranges may have
 *  dimension = 1. But it is not explicitly compressed by this
 *  function; so the dimension = 1 indices must still be addressed
 *  when using the new scheme; set their index value to 0.
 */
int
gsl_marray_idx_slice_restrict(
  gsl_marray_idx       pnew[],
  const gsl_marray_idx pold[],
  int rank_old,
  const gsl_marray_idx_slice sl[]
  );


/** Expose the full complexity of the affine indexing
 *  scheme by allowing the most general re-indexing.
 *  This is specified by a vector of length = rank,
 *  the shift values, and a (rank, rank) matrix of
 *  transformation. The map is defined by
 *
 *    i_old[n] = begin[n] + sum_m(delta[n][m] * i_new[m]).
 *
 *  Returns the offset of the new origin. Does not compress
 *  the new map, so the dimension = 1 indices must still be
 *  addressed, with index value 0.
 *
 *  Finally, the caller must specify the dimensions
 *  for the new indices.
 *  FIXME: How to calculate these??? I don't think you can.
 *
 *  The vector of dimensions corresponds to an extremal
 *  vector in the convex set of indices. We can calculate
 *  such an extremal vector (maybe delta has to be invertible?),
 *  but its meaning is not clear, and the user would have some
 *  specific use in mind anyway. I think the extremal vector is
 *  unique, since we're just skewing the cube, so at least it
 *  is well-defined. But perhaps not useful.
 */
int
gsl_marray_idx_slice_general(
  gsl_marray_idx       pnew[],
  const gsl_marray_idx pold[],
  int rank_old,
  const long begin[],
  const long ** delta,
  const int dims[]
  );



/** General operations on indexed data require some
 *  iteration methodology. When the rank is not a
 *  compile-time constant, it is not possible to
 *  write hierarchical for loops.
 *
 *  Like the rest of the indexing map stuff,
 *  normal clients do not really need to know
 *  about this. Also, it is not
 *  as efficient as the methods for statically
 *  known ranks. It is used to implement any
 *  less critical generic indexed operations.
 */

int
gsl_marray_idx_iter_begin(int index[], int rank);

int
gsl_marray_idx_iter_end(const gsl_marray_idx p[]);

int
gsl_marray_idx_iter_next(int offset, int index[], int rank, const gsl_marray_idx p[]);


#endif  /* !GSL_MARRAY_IDX_H_INCLUDED*/
