/*
* Copyright (c) 2012 Nathanael Hübbe
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions
* are met:
*
* 1. Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
* 2. Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the distribution.
*
* THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
* ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
* FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
* HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
* LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
* OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
* SUCH DAMAGE.
*/

#ifndef INCLUDE_GUARD_H5_EXTERNAL_MODULE_LOADER_H
#define INCLUDE_GUARD_H5_EXTERNAL_MODULE_LOADER_H

#include "H5api_adpt.h"

/*******************************************************************************
* This code searches for external modules and loads them.
* It should be called (in)directly by H5_init_library().
*******************************************************************************/

/*******************************************************************************
* External module definition:
*
* A module is a shared object file (dynamical library, but without the naming convention).
* Each module must implement the following functions:
*
* const char* H5_XM_get_module_name(void);
*     This should simply return the adress of a string constant.
*
* herr_t H5_XM_get_module_version(int* majorVersion, int* minorVersion, int* releaseNumber);
*
* herr_t H5_XM_init(void** privateData);
*     This should do any initialization necessary, for example to register a filter.
*     H5_XM_get_module_name() and/or H5_XM_get_module_version() might be called first.
*     privateData is only supplied for convenience, the external module can store anything it wants in *privateData.
*
* herr_t H5_XM_deinit(void* privateData);
*     This should do any tidy up necessary, releasing all resources.
*     There is no guarantee, that this function is actually called!
*     But if it is called, it is only called once for each H5_XM_init() call with the corresponding privateData.
*
* XXX: Should a dependency mechanism be added? Currently, this does not seem prudent, since the symbols exported by the modules are not made available, the only sensible thing a module can do is to register some callbacks. And those should not be dependent on each other.
*******************************************************************************/

/*******************************************************************************
* External module search paths:
*
* HDF5 will look for modules and load them in this order:
* /etc/h5modules/
* /usr/etc/h5modules/
* /usr/local/etc/h5modules/
* ~/etc/h5modules/
*     XXX: Is this wise? Could introduce security problems. This way it provides maximum flexibility.
* $H5_EXTERNAL_MODULES_PATH
*     This is a whitespace separated list of paths to search.
*     XXX: Is this wise? Could introduce security problems. This way it provides maximum flexibility.
*******************************************************************************/

//API methods:
H5_DLL herr_t H5get_loaded_external_modules(const char*** names, size_t* count);
H5_DLL herr_t H5get_external_module_version(const char* name, int* majorVersion, int* minorVersion, int* releaseNumber);

//Internal methods:
H5_DLL herr_t H5_load_external_modules(void);
H5_DLL herr_t H5_unload_external_modules(void);

#endif //INCLUDE_GUARD_H5_EXTERNAL_MODULE_LOADER_H
