/* vi:set ts=8 sw=8 noet: */

/*****************************************************************************/
/* Lexer options                                                             */

%option stack

/*****************************************************************************/
/* Program definitions                                                       */

%{
#include "parser_scanner.h"

#if defined(LEXER_DEBUG)
#define DPRINT(msg)	printf msg
#else
#define DPRINT(msg)
#endif

%}

	/*********************************************************************/
	/* Lexer definitions                                                 */

	/* macros */
DIGIT	[0-9]
IDENT	[_a-z][_a-z{DIGIT}]+
WS	[ \t]+

	/* parsing states */
%start	KIDSTATE
%start	OPTSTATE

%%
	/*********************************************************************/
	/* Lexer rules                                                       */

	/* ----------------------------------------------------------------- */
	/* general/widget keywords can only occur in initial state           */

	/* general keywords, start options parsing */
<INITIAL>dialog		{ BEGIN(OPTSTATE); return DIALOG; }

	/* widgets without id, start options parsing */
<INITIAL>action		{ BEGIN(OPTSTATE); return ACTION; }

	/* widgets with id, the colon will bring id 'widget id' state */
<INITIAL>input		{ return INPUT; }

	/* the colon marks start of a keyword identifier */
<INITIAL>:/{IDENT}	{ BEGIN(KIDSTATE); return yytext[0]; }

	/* ----------------------------------------------------------------- */
	/* get an identifier, then bring the options parsing state           */
<KIDSTATE>{IDENT}	{
			yylval.str = yytext;
			BEGIN(OPTSTATE);
			return WIDGETID;
			}

	/* ----------------------------------------------------------------- */
	/* options                                                           */

	/* options keywords */
<OPTSTATE>close		{ return ACTION_CLOSE; }
<OPTSTATE>okcancel	{ return ACTION_OKCANCEL; }
<OPTSTATE>lines		{ return OPTION_LINES; }
<OPTSTATE>ro		{ return OPTION_READONLY; }

	/* option identifier: id followed by a colon */
<OPTSTATE>{IDENT}/:	{
			yylval.str = yytext;
			DPRINT(("LEXER: got identifier '%s'\n", yylval.str));
			return ID;
			}

	/* option separators */
<OPTSTATE>:		{ return yytext[0]; }

	/* ----------------------------------------------------------------- */
	/* strings and numbers can only occur during options parsing         */

	/* number */
<OPTSTATE>{DIGIT}+	{
			yylval.num = atoi(yytext);
			return NUMBER;
			}

	/* double-quoted string: remove double-quotes */
<OPTSTATE>\"[^"]*\"	{
			yylval.str = yytext + 1;
			yylval.str[strlen(yytext) - 2] = 0;
			return QSTRING;
			}

	/* quoted string: remove quotes */
<OPTSTATE>'[^']*'	{
			yylval.str = yytext + 1;
			yylval.str[strlen(yytext) - 2] = 0;
			return QSTRING;
			}

	/* simple string */
<OPTSTATE>[^:,'" \t\n]+	{
			yylval.str = yytext;
			return STRING;
			}

	/* ----------------------------------------------------------------- */
	/* default rules                                                     */

	/* ignore whitespaces, comments and whitespace only lines */
{WS}			;
#.*$			;
^{WS}*\n		;

	/* end of script */
<<EOF>>			{ return 0; }
<INITIAL>end		{ return 0; }

	/* newline, back to initial state */
{WS}*\n			{ BEGIN(INITIAL); return EOL; }

%%

/*****************************************************************************/
/* Program routines                                                          */

