/* vi:set ts=8 sw=8 noet: */

/*****************************************************************************/
/* Scanner options                                                           */

%error-verbose
/*%locations*/

/*****************************************************************************/
/* Program definitions                                                       */

%{

#include <stdlib.h>
#include <stdio.h>

#if defined(SCANNER_DEBUG)
#define DPRINT(msg)	printf msg
#else
#define DPRINT(msg)
#endif

%}

/*****************************************************************************/
/* Scanner definitions                                                       */

/* start production */
%start	script

/* available types */
%union {
	char *str;
	int num;
}

/* typed non-terminals and terminals */
%type	<str>	string
%token	<str>	STRING QSTRING ID WIDGETID
%token	<num>	NUMBER

/* general tokens */
%token	EOL

/* general keywords */
%token	DIALOG

/* widget keywords without id */
%token	ACTION

/* widget keywords with id */
%token	INPUT

/* widget specific options */
%token	ACTION_CLOSE ACTION_OKCANCEL

/* generic options */
%token	OPTION_HELP OPTION_LINES OPTION_READONLY

%%

/*****************************************************************************/
/* Scanner rules                                                             */

/* script is made of individual lines */
script:
	lines
	;

/* one to several lines */
lines:
	line_content
	| line_content lines
	;

/* all lines are predefined, and stop with an 'end-of-line' marker */
line_content:
	dialog_line EOL
	| action_line EOL
	| input_line EOL
	| EOL
	;

/* dialog keyword ---------------------------------------------------------- */
dialog_line:
	DIALOG
	{ DPRINT(("* set dialog properties\n")); }
	dialog_size dialog_title dialog_descr
	;

/* size is optional, and made of 2 numbers */
dialog_size:
	/* empty */
	| NUMBER NUMBER
	{ DPRINT(("  - size: %dx%d\n", $1, $2)); }
	;

/* the title is a simple string and is mandatory */
dialog_title:
	string
	{ DPRINT(("  - title: %s\n", $1)); }
	;

/* the description is a simple string and is optional */
dialog_descr:
	/* empty */
	| string
	{ DPRINT(("  - description: %s\n", $1)); }
	;

	
/* action keyword ---------------------------------------------------------- */
action_line:
	ACTION
	{ DPRINT(("* add an action line\n")); }
	action_type action_help_button action_add_button
	;

/* mandatory set of buttons */
action_type:
	ACTION_CLOSE
	{ DPRINT(("  - close button\n")); }
	| ACTION_OKCANCEL
	{ DPRINT(("  - ok and cancel buttons\n")); }
	;

/* optional help button */
action_help_button:
	/* empty */
	| OPTION_HELP
	{ DPRINT(("  - help button\n")); }
	;

/* optional additional button: <id>:<label> */
action_add_button:
	/* empty */
	| ID
	{ DPRINT(("new button id: %s\n", $1)); }
	':' string
	{ DPRINT(("  - additional button #%s '%s'\n", $1, $4));
	  DPRINT(("id separator: %s\n", $<str>3)); }
	;

/* input keyword ----------------------------------------------------------- */
input_line:
	INPUT ':' WIDGETID
	{ DPRINT(("* add an input field #%s\n", $3)); }
	input_ro_option input_lines_option string string
	{ DPRINT(("  - label: %s\n  - initial text: %s\n", $7, $8)); }
	;

/* optional visible lines: the 'lines' keyword followed by a number */
input_lines_option:
	/* empty */
	| OPTION_LINES NUMBER
	{ DPRINT(("  - visible lines: %d\n", $2)); }
	;

/* optional read-only option */
input_ro_option:
	/* empty */
	| OPTION_READONLY
	{ DPRINT(("  - read-only\n")); }
	;

/* strings ----------------------------------------------------------------- */
string:
	STRING
	| QSTRING
	;

%%

/*****************************************************************************/
/* Program routines                                                          */

int yyerror(char *s)
{
	printf("dialog script: %s\n", s);
	exit(0);
	return 0;
}

