/***************************************************************************
 *   Copyright (C) 2009 by Kevin Channon                                   *
 *   kevinchannon@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "gslmm.h"

namespace gsl{

vector::vector()
{
	is_initialised = false;
}

vector::vector(int length)
{
	VECTOR = gsl_vector_alloc(length);
	is_initialised = true;
}

vector::vector(int length, double defaultElementValue)
{
	VECTOR = gsl_vector_alloc(length);
	is_initialised = true;
	gsl_vector_set_all(VECTOR, defaultElementValue);
}

vector::~vector()
{
//	if(is_initialised)
//		gsl_vector_free(VECTOR);
}

const vector &vector::operator=(const vector &right) throw (out_of_range)
{
	gsl_vector_memcpy(VECTOR, right.VECTOR);
}

const vector &vector::operator+=(const vector &right) throw (out_of_range)
{
	if(VECTOR->size != right.size())
		throw out_of_range();
	
	gsl_vector_add(VECTOR, right.VECTOR);
}

const vector &vector::operator+=(const double &right)
{
	gsl_vector_add_constant(VECTOR, right);
}

const vector &vector::operator-=(const vector &right) throw (out_of_range)
{
	if(VECTOR->size != right.size())
		throw out_of_range();
	
	gsl_vector_sub(VECTOR, right.VECTOR);
}

const vector &vector::operator-=(const double &right)
{
	gsl_vector_add_constant(VECTOR, -right);
}

/*
const vector &vector::operator*=(const matrix &right) throw (out_of_range)
{
	
}

const vector &vector::operator*=(const vector &right) throw (out_of_range)
{
}

const vector &vector::operator*=(const double &right)
{
}

const vector &vector::operator/=(const double &right)
{
}
*/

gsl_vector *vector::ptr()
{
	return VECTOR;
}
		
double vector::at(int element) const
{
	return gsl_vector_get(VECTOR, element);
}

double &vector::at(int element)
{
	if(element < VECTOR->size)
		return VECTOR->data[element];
}

void vector::zero()
{
	for(int i = 0; i < VECTOR->size; i++)
		VECTOR->data[i] = 0;
}

double vector::max() const
{
	return gsl_vector_max(VECTOR);
}

double vector::min() const
{
	return gsl_vector_min(VECTOR);
}

int vector::max_element() const
{
	return gsl_vector_max_index(VECTOR);
}

int vector::min_element() const
{
	return gsl_vector_min_index(VECTOR);
}

void vector::swap(int element1, int element2)
{
	gsl_vector_swap_elements(VECTOR, element1, element2);
}

void vector::reverse()
{
	gsl_vector_reverse(VECTOR);
}
		
int vector::size() const
{
	return VECTOR->size;
}

void vector::resize(const int newSize)
{
	if(is_initialised){
		if(newSize != VECTOR->size){
			double *temp = new double[newSize];
			for(int i = 0; i < newSize; i++)
				temp[i] = VECTOR->data[i];
			gsl_vector_free(VECTOR);
			VECTOR = gsl_vector_alloc(newSize);
			for(int i = 0; i < newSize; i++)
				VECTOR->data[i] = temp[i];
			delete[] temp;
		}
	}
	else{
		VECTOR = gsl_vector_alloc(newSize);
		is_initialised = true;
	}	
}

void vector::free()
{
	if(is_initialised){
		gsl_vector_free(VECTOR);
		is_initialised = false;
	}
}


matrix::matrix()
{
	is_initialised = false;
}

matrix::matrix(int rows, int columns)
{
	MATRIX = gsl_matrix_alloc(rows, columns);
	is_initialised = true;
}

matrix::matrix(int rows, int columns, double defaultElementValue)
{
	MATRIX = gsl_matrix_alloc(rows, columns);
	gsl_matrix_set_all(MATRIX, defaultElementValue);
	is_initialised = true;
}

matrix::~matrix()
{
}

const matrix &matrix::operator=(const matrix &right)
{
	gsl_matrix_memcpy(MATRIX, right.MATRIX);
}

const matrix &matrix::operator+=(const double &right)
{
	gsl_matrix_add_constant(MATRIX, right);
}

const matrix &matrix::operator+=(const matrix &right)
{
	gsl_matrix_add(MATRIX, right.MATRIX);
}

const matrix &matrix::operator-=(const double &right)
{
	gsl_matrix_add_constant(MATRIX, -right);
}

const matrix &matrix::operator-=(const matrix &right)
{
	gsl_matrix_sub(MATRIX, right.MATRIX);
}

gsl_matrix *matrix::ptr()
{
	return MATRIX;
}

std::allocator<double>::const_reference matrix::at(Point element) const
{
	return MATRIX->data[(int)element.x + (int)element.y*MATRIX->size2];
}

std::allocator<double>::reference matrix::at(unsigned n, unsigned m)
{
	return MATRIX->data[m + n*MATRIX->size2];
}

std::allocator<double>::const_reference matrix::at(unsigned n, unsigned m) const
{
	return MATRIX->data[m + n*MATRIX->size2];
}

std::allocator<double>::reference matrix::at(Point element)
{
	return MATRIX->data[(int)element.x + (int)element.y*MATRIX->size2];
}

double matrix::max() const
{
	double max = gsl_matrix_max(MATRIX);
	return gsl_matrix_max(MATRIX);
}

double matrix::min() const
{
	return gsl_matrix_min(MATRIX);
}

int matrix::rows() const
{
	return MATRIX->size1;
}

int matrix::cols() const
{
	return MATRIX->size2;
}

void matrix::resize(const int newRows, const int newCols)
{
	if(is_initialised){
		if(newRows != MATRIX->size1 || newCols != MATRIX->size2){
			double *temp = new double[newRows*newCols];
			for(int i = 0; i < newRows; i++){
				for(int j = 0; j < newCols; j++){
					temp[newCols*i + j] = gsl_matrix_get(MATRIX, i, j);
				}
			}
			gsl_matrix_free(MATRIX);
			MATRIX = gsl_matrix_alloc(newRows, newCols);
			for(int i = 0; i < newRows; i++){
				for(int j = 0; j < newCols; j++){
					gsl_matrix_set(MATRIX, i, j, temp[newCols*i + j]);
				}
			}
			delete[] temp;
		}
	}
	else{
		MATRIX = gsl_matrix_alloc(newRows, newCols);
		is_initialised = true;
	}
}

bool matrix::is_null() const
{
	return (bool)gsl_matrix_isnull(MATRIX);
}

void matrix::free()
{
	if(is_initialised){
		gsl_matrix_free(MATRIX);
		is_initialised = false;
	}
}

histogram::histogram()
{
	is_initialised = false;
}

histogram::histogram(int bins)
{
	HISTOGRAM = gsl_histogram_alloc(bins);
	is_initialised = true;
}

histogram::~histogram(){}

gsl_histogram *histogram::ptr()
{
	return HISTOGRAM;
}

void histogram::ranges(std::vector< double > inputRange)
{
	double *range = new double[inputRange.size()];
	for(int i = 0; i < inputRange.size(); i++)
		range[i] = inputRange[i];
	gsl_histogram_set_ranges(HISTOGRAM, range, inputRange.size() + 1);
	delete[] range;
}

std::vector< double > histogram::range(int bin) const
{
	double lower, upper;
	std::vector< double > range(2);
	gsl_histogram_get_range(HISTOGRAM, bin, &lower, &upper);
	range[0] = lower;
	range[1] = upper;
	return range;
}

void histogram::uniformRange(double min, double max)
{
	gsl_histogram_set_ranges_uniform(HISTOGRAM, min, max);
}

void histogram::input(double value)
{
	gsl_histogram_increment(HISTOGRAM, value);
}

void histogram::input(double value, double weight)
{
	gsl_histogram_accumulate(HISTOGRAM, value, weight);
}

double histogram::get(int bin) const
{
	return gsl_histogram_get(HISTOGRAM, bin);
}

double histogram::min() const
{
	return gsl_histogram_min(HISTOGRAM);
}

double histogram::max() const
{
	return gsl_histogram_min(HISTOGRAM);
}

int histogram::bins() const
{
	return gsl_histogram_bins(HISTOGRAM);
}

void histogram::reset()
{
	gsl_histogram_reset(HISTOGRAM);
}

int histogram::find(double value) const
{
	size_t bin = 0;
	gsl_histogram_find(HISTOGRAM, value, &bin);
	return (int)bin;
}

double histogram::maxVal() const
{
	return gsl_histogram_max_val(HISTOGRAM);
}

int histogram::maxBin() const
{
	return gsl_histogram_max_bin(HISTOGRAM);
}

double histogram::minVal() const
{
	return gsl_histogram_min_val(HISTOGRAM);
}

int histogram::minBin() const
{
	return gsl_histogram_min_bin(HISTOGRAM);
}

int histogram::mode() const
{
	return gsl_histogram_max_bin(HISTOGRAM);
}

double histogram::mean() const
{
	return gsl_histogram_mean(HISTOGRAM);
}

double histogram::variance() const
{
	return gsl_pow_2(gsl_histogram_sigma(HISTOGRAM));
}

double histogram::stdDev() const
{
	return gsl_histogram_sigma(HISTOGRAM);
}

double histogram::sum() const
{
	return gsl_histogram_sum(HISTOGRAM);
}

void histogram::saveAs(std::string fileName) const
{
	std::ofstream outFile(fileName.c_str(), std::ios::out);
	std::vector< double > limits(2);
	for(int i = 0; i < bins(); i++){
		limits = range(i);
		outFile << limits[0] << " 0" << std::endl;
		outFile << limits[0] << " " << get(i) << std::endl;
		outFile << limits[1] << " " << get(i) << std::endl;
		outFile << limits[1] << " 0" << std::endl;
	}
}

void histogram::free()
{
	if(is_initialised){
		gsl_histogram_free(HISTOGRAM);
		is_initialised = false;
	}
}

}
