/***************************************************************************
 *   Copyright (C) 2009 by Kevin Channon                                   *
 *   kevinchannon@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef GSLMM_H
#define GSLMM_H

#include <iostream>
#include <vector>
#include <fstream>
#include <string>
#include <stdexcept>

#include <gsl/gsl_vector.h>
#include <gsl/gsl_matrix.h>
#include <gsl/gsl_histogram.h>
#include <gsl/gsl_math.h>

#ifndef GSL_RANGE_CHECK_OFF
#define GSL_RANGE_CHECK_OFF
#endif


namespace gsl{
	
	class out_of_range : public std::runtime_error{
		public:
			out_of_range() : std::runtime_error("Unable to open video file"){}
	};
	
	class vector{
		public :
			vector();
			vector(int length);
			vector(int length, double defaultElementValue);
			~vector();
			
			const vector &operator=(const vector &right) throw (out_of_range);
			const vector &operator+=(const vector &right) throw (out_of_range);
			const vector &operator+=(const double &right);
			const vector &operator-=(const vector &right) throw (out_of_range);
			const vector &operator-=(const double &right);
			
			gsl_vector *ptr();
			
			double at(int element) const;
			double &at(int element);
			void zero();
			double max() const;
			double min() const;
			int max_element() const;
			int min_element() const;
			void swap(int element1, int element2);
			void reverse();
			
			int size() const;
			void resize(const int newSize);
			
			void free();
			
		private :
			gsl_vector *VECTOR;
			bool is_initialised;
	};
	
	class matrix{
		public :
			matrix();
			matrix(int rows, int columns);
			matrix(int rows, int columns, double defaultElementValue);
			~matrix();
			
			const matrix &operator=(const matrix &right);
			const matrix &operator+=(const double &right);
			const matrix &operator+=(const matrix &right);
			const matrix &operator-=(const double &right);
			const matrix &operator-=(const matrix &right);
			
			gsl_matrix *ptr();
			
			std::allocator<double>::reference at(Point element);
			std::allocator<double>::const_reference at(Point element) const;
			std::allocator<double>::reference at(unsigned n, unsigned m);
			std::allocator<double>::const_reference at(unsigned n, unsigned m) const;
			
			double max() const;
			double min() const;
			
			int rows() const;
			int cols() const;
			void resize(const int newRows, const int newCols);
			
			bool is_null() const;
			
			void free();
			
		private :
			gsl_matrix *MATRIX;
			bool is_initialised;
	};
	
	class histogram{
		public:
			histogram();
			histogram(int bins);
			~histogram();
			
			gsl_histogram *ptr();
			
			void ranges(std::vector< double > inputRange);
			std::vector< double > range(int bin) const;
			void uniformRange(double min, double max);
			void input(double value);
			void input(double value, double weight);
			double get(int bin) const;
			double min() const;
			double max() const;
			int bins() const;
			void reset();
			
			int find(double value) const;
			
			double maxVal() const;
			int maxBin() const;
			double minVal() const;
			int minBin() const;
			int mode() const;	/* This does the same thing as maxBin, just has a more appropriate name! */
			double mean() const;
			double variance() const;
			double stdDev() const;
			double sum() const;
			
			void saveAs(std::string fileName) const;
			
			void free();
			
		private:
			gsl_histogram *HISTOGRAM;
			bool is_initialised;
	}; 

}

#endif	/* GSLMM_H */
