#include <math.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <gsl/gsl_math.h>
#include <gsl/gsl_odeiv2.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_matrix.h>

// ===================================================================

int ode_system_rhs_function( double t, const double y[], double f[], void *par )
{
  // r.h.s. of the ODE system
  f[0] = 2.0 - 3.0 * y[0];
  f[1] = 2.0 * y[0] - 2.0 * y[0] * y[1] - y[1];
  // test
  fprintf(stdout,"ode_system_rhs_function\tt=%.20e\n",t);fflush(stdout);
  // return value
  return GSL_SUCCESS;
}

// -----------------------------------------------------------------------

int ode_system_rhs_jacobian( double t, const double y[], double *dfdy, double dfdt[], void *par )
{
  // local variables
  gsl_matrix_view dfdy_mat = gsl_matrix_view_array (dfdy, 2, 2);
  gsl_matrix * m = &dfdy_mat.matrix; 
  // Jacobian
  gsl_matrix_set( m, 0, 0, -3.0 );
  gsl_matrix_set( m, 0, 1,  0.0 );
  gsl_matrix_set( m, 1, 0,  2.0 - 2.0 * y[1] );
  gsl_matrix_set( m, 1, 1, -1.0 - 2.0 * y[0] );
  // no explicit time dependence
  dfdt[0] = 0.0;
  dfdt[1] = 0.0;
  // return value
  return GSL_SUCCESS;
}

// =========================================================

int main()
{
  // --------------- definitions ----------------------
  // the variables used for solving the ODE system
  double y[2]; y[0]=1.0; y[1]=1.0; // the 3 independent variables of the ODE system
  double t = 0.0; // current time, start time
  double tEnd = 128.0; // end time
  
  // ------------- solving the ODE system ---------------
  // define the ODE system
  // ode_system_rhs_function: the right hand side (rhs) of the ODE system
  gsl_odeiv2_system sys = {ode_system_rhs_function, ode_system_rhs_jacobian, 2, NULL};
  // currently suggested simulation time step size for the next simulation step
  // (the driver will change this value, if necessary)
  double h = 0.125;
  // the driver solving the ODE system
  gsl_odeiv2_driver * driver = gsl_odeiv2_driver_alloc_y_new ( &sys, gsl_odeiv2_step_rk2imp, h, 1e-6, 1e-6 );
  // status of the driver
  int status = 0;
  // in one update the driver should evolve the ODE system over this amount time
  double evolve_interval_size = 0.125;
  // write data
  fprintf(stdout,"main\tt %e\ty[0] %g\ty[1] %g\tstatus %d\n",t,y[0],y[1],status);fflush(stdout);
  // solve the ODE system
  do{
      // try to move ahead
      status = gsl_odeiv2_driver_apply( driver, &t, t + evolve_interval_size, y );
      // write data
      fprintf(stdout,"main\tt %e\ty[0] %g\ty[1] %g\tstatus %d\n",t,y[0],y[1],status);fflush(stdout);
  // do we want to stay in the loop?
  }while( GSL_SUCCESS == status && tEnd > t );

  // ------------- closing ---------------------
  // free locally used memory
  gsl_odeiv2_driver_free( driver );
  // done
  return 1;
}
