#encoding: utf-8
#cython: boundscheck=False
#cython: wraparound=False
#cython: cdivision=True
#cython: profile=False
#cython: annotate=True

from cython_gsl cimport *
from cython.operator cimport dereference as deref
from libc.math cimport sin,cos,tanh,sqrt
from libc.math cimport pow as cpow
from libc.stdio cimport printf

import mpmath
import cmath as cm
from scipy import integrate
import numpy as np
cimport numpy as np

#Define a numpy compile-time type for 2x2 arrays
ctypedef np.complex128_t DTYPE_t
ctypedef np.double_t DTYPE_t_r

#Define a structure for the func and jac - used by gsl_odeiv2
ctypedef struct my_params:
	double u
	gsl_complex d_up
	gsl_complex d_down
	gsl_complex D_up
	gsl_complex E_up
	gsl_complex t_up
	gsl_complex D_down
	gsl_complex E_down
	gsl_complex t_down
	gsl_matrix_complex * P
	gsl_complex detP

#In order to define all the functions with global constants appended we will use a Python/Cython class

cdef class SI_CI:
	cdef public double T_up, T_down, theta, alpha, zeta, Ez, eV
	cdef readonly double l_sf, kT, reg_c, R_up, R_down
	cdef double eta_1plus, eta_1minus, eta_2
	
	def __init__(self,double l_sf,double kT,double reg_c,double alpha,double zeta,double T_up,double T_down,double theta,double Ez,double eV):
		self.T_up=T_up
		self.T_down=T_down
		self.R_up=1-T_up
		self.R_down=1-T_down
		self.theta,self.alpha,self.zeta=theta,alpha,zeta
		self.kT,self.l_sf,self.reg_c,self.Ez,self.eV=kT,l_sf,reg_c,Ez,eV
		#Parameters used to define the \Pi-matrix (see below)
		self.eta_1plus=self.R_up+self.R_down+2*sqrt(self.R_up*self.R_down)*cos(theta)
		self.eta_1minus=self.R_up+self.R_down-2*sqrt(self.R_up*self.R_down)*cos(theta)
		self.eta_2=2*sqrt(self.R_up*self.R_down)*sin(theta)
	
	#Void function, which updates the dependent attributes in case we change one of the independent parameters
	def update_attrs(self):
		self.R_up=1-self.T_up
		self.R_down=1-self.T_down
		#Update the eta parameters
		self.eta_1plus=self.R_up+self.R_down+2*sqrt(self.R_up*self.R_down)*cos(self.theta)
		self.eta_1minus=self.R_up+self.R_down-2*sqrt(self.R_up*self.R_down)*cos(self.theta)
		self.eta_2=2*sqrt(self.R_up*self.R_down)*sin(self.theta)
		cdef const char* SUCCESS="Dependent attributes have been updated!"
		printf ("%s \n", SUCCESS)
	
#####################
## Basic functions ##
#####################

#Delta: 1.-1./(2*Sys.l_sf*cm.sqrt(1-x_bar*x_bar))
cdef gsl_complex _Delta(SI_CI Sys,gsl_complex x_bar) nogil:
	return gsl_complex_add_real(gsl_complex_negative(gsl_complex_inverse(gsl_complex_mul_real(gsl_complex_sqrt(gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(x_bar,2)),1.0)),2*Sys.l_sf))),1.0)

#Sigma: x_bar+x_bar/(2*Sys.l_sf*cm.sqrt(1-x_bar*x_bar))
cdef gsl_complex _Sigma(SI_CI Sys,gsl_complex x_bar) nogil:
	return gsl_complex_mul(gsl_complex_add_real(gsl_complex_inverse(gsl_complex_mul_real(gsl_complex_sqrt(gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(x_bar,2)),1.0)),2*Sys.l_sf)),1.0),x_bar)

#gamma: -D/(E+1.j*cm.sqrt(D**2-E**2))
cdef gsl_complex _gamma(gsl_complex D,gsl_complex E) nogil:
	return gsl_complex_negative(gsl_complex_div(D,gsl_complex_add(gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D,2),gsl_complex_pow_real(E,2))),1.0),E)))

#V: cm.exp(-z*cm.sqrt(D**2-E**2)/u)
cdef gsl_complex _V(gsl_complex D,gsl_complex E,double u,double z) nogil:
	return gsl_complex_exp(gsl_complex_negative(gsl_complex_mul_real(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D,2),gsl_complex_pow_real(E,2))),z/u)))

#W: t*(1.-V**2)/(1.-t**2)
cdef gsl_complex _W(gsl_complex t, gsl_complex V) nogil:
	return gsl_complex_div(gsl_complex_mul(gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(V,2)),1),t),gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t,2)),1))
	
#############################################
## Python wrappers for the basic functions ##
#############################################

#Delta_up
cpdef double complex Delta_up(SI_CI Sys,double x):
	cdef gsl_complex x_bar,res
	x_bar=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	res=_Delta(Sys,x_bar)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#Delta_down
cpdef double complex Delta_down(SI_CI Sys,double x):
	cdef gsl_complex x_bar,res
	x_bar=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	res=_Delta(Sys,x_bar)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#Sigma_up
cpdef double complex Sigma_up(SI_CI Sys,double x):
	cdef gsl_complex x_bar,res
	x_bar=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	res=_Sigma(Sys,x_bar)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#Sigma_down
cpdef double complex Sigma_down(SI_CI Sys,double x):
	cdef gsl_complex x_bar,res
	x_bar=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	res=_Sigma(Sys,x_bar)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#gamma_up
cpdef double complex gamma_up(SI_CI Sys,double x):
	cdef gsl_complex x_bar,D,E,res
	x_bar=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	D=_Delta(Sys,x_bar)
	E=_Sigma(Sys,x_bar)
	res=_gamma(D,E)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#gamma_down
cpdef double complex gamma_down(SI_CI Sys,double x):
	cdef gsl_complex x_bar,D,E,res
	x_bar=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	D=_Delta(Sys,x_bar)
	E=_Sigma(Sys,x_bar)
	res=_gamma(D,E)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#gamma_up,gamma_down -- optimized
cpdef double complex gamma(double complex D,double complex E):
	cdef gsl_complex DD,EE,res
	DD=gsl_complex_rect(D.real,D.imag)
	EE=gsl_complex_rect(E.real,E.imag)
	res=_gamma(DD,EE)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#V_up
cpdef double complex V_up(SI_CI Sys,double u,double x,double z):
	cdef gsl_complex x_bar,D,E,res
	x_bar=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	D=_Delta(Sys,x_bar)
	E=_Sigma(Sys,x_bar)
	res=_V(D,E,u,z)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#V_down
cpdef double complex V_down(SI_CI Sys,double u,double x,double z):
	cdef gsl_complex x_bar,D,E,res
	x_bar=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	D=_Delta(Sys,x_bar)
	E=_Sigma(Sys,x_bar)
	res=_V(D,E,u,z)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#V_up, V_down -- optimized
cpdef double complex V(double complex D,double complex E,double u,double z):
	cdef gsl_complex DD,EE,res
	DD=gsl_complex_rect(D.real,D.imag)
	EE=gsl_complex_rect(E.real,E.imag)
	res=_V(DD,EE,u,z)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#W_up
cpdef double complex W_up(SI_CI Sys,double u,double x,double z):
	cdef gsl_complex x_bar,D,E,V,t,res
	x_bar=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	D=_Delta(Sys,x_bar)
	E=_Sigma(Sys,x_bar)
	V=_V(D,E,u,z)
	t=_gamma(D,E)
	res=_W(t,V)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#W_down
cpdef double complex W_down(SI_CI Sys,double u,double x,double z):
	cdef gsl_complex x_bar,D,E,V,t,res
	x_bar=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	D=_Delta(Sys,x_bar)
	E=_Sigma(Sys,x_bar)
	V=_V(D,E,u,z)
	t=_gamma(D,E)
	res=_W(t,V)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#W_up, W_down -- optimized
cpdef double complex W(double complex t, double complex V):
	cdef gsl_complex VV,tt,res
	tt=gsl_complex_rect(t.real,t.imag)
	VV=gsl_complex_rect(V.real,V.imag)
	res=_W(tt,VV)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)
	
################################################
## Now we define the elements of a \Pi-matrix ##
################################################

#P_up_up: 0.25*sin(Sys.alpha)*cm.exp(-1.j*Sys.zeta)*((t_up-t_down)*(Sys.R_up-Sys.R_down+cos(Sys.alpha)*Sys.eta_1minus)-1.j*(t_up+t_down)*Sys.eta_2)
cdef gsl_complex _P_up_up(SI_CI Sys,gsl_complex t_up,gsl_complex t_down) nogil:
	cdef gsl_complex tmp
	tmp=gsl_complex_sub(gsl_complex_mul_real(gsl_complex_sub(t_up,t_down),(Sys.R_up-Sys.R_down+cos(Sys.alpha)*Sys.eta_1minus)),gsl_complex_mul_imag(gsl_complex_add(t_up,t_down),Sys.eta_2))
	return gsl_complex_mul_real(gsl_complex_mul(tmp,gsl_complex_exp(gsl_complex_rect(0,-Sys.zeta))),0.25*sin(Sys.alpha))

#P_down_down: 0.25*sin(Sys.alpha)*cm.exp(1.j*Sys.zeta)*((t_up-t_down)*(Sys.R_up-Sys.R_down-cos(Sys.alpha)*Sys.eta_1minus)+1.j*(t_up+t_down)*Sys.eta_2)
cdef gsl_complex _P_down_down(SI_CI Sys,gsl_complex t_up,gsl_complex t_down) nogil:
	cdef gsl_complex tmp
	tmp=gsl_complex_add(gsl_complex_mul_real(gsl_complex_sub(t_up,t_down),(Sys.R_up-Sys.R_down-cos(Sys.alpha)*Sys.eta_1minus)),gsl_complex_mul_imag(gsl_complex_add(t_up,t_down),Sys.eta_2))
	return gsl_complex_mul_real(gsl_complex_mul(tmp,gsl_complex_exp(gsl_complex_rect(0,Sys.zeta))),0.25*sin(Sys.alpha))

#P_up_down: 0.25*t_up*(Sys.eta_1plus-Sys.eta_1minus*cos(Sys.alpha)**2+2.j*cos(Sys.alpha)*Sys.eta_2-4)-0.25*t_down*Sys.eta_1minus*sin(Sys.alpha)**2
cdef gsl_complex _P_up_down(SI_CI Sys,gsl_complex t_up,gsl_complex t_down) nogil:
	return gsl_complex_mul_real(gsl_complex_sub(gsl_complex_mul(t_up,gsl_complex_rect(Sys.eta_1plus-Sys.eta_1minus*cpow(cos(Sys.alpha),2)-4,2*cos(Sys.alpha)*Sys.eta_2)),gsl_complex_mul_real(t_down,Sys.eta_1minus*cpow(sin(Sys.alpha),2))),0.25)

#P_down_up: 0.25*t_up*Sys.eta_1minus*sin(Sys.alpha)**2-0.25*t_down*(Sys.eta_1plus-Sys.eta_1minus*cos(Sys.alpha)**2-2.j*cos(Sys.alpha)*Sys.eta_2-4)
cdef gsl_complex _P_down_up(SI_CI Sys,gsl_complex t_up,gsl_complex t_down) nogil:
	return gsl_complex_mul_real(gsl_complex_sub(gsl_complex_mul_real(t_up,Sys.eta_1minus*cpow(sin(Sys.alpha),2)),gsl_complex_mul(t_down,gsl_complex_rect(Sys.eta_1plus-Sys.eta_1minus*cpow(cos(Sys.alpha),2)-4,-2*cos(Sys.alpha)*Sys.eta_2))),0.25)

####################################################
## Python wrapper to the elements of a \Pi-matrix ##
####################################################

#P_up_up
cpdef double complex P_up_up(SI_CI Sys,double complex t_up,double complex t_down):
	cdef gsl_complex tt_up,tt_down,res
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	res=_P_up_up(Sys,tt_up,tt_down)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#P_down_down
cpdef double complex P_down_down(SI_CI Sys,double complex t_up,double complex t_down):
	cdef gsl_complex tt_up,tt_down,res
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	res=_P_down_down(Sys,tt_up,tt_down)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#P_up_down
cpdef double complex P_up_down(SI_CI Sys,double complex t_up,double complex t_down):
	cdef gsl_complex tt_up,tt_down,res
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	res=_P_up_down(Sys,tt_up,tt_down)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#P_down_up
cpdef double complex P_down_up(SI_CI Sys,double complex t_up,double complex t_down):
	cdef gsl_complex tt_up,tt_down,res
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	res=_P_down_up(Sys,tt_up,tt_down)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

###############################################################################
## Here we define general methods useful for creating Numpy and GSL matrices ##
###############################################################################

#Let us define a general method, which creates a 2x2 GSL matrix of 4 supplied elements and returns its pointer
cdef gsl_matrix_complex * Els_to_Mat(gsl_complex a00,gsl_complex a01,gsl_complex a10,gsl_complex a11) nogil:
	cdef gsl_matrix_complex * m = gsl_matrix_complex_calloc(2,2)
	gsl_matrix_complex_set(m,0,0,a00)
	gsl_matrix_complex_set(m,0,1,a01)
	gsl_matrix_complex_set(m,1,0,a10)
	gsl_matrix_complex_set(m,1,1,a11)
	return m

#Method which turns the GSL matrix into a corresponding Numpy array
cdef np.ndarray[DTYPE_t, ndim=2] GSL_to_Numpy_mat(gsl_matrix_complex * m):
	cdef np.ndarray[DTYPE_t, ndim=2] res=np.zeros([2,2],dtype=np.complex128)
	cdef gsl_complex a00,a01,a10,a11
	a00=gsl_matrix_complex_get(m,0,0)
	a01=gsl_matrix_complex_get(m,0,1)
	a10=gsl_matrix_complex_get(m,1,0)
	a11=gsl_matrix_complex_get(m,1,1)
	res[0,0]=GSL_REAL(a00)+1.j*GSL_IMAG(a00)
	res[0,1]=GSL_REAL(a01)+1.j*GSL_IMAG(a01)
	res[1,0]=GSL_REAL(a10)+1.j*GSL_IMAG(a10)
	res[1,1]=GSL_REAL(a11)+1.j*GSL_IMAG(a11)
	return res

#Method which turns the Numpy array into a corresponding GSL matrix
cdef gsl_matrix_complex * Numpy_to_GSL_mat(double complex[:,::1] m) nogil:
	cdef gsl_matrix_complex * res = gsl_matrix_complex_calloc(2,2)
	cdef gsl_complex a00,a01,a10,a11
	cdef double complex b00,b01,b10,b11
	b00,b01,b10,b11=m[0,0],m[0,1],m[1,0],m[1,1]
	a00=gsl_complex_rect(b00.real,b00.imag)
	a01=gsl_complex_rect(b01.real,b01.imag)
	a10=gsl_complex_rect(b10.real,b10.imag)
	a11=gsl_complex_rect(b11.real,b11.imag)
	return Els_to_Mat(a00,a01,a10,a11)

#Let us define a general method, which creates a 2x2 Numpy array of 4 supplied elements
cpdef np.ndarray[DTYPE_t, ndim=2] Els_to_Arr(double complex a11,double complex a12,double complex a21,double complex a22):
	cdef np.ndarray[DTYPE_t, ndim=2] res=np.zeros([2,2],dtype=np.complex128)
	res[0,0],res[0,1],res[1,0],res[1,1]=a11,a12,a21,a22
	return res

#Let us define a function, which returns the determinant of a gsl_matrix_complex
cdef gsl_complex det(gsl_matrix_complex *P) nogil:
	cdef int sign=0 #initialize the sign parameter
	cdef gsl_permutation * p = gsl_permutation_calloc(2) #create a permutation
	cdef gsl_matrix_complex * tmp_ptr = gsl_matrix_complex_calloc(2,2) #create a temporary matrix pointer
	gsl_matrix_complex_memcpy(tmp_ptr,P) #copying the elements of matrix m to the temporary one in order not to spoil P
	gsl_linalg_complex_LU_decomp(tmp_ptr,p,&sign) #finding the LU decomposition and the corresponding permutation sign
	cdef gsl_complex res
	res=gsl_linalg_complex_LU_det(tmp_ptr,sign) #calculate the determinant from the LU decomposition
	gsl_permutation_free(p) #release the memory taken by the permutation
	gsl_matrix_complex_free(tmp_ptr) #release the memory taken by the supplementary function
	return res

#####################################################################
## Now we can use the aboved above methods to create a \Pi-matrix. ##
## Both GSL and Numpy versions are provided below                  ##
#####################################################################

#Now we define the GSL \Pi-matrix
cdef gsl_matrix_complex * _P_matrix(SI_CI Sys,gsl_complex t_up,gsl_complex t_down) nogil:
	cdef gsl_complex a00,a01,a10,a11
	a00,a01,a10,a11=_P_up_up(Sys,t_up,t_down),_P_up_down(Sys,t_up,t_down),_P_down_up(Sys,t_up,t_down),_P_down_down(Sys,t_up,t_down)
	return Els_to_Mat(a00,a01,a10,a11)

#Now we define the Numpy \Pi-matrix
cpdef np.ndarray[DTYPE_t, ndim=2] P_matrix(SI_CI Sys,double complex t_up,double complex t_down):
	cdef double complex a11,a12,a21,a22
	a11,a12,a21,a22=P_up_up(Sys,t_up,t_down),P_up_down(Sys,t_up,t_down),P_down_up(Sys,t_up,t_down),P_down_down(Sys,t_up,t_down)
	return Els_to_Arr(a11,a12,a21,a22)

#############################################################
## *****************  Matrix G_r_out **********************##
#############################################################

#DD: (1-t_up**2)*(1-t_down**2)+(1-t_up**2)*t_down*P[1,0]-(1-t_down**2)*t_up*P[0,1]+t_up*t_down*detP
cdef gsl_complex _DD(gsl_complex t_up,gsl_complex t_down,gsl_matrix_complex * P,gsl_complex detP) nogil:
	cdef gsl_complex tmp_up, tmp_down, P10, P01
	tmp_up=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_up,2)),1)
	tmp_down=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_down,2)),1)
	P01=gsl_matrix_complex_get(P,0,1)
	P10=gsl_matrix_complex_get(P,1,0)
	return gsl_complex_add(gsl_complex_sub(gsl_complex_add(gsl_complex_mul(tmp_up,tmp_down),gsl_complex_mul(gsl_complex_mul(tmp_up,t_down),P10)),gsl_complex_mul(gsl_complex_mul(tmp_down,t_up),P01)),gsl_complex_mul(gsl_complex_mul(t_up,t_down),detP))

#Python wrapper to DD -- get DD_GSL
cpdef double complex get_DD_GSL(double complex t_up,double complex t_down,np.ndarray[DTYPE_t, ndim=2] P,double complex detP):
	cdef gsl_complex ddetP,tt_up,tt_down,res
	cdef gsl_matrix_complex * PP
	PP=Numpy_to_GSL_mat(P)
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	ddetP=gsl_complex_rect(detP.real,detP.imag)
	res=_DD(tt_up,tt_down,PP,ddetP)
	gsl_matrix_complex_free(PP) #IMPORTANT: don't forget to release memory taken by a GSL matrix
	return GSL_REAL(res)+1.j*GSL_IMAG(res)
	
#OBSOLETE: Define the Python version of the denominator of the G_r_out matrix
cpdef double complex DD(double complex t_up,double complex t_down,np.ndarray[DTYPE_t, ndim=2] P,double complex detP):
	return (1-t_up**2)*(1-t_down**2)+(1-t_up**2)*t_down*P[1,0]-(1-t_down**2)*t_up*P[0,1]+t_up*t_down*detP

## Define K_up and K_down functions, useful for many functions below

#K_up: ((1-t_down**2)*P[0,1]-t_down*detP)*t_up*V_up**2/((1-t_up**2)*DD)
cdef gsl_complex _K_up(gsl_complex t_up,gsl_complex t_down,gsl_complex DD,gsl_complex V_up,gsl_matrix_complex * P,gsl_complex detP) nogil:
	cdef gsl_complex P01,tmp_up,tmp_down
	P01=gsl_matrix_complex_get(P,0,1)
	tmp_up=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_up,2)),1)
	tmp_down=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_down,2)),1)
	return gsl_complex_div(gsl_complex_mul(gsl_complex_mul(gsl_complex_sub(gsl_complex_mul(tmp_down,P01),gsl_complex_mul(t_down,detP)),t_up),gsl_complex_pow_real(V_up,2)),gsl_complex_mul(tmp_up,DD))

#K_down: ((1-t_up**2)*P[1,0]+t_up*detP)*t_down*V_down**2/((1-t_down**2)*DD)
cdef gsl_complex _K_down(gsl_complex t_up,gsl_complex t_down,gsl_complex DD,gsl_complex V_down,gsl_matrix_complex * P,gsl_complex detP) nogil:
	cdef gsl_complex P10,tmp_up,tmp_down
	P10=gsl_matrix_complex_get(P,1,0)
	tmp_up=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_up,2)),1)
	tmp_down=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_down,2)),1)
	return gsl_complex_div(gsl_complex_mul(gsl_complex_mul(gsl_complex_add(gsl_complex_mul(tmp_up,P10),gsl_complex_mul(t_up,detP)),t_down),gsl_complex_pow_real(V_down,2)),gsl_complex_mul(tmp_down,DD))

#Below we define Python wrappers for the two above functions to test their functionality
cpdef double complex get_K_up_GSL(double complex t_up,double complex t_down,double complex DD,double complex V_up,np.ndarray[DTYPE_t, ndim=2] P,double complex detP):
	cdef gsl_complex tt_up,tt_down,D,VV_up,ddetP,res
	cdef gsl_matrix_complex * PP
	PP=Numpy_to_GSL_mat(P)
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	D=gsl_complex_rect(DD.real,DD.imag)
	VV_up=gsl_complex_rect(V_up.real,V_up.imag)
	ddetP=gsl_complex_rect(detP.real,detP.imag)
	res=_K_up(tt_up,tt_down,D,VV_up,PP,ddetP)
	gsl_matrix_complex_free(PP) #IMPORTANT: don't forget to release memory taken by a GSL matrix
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

cpdef double complex get_K_down_GSL(double complex t_up,double complex t_down,double complex DD,double complex V_down,np.ndarray[DTYPE_t, ndim=2] P,double complex detP):
	cdef gsl_complex tt_up,tt_down,D,VV_down,ddetP,res
	cdef gsl_matrix_complex * PP
	PP=Numpy_to_GSL_mat(P)
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	D=gsl_complex_rect(DD.real,DD.imag)
	VV_down=gsl_complex_rect(V_down.real,V_down.imag)
	ddetP=gsl_complex_rect(detP.real,detP.imag)
	res=_K_down(tt_up,tt_down,D,VV_down,PP,ddetP)
	gsl_matrix_complex_free(PP) #IMPORTANT: don't forget to release memory taken by a GSL matrix
	return GSL_REAL(res)+1.j*GSL_IMAG(res)
	
#OBSOLETE: Now we define the Python version of the two useful functions defined above
cpdef double complex K_up(SI_CI Sys,double complex t_up,double complex t_down,double complex DD,double complex V_up,np.ndarray[DTYPE_t, ndim=2] P,double complex detP):
	return ((1-t_down**2)*P[0,1]-t_down*detP)*t_up*V_up**2/((1-t_up**2)*DD)

cpdef double complex K_down(SI_CI Sys,double complex t_up,double complex t_down,double complex DD,double complex V_down,np.ndarray[DTYPE_t, ndim=2] P,double complex detP):
	return ((1-t_up**2)*P[1,0]+t_up*detP)*t_down*V_down**2/((1-t_down**2)*DD)

############################################################
## Below we define all 4 elements of the G_r_out function ##
############################################################

#G_r_out_up_up <-- use K_up: 1./(1-t_up**2)+K_up(t_up,t_down,DD,V_up,P,detP)
cdef gsl_complex _G_r_out_up_up(gsl_complex t_up,gsl_complex t_down,gsl_complex DD,gsl_complex V_up,gsl_matrix_complex * P,gsl_complex detP) nogil:
	cdef gsl_complex Kup,tmp_up
	Kup=_K_up(t_up,t_down,DD,V_up,P,detP)
	tmp_up=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_up,2)),1)
	return gsl_complex_add(gsl_complex_inverse(tmp_up),Kup)

#G_r_out_down_down <-- use K_down: 1./(1-t_down**2)-K_down(t_up,t_down,DD,V_down,P,detP)
cdef gsl_complex _G_r_out_down_down(gsl_complex t_up,gsl_complex t_down,gsl_complex DD,gsl_complex V_down,gsl_matrix_complex * P,gsl_complex detP) nogil:
	cdef gsl_complex Kdown,tmp_down
	Kdown=_K_down(t_up,t_down,DD,V_down,P,detP)
	tmp_down=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_down,2)),1)
	return gsl_complex_sub(gsl_complex_inverse(tmp_down),Kdown)

#G_r_out_up_down
cdef gsl_complex _G_r_out_up_down(gsl_complex t_down,gsl_complex V_up,gsl_complex V_down,gsl_complex DD,gsl_matrix_complex * P) nogil:
	cdef gsl_complex P00
	P00=gsl_matrix_complex_get(P,0,0)
	return gsl_complex_negative(gsl_complex_div(gsl_complex_mul(gsl_complex_mul(gsl_complex_mul(t_down,V_up),V_down),P00),DD))

#G_r_out_down_up
cdef gsl_complex _G_r_out_down_up(gsl_complex t_up,gsl_complex V_up,gsl_complex V_down,gsl_complex DD,gsl_matrix_complex * P) nogil:
	cdef gsl_complex P11
	P11=gsl_matrix_complex_get(P,1,1)
	return gsl_complex_div(gsl_complex_mul(gsl_complex_mul(gsl_complex_mul(t_up,V_up),V_down),P11),DD)

#Now we will define the elements of the G_r_out_tilde function. We define only the off-diagonal elements since
#G_r_out_tilde_up_up=G_r_out_down_down, G_r_out_tilde_down_down=G_r_out_up_up

#G_r_out_tilde_up_down
cdef gsl_complex _G_r_out_tilde_up_down(gsl_complex t_down,gsl_complex V_up,gsl_complex V_down,gsl_complex DD,gsl_matrix_complex * P) nogil:
	cdef gsl_complex P11
	P11=gsl_matrix_complex_get(P,1,1)
	return gsl_complex_negative(gsl_complex_div(gsl_complex_mul(gsl_complex_mul(gsl_complex_mul(t_down,V_up),V_down),P11),DD))

#G_r_out_tilde_down_up
cdef gsl_complex _G_r_out_tilde_down_up(gsl_complex t_up,gsl_complex V_up,gsl_complex V_down,gsl_complex DD,gsl_matrix_complex * P) nogil:
	cdef gsl_complex P00
	P00=gsl_matrix_complex_get(P,0,0)
	return gsl_complex_div(gsl_complex_mul(gsl_complex_mul(gsl_complex_mul(t_up,V_up),V_down),P00),DD)

###########################################
## Python wrappers for the above methods ##
###########################################

#G_r_out_up_up <-- use K_up
cpdef double complex get_G_r_out_up_up(double complex t_up,double complex t_down,double complex DD,double complex V_up,np.ndarray[DTYPE_t, ndim=2] P,double complex detP):
	cdef gsl_complex tt_up,tt_down,D,VV_up,ddetP,res
	cdef gsl_matrix_complex *PP
	PP=Numpy_to_GSL_mat(P)
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	D=gsl_complex_rect(DD.real,DD.imag)
	VV_up=gsl_complex_rect(V_up.real,V_up.imag)
	ddetP=gsl_complex_rect(detP.real,detP.imag)
	res=_G_r_out_up_up(tt_up,tt_down,D,VV_up,PP,ddetP)
	gsl_matrix_complex_free(PP)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#G_r_out_down_down <-- use K_down
cpdef double complex get_G_r_out_down_down(double complex t_up,double complex t_down,double complex DD,double complex V_down,np.ndarray[DTYPE_t, ndim=2] P,double complex detP):
	cdef gsl_complex tt_up,tt_down,D,VV_down,ddetP,res
	cdef gsl_matrix_complex *PP
	PP=Numpy_to_GSL_mat(P)
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	D=gsl_complex_rect(DD.real,DD.imag)
	VV_down=gsl_complex_rect(V_down.real,V_down.imag)
	ddetP=gsl_complex_rect(detP.real,detP.imag)
	res=_G_r_out_down_down(tt_up,tt_down,D,VV_down,PP,ddetP)
	gsl_matrix_complex_free(PP)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#G_r_out_up_down
cpdef double complex get_G_r_out_up_down(double complex t_down,double complex V_up,double complex V_down,double complex DD,np.ndarray[DTYPE_t, ndim=2] P):
	cdef gsl_complex tt_down,VV_up,VV_down,res
	cdef gsl_matrix_complex *PP
	PP=Numpy_to_GSL_mat(P)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	VV_up=gsl_complex_rect(V_up.real,V_up.imag)
	VV_down=gsl_complex_rect(V_down.real,V_down.imag)
	D=gsl_complex_rect(DD.real,DD.imag)
	res=_G_r_out_up_down(tt_down,VV_up,VV_down,D,PP)
	gsl_matrix_complex_free(PP)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#G_r_out_down_up
cpdef double complex get_G_r_out_down_up(double complex t_up,double complex V_up,double complex V_down,double complex DD,np.ndarray[DTYPE_t, ndim=2] P):
	cdef gsl_complex tt_up,VV_up,VV_down,res
	cdef gsl_matrix_complex *PP
	PP=Numpy_to_GSL_mat(P)
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	VV_up=gsl_complex_rect(V_up.real,V_up.imag)
	VV_down=gsl_complex_rect(V_down.real,V_down.imag)
	D=gsl_complex_rect(DD.real,DD.imag)
	res=_G_r_out_down_up(tt_up,VV_up,VV_down,D,PP)
	gsl_matrix_complex_free(PP)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#G_r_out_tilde_up_down
cpdef double complex get_G_r_out_tilde_up_down(double complex t_down,double complex V_up,double complex V_down,double complex DD,np.ndarray[DTYPE_t, ndim=2] P):
	cdef gsl_complex tt_down,VV_up,VV_down,res
	cdef gsl_matrix_complex *PP
	PP=Numpy_to_GSL_mat(P)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	VV_up=gsl_complex_rect(V_up.real,V_up.imag)
	VV_down=gsl_complex_rect(V_down.real,V_down.imag)
	D=gsl_complex_rect(DD.real,DD.imag)
	res=_G_r_out_tilde_up_down(tt_down,VV_up,VV_down,D,PP)
	gsl_matrix_complex_free(PP)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#G_r_out_tilde_down_up
cpdef double complex get_G_r_out_tilde_down_up(double complex t_up,double complex V_up,double complex V_down,double complex DD,np.ndarray[DTYPE_t, ndim=2] P):
	cdef gsl_complex tt_up,VV_up,VV_down,res
	cdef gsl_matrix_complex *PP
	PP=Numpy_to_GSL_mat(P)
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	VV_up=gsl_complex_rect(V_up.real,V_up.imag)
	VV_down=gsl_complex_rect(V_down.real,V_down.imag)
	D=gsl_complex_rect(DD.real,DD.imag)
	res=_G_r_out_tilde_down_up(tt_up,VV_up,VV_down,D,PP)
	gsl_matrix_complex_free(PP)
	return GSL_REAL(res)+1.j*GSL_IMAG(res)

#####################################
## Incoming distribution functions ##
#####################################

#Supplementary functions -- version with GIL
cpdef double h_up(SI_CI Sys,double x):
	return tanh((x-Sys.Ez)/(2*Sys.kT))

cpdef double h_down(SI_CI Sys,double x):
	return tanh((x+Sys.Ez)/(2*Sys.kT))

#Supplementary functions -- NOGIL version
cdef double _h_up(SI_CI Sys,double x) nogil:
	return tanh((x-Sys.Ez)/(2*Sys.kT))

cdef double _h_down(SI_CI Sys,double x) nogil:
	return tanh((x+Sys.Ez)/(2*Sys.kT))

#Distribution functions in F
cdef double _x1_up(SI_CI Sys,double x) nogil:
	return _h_up(Sys,x-Sys.eV)

cdef double _x1_down(SI_CI Sys,double x) nogil:
	return _h_down(Sys,x-Sys.eV)

cdef double _x1_up_tilde(SI_CI Sys,double x) nogil:
	return -_h_down(Sys,x+Sys.eV)

cdef double _x1_down_tilde(SI_CI Sys,double x) nogil:
	return -_h_up(Sys,x+Sys.eV)

#Distribution functions in S
cdef double _x2_up(SI_CI Sys,double x,gsl_complex t_up) nogil:
	return (1-gsl_complex_abs2(t_up))*_h_up(Sys,x)

cdef double _x2_down(SI_CI Sys,double x,gsl_complex t_down) nogil:
	return (1-gsl_complex_abs2(t_down))*_h_down(Sys,x)

###############################################################################################
## Here go the functions for calculating LDOS and the spectral part of induced magnetization ##
###############################################################################################

#Integrals in u of K_up and K_down functions - can be taken explicitly using the (incomplete) gamma function
cpdef double complex int_u_K_up(double complex t_up,double complex t_down,double complex DD,np.ndarray[DTYPE_t, ndim=2] P,double complex detP,double complex E_up,double complex D_up,double z):
	cdef double complex a,int_V_up_2
	a=2*z*cm.sqrt(D_up**2-E_up**2)
	int_V_up_2=cm.exp(-a)-a*mpmath.gammainc(0,a)
	return ((1-t_down**2)*P[0,1]-t_down*detP)*t_up*int_V_up_2/((1-t_up**2)*DD)

cpdef double complex int_u_K_down(double complex t_up,double complex t_down,double complex DD,np.ndarray[DTYPE_t, ndim=2] P,double complex detP,double complex E_down,double complex D_down,double z):
	cdef double complex a,int_V_down_2
	a=2*z*cm.sqrt(D_down**2-E_down**2)
	int_V_down_2=cm.exp(-a)-a*mpmath.gammainc(0,a)
	return ((1-t_up**2)*P[1,0]+t_up*detP)*t_down*int_V_down_2/((1-t_down**2)*DD)

#Total LDOS, LDOS_up, and LDOS_down
cpdef double LDOS(SI_CI Sys,double x,double z):
	cdef double complex t_up,t_down,valDD,E_up,E_down,D_up,D_down,res,detP
	cdef np.ndarray[DTYPE_t, ndim=2] P
	E_up,E_down=Sigma_up(Sys,x),Sigma_down(Sys,x)
	D_up,D_down=Delta_up(Sys,x),Delta_down(Sys,x)
	t_up,t_down=gamma(D_up,E_up),gamma(D_down,E_down)
	P=P_matrix(Sys,t_up,t_down)
	detP=np.linalg.det(P)
	valDD=DD(t_up,t_down,P,detP)
	res=(1+t_up**2)/(1-t_up**2)+(1+t_down**2)/(1-t_down**2)+2*int_u_K_up(t_up,t_down,valDD,P,detP,E_up,D_up,z)-2*int_u_K_down(t_up,t_down,valDD,P,detP,E_down,D_down,z)
	return res.real

cpdef double LDOS_up(SI_CI Sys,double x,double z):
	cdef double complex t_up,t_down,valDD,E_up,E_down,D_up,D_down,res,detP
	cdef np.ndarray[DTYPE_t, ndim=2] P
	E_up,E_down=Sigma_up(Sys,x),Sigma_down(Sys,x)
	D_up,D_down=Delta_up(Sys,x),Delta_down(Sys,x)
	t_up,t_down=gamma(D_up,E_up),gamma(D_down,E_down)
	P=P_matrix(Sys,t_up,t_down)
	detP=np.linalg.det(P)
	valDD=DD(t_up,t_down,P,detP)
	res=(1+t_up**2)/(1-t_up**2)+2*int_u_K_up(t_up,t_down,valDD,P,detP,E_up,D_up,z)
	return res.real

cpdef double LDOS_down(SI_CI Sys,double x,double z):
	cdef double complex t_up,t_down,valDD,E_up,E_down,D_up,D_down,res,detP
	cdef np.ndarray[DTYPE_t, ndim=2] P
	E_up,E_down=Sigma_up(Sys,x),Sigma_down(Sys,x)
	D_up,D_down=Delta_up(Sys,x),Delta_down(Sys,x)
	t_up,t_down=gamma(D_up,E_up),gamma(D_down,E_down)
	P=P_matrix(Sys,t_up,t_down)
	detP=np.linalg.det(P)
	valDD=DD(t_up,t_down,P,detP)
	res=(1+t_down**2)/(1-t_down**2)-2*int_u_K_down(t_up,t_down,valDD,P,detP,E_down,D_down,z)
	return res.real

#Integral in u of the A-function for calculating the spectral part of the induced magnetization (x and y components)
#NOTE: 1/2 from formula (123) is included here
cpdef double complex int_u_A(SI_CI Sys,double complex t_up,double complex t_down,double complex DD,double complex E_up,double complex D_up,double complex E_down,double complex D_down,double z):
	cdef double complex a,int_V_upV_down
	a=z*(cm.sqrt(D_up**2-E_up**2)+cm.sqrt(D_down**2-E_down**2))
	int_V_upV_down=cm.exp(-a)-a*mpmath.gammainc(0,a)
	return (t_up+t_down)*int_V_upV_down*(1.j*(t_up+t_down)*Sys.eta_2-cos(Sys.alpha)*(t_up-t_down)*Sys.eta_1minus)/(4*DD)

#Integrands for the components of the spectral part of induced magnetization
#NOTE: 1/2 from formula (123) is already included here
cpdef double M_spx_x(double x,SI_CI Sys,double z):
	cdef double complex t_up,t_down,E_up,E_down,D_up,D_down,valDD,res_int_u_A
	cdef np.ndarray[DTYPE_t, ndim=2] P
	E_up,E_down=Sigma_up(Sys,x),Sigma_down(Sys,x)
	D_up,D_down=Delta_up(Sys,x),Delta_down(Sys,x)
	t_up,t_down=gamma(D_up,E_up),gamma(D_down,E_down)
	P=P_matrix(Sys,t_up,t_down)
	valDD=DD(t_up,t_down,P,np.linalg.det(P))
	res_int_u_A=int_u_A(Sys,t_up,t_down,valDD,E_up,D_up,E_down,D_down,z)
	return (h_up(Sys,x)+h_down(Sys,x))*sin(Sys.alpha)*cos(Sys.zeta)*res_int_u_A.real-(h_up(Sys,x)-h_down(Sys,x))*sin(Sys.alpha)*sin(Sys.zeta)*res_int_u_A.imag

#NOTE: 1/2 from formula (123) is already included here
cpdef double M_spy_x(double x,SI_CI Sys,double z):
	cdef double complex t_up,t_down,E_up,E_down,D_up,D_down,valDD,res_int_u_A
	cdef np.ndarray[DTYPE_t, ndim=2] P
	E_up,E_down=Sigma_up(Sys,x),Sigma_down(Sys,x)
	D_up,D_down=Delta_up(Sys,x),Delta_down(Sys,x)
	t_up,t_down=gamma(D_up,E_up),gamma(D_down,E_down)
	P=P_matrix(Sys,t_up,t_down)
	valDD=DD(t_up,t_down,P,np.linalg.det(P))
	res_int_u_A=int_u_A(Sys,t_up,t_down,valDD,E_up,D_up,E_down,D_down,z)
	return (h_up(Sys,x)+h_down(Sys,x))*sin(Sys.alpha)*sin(Sys.zeta)*res_int_u_A.real+(h_up(Sys,x)-h_down(Sys,x))*sin(Sys.alpha)*cos(Sys.zeta)*res_int_u_A.imag

#NOTE: 1/2 from formula (123) is already included here
cpdef double M_spz_x(double x,SI_CI Sys,double z):
	cdef double complex t_up,t_down,valDD,E_up,E_down,D_up,D_down,res,detP
	cdef np.ndarray[DTYPE_t, ndim=2] P
	E_up,E_down=Sigma_up(Sys,x),Sigma_down(Sys,x)
	D_up,D_down=Delta_up(Sys,x),Delta_down(Sys,x)
	t_up,t_down=gamma(D_up,E_up),gamma(D_down,E_down)
	P=P_matrix(Sys,t_up,t_down)
	detP=np.linalg.det(P)
	valDD=DD(t_up,t_down,P,detP)
	res=h_up(Sys,x)*((1+t_up**2)/(1-t_up**2)+2*int_u_K_up(t_up,t_down,valDD,P,detP,E_up,D_up,z))-h_down(Sys,x)*((1+t_down**2)/(1-t_down**2)-2*int_u_K_down(t_up,t_down,valDD,P,detP,E_down,D_down,z))
	return res.real

#Integrand which is necessary to calculate the background induced magnetization (present when B!=0)
cpdef double M_spz_x_background(double x,SI_CI Sys,double z):
	cdef double complex t_up,t_down,E_up,E_down,D_up,D_down,res
	E_up,E_down=Sigma_up(Sys,x),Sigma_down(Sys,x)
	D_up,D_down=Delta_up(Sys,x),Delta_down(Sys,x)
	t_up,t_down=gamma(D_up,E_up),gamma(D_down,E_down)
	res=h_up(Sys,x)*((1+t_up**2)/(1-t_up**2))-h_down(Sys,x)*((1+t_down**2)/(1-t_down**2))
	return res.real

#Now we define the functions which use the above M_spx,y,z_x to get the values of induced magnetization

#M_spx
cpdef np.ndarray[DTYPE_t_r, ndim=1] M_spx(SI_CI Sys,double x_cutoff,double z):
	cdef double eabs,erel,x_left,x_right,xdiv_l,xdiv_r,res,error,Nevals,Nwarn
	cdef int Nlim=1000
	cdef np.ndarray[DTYPE_t_r, ndim=1] pts
	eabs,erel=100*Sys.reg_c,1e-8
	res,error,Nevals,Nwarn=0,0,0,0
	x_left=cos(Sys.theta/2)-Sys.T_up-Sys.T_down+Sys.Ez
	x_right=cos(Sys.theta/2)+Sys.T_up+Sys.T_down+Sys.Ez
	xdiv_l=1-Sys.Ez
	xdiv_r=1+Sys.Ez
	if (x_left<=0):
		pts=np.array([x_right,xdiv_l,xdiv_r],dtype=np.double)
	else:
		pts=np.array([x_left,x_right,xdiv_l,xdiv_r],dtype=np.double)
	tmp=integrate.quad(M_spx_x,0.0,10*x_cutoff,args=(Sys,z),full_output=0,epsabs=eabs,epsrel=erel,limit=Nlim,points=pts)
	res+=tmp[0]
	error+=tmp[1]
	Nevals+=tmp[2]['neval']
	if (len(tmp)>3):
		Nwarn+=1
	return np.array([res,error,Nevals,Nwarn],dtype=np.double)

#M_spy
cpdef np.ndarray[DTYPE_t_r, ndim=1] M_spy(SI_CI Sys,double x_cutoff,double z):
	cdef double eabs,erel,x_left,x_right,xdiv_l,xdiv_r,res,error,Nevals,Nwarn
	cdef int Nlim=1000
	cdef np.ndarray[DTYPE_t_r, ndim=1] pts
	eabs,erel=100*Sys.reg_c,1e-8
	res,error,Nevals,Nwarn=0,0,0,0
	x_left=cos(Sys.theta/2)-Sys.T_up-Sys.T_down+Sys.Ez
	x_right=cos(Sys.theta/2)+Sys.T_up+Sys.T_down+Sys.Ez
	xdiv_l=1-Sys.Ez
	xdiv_r=1+Sys.Ez
	if (x_left<=0):
		pts=np.array([x_right,xdiv_l,xdiv_r],dtype=np.double)
	else:
		pts=np.array([x_left,x_right,xdiv_l,xdiv_r],dtype=np.double)
	tmp=integrate.quad(M_spy_x,0.0,10*x_cutoff,args=(Sys,z),full_output=0,epsabs=eabs,epsrel=erel,limit=Nlim,points=pts)
	res+=tmp[0]
	error+=tmp[1]
	Nevals+=tmp[2]['neval']
	if (len(tmp)>3):
		Nwarn+=1
	return np.array([res,error,Nevals,Nwarn],dtype=np.double)

#M_spz
cpdef np.ndarray[DTYPE_t_r, ndim=1] M_spz(SI_CI Sys,double x_cutoff,double z):
	cdef double eabs,erel,x_left,x_right,xdiv_l,xdiv_r,res,error,Nevals,Nwarn
	cdef int Nlim=1000
	cdef np.ndarray[DTYPE_t_r, ndim=1] pts
	eabs,erel=100*Sys.reg_c,1e-8
	res,error,Nevals,Nwarn=0,0,0,0
	x_left=cos(Sys.theta/2)-Sys.T_up-Sys.T_down+Sys.Ez
	x_right=cos(Sys.theta/2)+Sys.T_up+Sys.T_down+Sys.Ez
	xdiv_l=1-Sys.Ez
	xdiv_r=1+Sys.Ez
	if (x_left<=0):
		pts=np.array([x_right,xdiv_l,xdiv_r],dtype=np.double)
	else:
		pts=np.array([x_left,x_right,xdiv_l,xdiv_r],dtype=np.double)
	tmp=integrate.quad(M_spz_x,0.0,10*x_cutoff,args=(Sys,z),full_output=0,epsabs=eabs,epsrel=erel,limit=Nlim,points=pts)
	res+=tmp[0]
	error+=tmp[1]
	Nevals+=tmp[2]['neval']
	if (len(tmp)>3):
		Nwarn+=1
	return np.array([res,error,Nevals,Nwarn],dtype=np.double)

#M_spz_background
cpdef np.ndarray[DTYPE_t_r, ndim=1] M_spz_background(SI_CI Sys,double x_cutoff,double z):
	cdef double eabs,erel,x_left,x_right,xdiv_l,xdiv_r,res,error,Nevals,Nwarn
	cdef int Nlim=1000
	cdef np.ndarray[DTYPE_t_r, ndim=1] pts
	eabs,erel=100*Sys.reg_c,1e-8
	res,error,Nevals,Nwarn=0,0,0,0
	x_left=cos(Sys.theta/2)-Sys.T_up-Sys.T_down+Sys.Ez
	x_right=cos(Sys.theta/2)+Sys.T_up+Sys.T_down+Sys.Ez
	xdiv_l=fmin(1-Sys.Ez,1+Sys.Ez)
	xdiv_r=fmax(1-Sys.Ez,1+Sys.Ez)
	if (x_left<=0):
		pts=np.array([x_right,xdiv_l,xdiv_r],dtype=np.double)
	else:
		pts=np.array([x_left,x_right,xdiv_l,xdiv_r],dtype=np.double)
	tmp=integrate.quad(M_spz_x_background,0.0,10*x_cutoff,args=(Sys,z),full_output=0,epsabs=eabs,epsrel=erel,limit=Nlim,points=pts)
	res+=tmp[0]
	error+=tmp[1]
	Nevals+=tmp[2]['neval']
	if (len(tmp)>3):
		Nwarn+=1
	return np.array([res,error,Nevals,Nwarn],dtype=np.double)

####################################################################################################################
## Now we will introduce the functions necessary to numerically find the outgoing anomalous distribution function ##
####################################################################################################################

#NOTE: This part is written using only GSL library (with 'nogil' directive)

#Reflectrion and transmission scattering matrices
cdef gsl_matrix_complex * M(SI_CI Sys,double Kup,double Kdown) nogil:
	cdef double theta,alpha,zeta
	theta,alpha,zeta=Sys.theta,Sys.alpha,Sys.zeta
	cdef gsl_complex a11,a12,a21,a22
	#a11
	a11=gsl_complex_add(gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0,theta/2)),0.5*sqrt(Kup)*(1+cos(alpha))),gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0,-theta/2)),0.5*sqrt(Kdown)*(1-cos(alpha))))
	#a12
	a12=gsl_complex_mul(gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0.0,-zeta)),0.5*sin(alpha)),gsl_complex_sub(gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0.0,theta/2)),sqrt(Kup)),gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0.0,-theta/2)),sqrt(Kdown))))
	#a21
	a21=gsl_complex_mul(gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0.0,zeta)),0.5*sin(alpha)),gsl_complex_sub(gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0.0,theta/2)),sqrt(Kup)),gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0.0,-theta/2)),sqrt(Kdown))))
	#a22
	a22=gsl_complex_add(gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0,theta/2)),0.5*sqrt(Kup)*(1-cos(alpha))),gsl_complex_mul_real(gsl_complex_exp(gsl_complex_rect(0,-theta/2)),0.5*sqrt(Kdown)*(1+cos(alpha))))
	return Els_to_Mat(a11,a12,a21,a22)

cdef gsl_matrix_complex * R_22(SI_CI Sys) nogil:
	cdef double R_up,R_down
	R_up,R_down=Sys.R_up,Sys.R_down
	return M(Sys,R_up,R_down)

cdef gsl_matrix_complex * T_21(SI_CI Sys) nogil:
	cdef double T_up,T_down
	T_up,T_down=Sys.T_up,Sys.T_down
	return M(Sys,T_up,T_down)

############################################
## Python wrapper for scattering matrices ##
############################################
cpdef np.ndarray[DTYPE_t, ndim=2] get_R_22(SI_CI Sys):
	cdef gsl_matrix_complex * m
	m=R_22(Sys)
	cdef np.ndarray[DTYPE_t, ndim=2] res
	res=GSL_to_Numpy_mat(m)
	gsl_matrix_complex_free(m)
	return res

cpdef np.ndarray[DTYPE_t, ndim=2] get_T_21(SI_CI Sys):
	cdef gsl_matrix_complex * m
	m=T_21(Sys)
	cdef np.ndarray[DTYPE_t, ndim=2] res
	res=GSL_to_Numpy_mat(m)
	gsl_matrix_complex_free(m)
	return res

#Now we define the function for calculating the initial condition for the outgoing distribution function x0 -- x0_i
cdef gsl_matrix_complex * x0_i(SI_CI Sys,double x,gsl_complex t_up,gsl_complex t_down) nogil:
	cdef gsl_matrix_complex *T21,*R22,*x1,*x1_tilde,*x2,*gamma_r
	T21,R22=T_21(Sys),R_22(Sys)
	#define x1
	x1=gsl_matrix_complex_calloc(2,2) #initialize a matrix with zero elements
	gsl_matrix_complex_set(x1,0,0,gsl_complex_rect(_x1_up(Sys,x),0))
	gsl_matrix_complex_set(x1,1,1,gsl_complex_rect(_x1_down(Sys,x),0))
	#define x1_tilde
	x1_tilde=gsl_matrix_complex_calloc(2,2) #initialize a matrix with zero elements
	gsl_matrix_complex_set(x1_tilde,0,0,gsl_complex_rect(_x1_up_tilde(Sys,x),0))
	gsl_matrix_complex_set(x1_tilde,1,1,gsl_complex_rect(_x1_down_tilde(Sys,x),0))
	#define x2
	x2=gsl_matrix_complex_calloc(2,2) #initialize a matrix with zero elements
	gsl_matrix_complex_set(x2,0,0,gsl_complex_rect(_x2_up(Sys,x,t_up),0))
	gsl_matrix_complex_set(x2,1,1,gsl_complex_rect(_x2_down(Sys,x,t_down),0))
	#define gamma_r
	gamma_r=gsl_matrix_complex_calloc(2,2) #initialize a matrix with zero elements
	gsl_matrix_complex_set(gamma_r,0,1,t_up)
	gsl_matrix_complex_set(gamma_r,1,0,gsl_complex_negative(t_down))
	#Now we use the GSL interface to BLAS to get matrix multiplication
	cdef gsl_complex alpha,beta
	alpha=gsl_complex_rect(1,0)
	beta=gsl_complex_rect(0,0)
	cdef gsl_matrix_complex *res1,*res2,*res3
	#calculating res1: T21*x1*T21^{\dagger}
	cdef gsl_matrix_complex *tmp1=gsl_matrix_complex_calloc(2,2) #first supplementary matrix
	res1=gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm(CblasNoTrans,CblasNoTrans,alpha,T21,x1,beta,tmp1)
	gsl_blas_zgemm(CblasNoTrans,CblasConjTrans,alpha,tmp1,T21,beta,res1)
	gsl_matrix_complex_set_zero(tmp1) #we reset the matrix tmp1 for later use
	#calculating res2: R22*x2*R22^{\dagger}
	res2=gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm(CblasNoTrans,CblasNoTrans,alpha,R22,x2,beta,tmp1)
	gsl_blas_zgemm(CblasNoTrans,CblasConjTrans,alpha,tmp1,R22,beta,res2)
	gsl_matrix_complex_set_zero(tmp1) #we reset the matrix tmp1 for later use
	#calculating res3: R22*gamma_r*T21_tilde*x1_tilde*T21_tilde^{\dagger}*gamma_r^{\dagger}*R22^{\dagger}
	cdef gsl_matrix_complex *tmp2=gsl_matrix_complex_calloc(2,2) #second supplementary matrix
	gsl_blas_zgemm(CblasConjTrans,CblasConjTrans,alpha,gamma_r,R22,beta,tmp1)
	gsl_blas_zgemm(CblasTrans,CblasNoTrans,alpha,T21,tmp1,beta,tmp2)
	gsl_matrix_complex_set_zero(tmp1) #we reset the matrix tmp1 for later use
	gsl_blas_zgemm(CblasNoTrans,CblasNoTrans,alpha,x1_tilde,tmp2,beta,tmp1)
	res3=gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm(CblasConjTrans,CblasNoTrans,alpha,tmp2,tmp1,beta,res3)
	gsl_matrix_complex_free(tmp1) #release the memory taken by tmp1
	gsl_matrix_complex_free(tmp2) #release the memory taken by tmp2
	#We collect the final result: res1+res2-res3
	cdef gsl_matrix_complex *res=gsl_matrix_complex_calloc(2,2)
	gsl_matrix_complex_add(res,res1)
	gsl_matrix_complex_add(res,res2)
	gsl_matrix_complex_sub(res,res3)
	#Now we release the memory allocated for various matrices, not yet freed: T21,R22,x1,x1_tilde,x2,gamma_r,res1,res2,res3
	gsl_matrix_complex_free(T21)
	gsl_matrix_complex_free(R22)
	gsl_matrix_complex_free(x1)
	gsl_matrix_complex_free(x1_tilde)
	gsl_matrix_complex_free(x2)
	gsl_matrix_complex_free(gamma_r)
	gsl_matrix_complex_free(res1)
	gsl_matrix_complex_free(res2)
	gsl_matrix_complex_free(res3)
	return res

#Python wrapper to x0_i
cpdef np.ndarray[DTYPE_t, ndim=2] get_x0_i(SI_CI Sys,double x,double complex t_up,double complex t_down):
	cdef gsl_complex tt_up,tt_down
	tt_up=gsl_complex_rect(t_up.real,t_up.imag)
	tt_down=gsl_complex_rect(t_down.real,t_down.imag)
	cdef gsl_matrix_complex *m
	m=x0_i(Sys,x,tt_up,tt_down)
	cdef np.ndarray[DTYPE_t, ndim=2] res
	res=GSL_to_Numpy_mat(m)
	gsl_matrix_complex_free(m)
	return res
	
#Now we define the 2 functions, which help us calculate F1-F4 and dF1-F4/dz. They are 'void', which means that they are used to fill in the elements of the provided externally C++ array of 4 gsl_complex as the last parameter
cdef void F1_F4(double z,double u,gsl_complex d_up,gsl_complex d_down,gsl_complex D_up,gsl_complex E_up,gsl_complex t_up,gsl_complex D_down,gsl_complex E_down,gsl_complex t_down,gsl_matrix_complex *P,gsl_complex detP,gsl_complex res[]) nogil:
	#First we get the elements of a P-matrix
	cdef gsl_complex P00,P01,P10,P11
	P00=gsl_matrix_complex_get(P,0,0)
	P01=gsl_matrix_complex_get(P,0,1)
	P10=gsl_matrix_complex_get(P,1,0)
	P11=gsl_matrix_complex_get(P,1,1)
	#Now we find V_up,V_down and W_up,W_down
	cdef gsl_complex V_up,V_down,W_up,W_down
	#V_up
	V_up=gsl_complex_exp(gsl_complex_negative(gsl_complex_mul_real(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_up,2),gsl_complex_pow_real(E_up,2))),z/u)))
	#V_down
	V_down=gsl_complex_exp(gsl_complex_negative(gsl_complex_mul_real(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_down,2),gsl_complex_pow_real(E_down,2))),z/u)))
	#W_up
	W_up=gsl_complex_div(gsl_complex_mul(gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(V_up,2)),1),t_up),gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_up,2)),1))
	#W_down
	W_down=gsl_complex_div(gsl_complex_mul(gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(V_down,2)),1),t_down),gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_down,2)),1))
	#Next we calculate detK
	cdef gsl_complex detK
	detK=gsl_complex_add_real(gsl_complex_add(gsl_complex_sub(gsl_complex_mul(W_down,P10),gsl_complex_mul(W_up,P01)),gsl_complex_mul(gsl_complex_mul(W_up,W_down),detP)),1.0)
	#Next we get the elements of a dGamma matrix
	cdef gsl_complex dGamma_up_down,dGamma_down_up,dGamma_up_up,dGamma_down_down
	#dGamma_up_down
	dGamma_up_down=gsl_complex_div(gsl_complex_mul(gsl_complex_sub(P01,gsl_complex_mul(W_down,detP)),gsl_complex_pow_real(V_up,2)),detK)
	#dGamma_down_up
	dGamma_down_up=gsl_complex_div(gsl_complex_mul(gsl_complex_add(P10,gsl_complex_mul(W_up,detP)),gsl_complex_pow_real(V_down,2)),detK)
	#dGamma_up_up
	dGamma_up_up=gsl_complex_div(gsl_complex_mul(gsl_complex_mul(P00,V_up),V_down),detK)
	#dGamma_down_down
	dGamma_down_down=gsl_complex_div(gsl_complex_mul(gsl_complex_mul(P11,V_up),V_down),detK)
	#Now as we have all these functions, we can define the F1-F4 values
	cdef gsl_complex F1,F2,F3,F4
	#F1
	F1=gsl_complex_div_real(gsl_complex_sub(d_up,gsl_complex_mul(D_up,dGamma_up_down)),u)
	#F2
	F2=gsl_complex_div_real(gsl_complex_add(d_down,gsl_complex_mul(D_down,dGamma_down_up)),u)
	#F3
	F3=gsl_complex_div_real(gsl_complex_mul(D_down,dGamma_up_up),u)
	#F4
	F4=gsl_complex_div_real(gsl_complex_mul(D_up,dGamma_down_down),u)
	#Finally we assign the found values to the externally provided C++ array of 4 gsl_complex
	res[0],res[1],res[2],res[3]=F1,F2,F3,F4

cdef void dF1_F4_dz(double z,double u,gsl_complex d_up,gsl_complex d_down,gsl_complex D_up,gsl_complex E_up,gsl_complex t_up,gsl_complex D_down,gsl_complex E_down,gsl_complex t_down,gsl_matrix_complex *P,gsl_complex detP,gsl_complex d_dz_res[]) nogil:
	#First we get the elements of a P-matrix
	cdef gsl_complex P00,P01,P10,P11
	P00=gsl_matrix_complex_get(P,0,0)
	P01=gsl_matrix_complex_get(P,0,1)
	P10=gsl_matrix_complex_get(P,1,0)
	P11=gsl_matrix_complex_get(P,1,1)
	#Next we define the elementary building blocks
	cdef gsl_complex f_up,f_down,V_up,dV_up,V_down,dV_down,W_up,dW_up,W_down,dW_down
	#f_up,f_down: it differs form eq.(177) by the minus sign
	f_up=gsl_complex_negative(gsl_complex_div_real(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_up,2),gsl_complex_pow_real(E_up,2))),u))
	f_down=gsl_complex_negative(gsl_complex_div_real(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_down,2),gsl_complex_pow_real(E_down,2))),u))
	#V_up,dV_up,V_down,dV_down
	V_up=gsl_complex_exp(gsl_complex_mul_real(f_up,z))
	dV_up=gsl_complex_mul(f_up,V_up)
	V_down=gsl_complex_exp(gsl_complex_mul_real(f_down,z))
	dV_down=gsl_complex_mul(f_down,V_down)
	#W_up,dW_up,W_down,dW_down
	tmp_up=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_up,2)),1.0)
	tmp_down=gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(t_down,2)),1.0)
	W_up=gsl_complex_mul(gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(V_up,2)),1.0),tmp_up)
	dW_up=gsl_complex_negative(gsl_complex_mul_real(gsl_complex_mul(gsl_complex_mul(V_up,dV_up),tmp_up),2.0))
	W_down=gsl_complex_mul(gsl_complex_add_real(gsl_complex_negative(gsl_complex_pow_real(V_down,2)),1.0),tmp_down)
	dW_down=gsl_complex_negative(gsl_complex_mul_real(gsl_complex_mul(gsl_complex_mul(V_down,dV_down),tmp_down),2.0))
	#define two supplementary values
	cdef gsl_complex M_up,M_down
	M_up=gsl_complex_sub(P01,gsl_complex_mul(W_down,detP))
	M_down=gsl_complex_add(P10,gsl_complex_mul(W_up,detP))
	#Now we define the detK and its derivative in z
	cdef gsl_complex detK,ddetK
	detK=gsl_complex_add_real(gsl_complex_add(gsl_complex_sub(gsl_complex_mul(W_down,P10),gsl_complex_mul(W_up,P01)),gsl_complex_mul(gsl_complex_mul(W_up,W_down),detP)),1.0)
	ddetK=gsl_complex_sub(gsl_complex_mul(dW_down,M_down),gsl_complex_mul(dW_up,M_up))
	#Next we define the derivatives of the dGamma function
	cdef gsl_complex ddGamma_up_down,ddGamma_down_up,ddGamma_up_up,ddGamma_down_down
	#ddGamma_up_down
	ddGamma_up_down=gsl_complex_div(gsl_complex_mul(gsl_complex_sub(gsl_complex_sub(gsl_complex_mul_real(gsl_complex_mul(M_up,dV_up),2.0),gsl_complex_mul(gsl_complex_mul(dW_down,detP),V_up)),gsl_complex_div(gsl_complex_mul(gsl_complex_mul(M_up,ddetK),V_up),detK)),V_up),detK)
	#ddGamma_down_up
	ddGamma_down_up=gsl_complex_div(gsl_complex_mul(gsl_complex_sub(gsl_complex_add(gsl_complex_mul_real(gsl_complex_mul(M_down,dV_down),2.0),gsl_complex_mul(gsl_complex_mul(dW_up,detP),V_down)),gsl_complex_div(gsl_complex_mul(gsl_complex_mul(M_down,ddetK),V_down),detK)),V_down),detK)
	#ddGamma_up_up
	ddGamma_up_up=gsl_complex_div(gsl_complex_mul(gsl_complex_sub(gsl_complex_add(gsl_complex_mul(V_up,dV_down),gsl_complex_mul(V_down,dV_up)),gsl_complex_div(gsl_complex_mul(gsl_complex_mul(V_up,V_down),ddetK),detK)),P00),detK)
	#ddGamma_down_down
	ddGamma_down_down=gsl_complex_div(gsl_complex_mul(gsl_complex_sub(gsl_complex_add(gsl_complex_mul(V_up,dV_down),gsl_complex_mul(V_down,dV_up)),gsl_complex_div(gsl_complex_mul(gsl_complex_mul(V_up,V_down),ddetK),detK)),P11),detK)
	#Finally we find the derivatives dF1-dF4 in z
	cdef gsl_complex dF1,dF2,dF3,dF4
	dF1=gsl_complex_negative(gsl_complex_div_real(gsl_complex_mul(D_up,ddGamma_up_down),u))
	dF2=gsl_complex_div_real(gsl_complex_mul(D_down,ddGamma_down_up),u)
	dF3=gsl_complex_div_real(gsl_complex_mul(D_down,ddGamma_up_up),u)
	dF4=gsl_complex_div_real(gsl_complex_mul(D_up,ddGamma_down_down),u)
	#In the end we just use the found values to set the elements of d_dz_res[]
	d_dz_res[0],d_dz_res[1],d_dz_res[2],d_dz_res[3]=dF1,dF2,dF3,dF4

#Now we have everything to define the rhs of the differential equation and its Jacobian. We need this to use gsl_odeiv_* functions to find the solution to our system of ODEs
cdef int func(double z, double y[], double f[], void *params) nogil:
	#We first dereference our parameters from the void pointer provided
	cdef my_params p = deref (<my_params *> params)
	cdef double u = p.u
	cdef gsl_complex d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,detP
	d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,detP=p.d_up,p.d_down,p.D_up,p.E_up,p.t_up,p.D_down,p.E_down,p.t_down,p.detP
	cdef gsl_matrix_complex *P = p.P
	#Now we calculate the values of F1-F4
	cdef gsl_complex resF1_F4[4]
	F1_F4(z,u,d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,P,detP,resF1_F4)
	#Now as we have recovered all our parameters, we set the rhs of our system of ODEs
	#f[0]
	f[0]=2*GSL_IMAG(resF1_F4[0])*y[0]+2*GSL_IMAG(resF1_F4[2])*y[1]-2*GSL_REAL(resF1_F4[2])*y[2]
	#f[1]
	f[1]=(GSL_IMAG(resF1_F4[0])+GSL_IMAG(resF1_F4[1]))*y[1]+(GSL_REAL(resF1_F4[0])-GSL_REAL(resF1_F4[1]))*y[2]-GSL_IMAG(resF1_F4[3])*y[0]+GSL_IMAG(resF1_F4[2])*y[3]
	#f[2]
	f[2]=(GSL_IMAG(resF1_F4[0])+GSL_IMAG(resF1_F4[1]))*y[2]-(GSL_REAL(resF1_F4[0])-GSL_REAL(resF1_F4[1]))*y[1]-GSL_REAL(resF1_F4[3])*y[0]-GSL_REAL(resF1_F4[2])*y[3]
	#f[3]
	f[3]=2*GSL_IMAG(resF1_F4[1])*y[3]-2*GSL_IMAG(resF1_F4[3])*y[1]-2*GSL_REAL(resF1_F4[3])*y[2]
	return GSL_SUCCESS

cdef int jac(double z, double y[], double *dfdy, double dfdt[], void *params) nogil:
	#We first dereference our parameters from the void pointer provided
	cdef my_params p = deref (<my_params *> params)
	cdef double u = p.u
	cdef gsl_complex d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,detP
	d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,detP=p.d_up,p.d_down,p.D_up,p.E_up,p.t_up,p.D_down,p.E_down,p.t_down,p.detP
	cdef gsl_matrix_complex *P = p.P
	#Now we calculate the values of F1-F4
	cdef gsl_complex resF1_F4[4]
	F1_F4(z,u,d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,P,detP,resF1_F4)
	#Now we calculate the values of dF1-dF4
	cdef gsl_complex resdF1_F4[4]
	dF1_F4_dz(z,u,d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,P,detP,resdF1_F4)
	#Now as we have it all we can easily define our Jacobian
	cdef gsl_matrix_view dfdy_mat
	dfdy_mat = gsl_matrix_view_array(dfdy, 4, 4)
	cdef gsl_matrix * m
	m = &dfdy_mat.matrix
	#First row
	gsl_matrix_set(m,0,0,2*GSL_IMAG(resF1_F4[0]))
	gsl_matrix_set(m,0,1,2*GSL_IMAG(resF1_F4[2]))
	gsl_matrix_set(m,0,2,-2*GSL_REAL(resF1_F4[2]))
	gsl_matrix_set(m,0,3,0.0)
	#Second row
	gsl_matrix_set(m,1,0,-GSL_IMAG(resF1_F4[3]))
	gsl_matrix_set(m,1,1,GSL_IMAG(resF1_F4[0])+GSL_IMAG(resF1_F4[1]))
	gsl_matrix_set(m,1,2,GSL_REAL(resF1_F4[0])-GSL_REAL(resF1_F4[1]))
	gsl_matrix_set(m,1,3,GSL_IMAG(resF1_F4[2]))
	#Third row
	gsl_matrix_set(m,2,0,-GSL_REAL(resF1_F4[3]))
	gsl_matrix_set(m,2,1,GSL_REAL(resF1_F4[1])-GSL_REAL(resF1_F4[0]))
	gsl_matrix_set(m,2,2,GSL_IMAG(resF1_F4[0])+GSL_IMAG(resF1_F4[1]))
	gsl_matrix_set(m,2,3,-GSL_REAL(resF1_F4[2]))
	#Fourth row
	gsl_matrix_set(m,3,0,0.0)
	gsl_matrix_set(m,3,1,-2*GSL_IMAG(resF1_F4[3]))
	gsl_matrix_set(m,3,2,-2*GSL_REAL(resF1_F4[3]))
	gsl_matrix_set(m,3,3,2*GSL_IMAG(resF1_F4[1]))
	#And finally the array dfdt
	#dfdt[0]
	dfdt[0]=2*GSL_IMAG(resdF1_F4[0])*y[0]+2*GSL_IMAG(resdF1_F4[2])*y[1]-2*GSL_REAL(resdF1_F4[2])*y[2]
	#dfdt[1]
	dfdt[1]=(GSL_IMAG(resdF1_F4[0])+GSL_IMAG(resdF1_F4[1]))*y[1]+(GSL_REAL(resdF1_F4[0])-GSL_REAL(resdF1_F4[1]))*y[2]-GSL_IMAG(resdF1_F4[3])*y[0]+GSL_IMAG(resdF1_F4[2])*y[3]
	#dfdt[2]
	dfdt[2]=(GSL_IMAG(resdF1_F4[0])+GSL_IMAG(resdF1_F4[1]))*y[2]-(GSL_REAL(resdF1_F4[0])-GSL_REAL(resdF1_F4[1]))*y[1]-GSL_REAL(resdF1_F4[3])*y[0]-GSL_REAL(resdF1_F4[2])*y[3]
	#dfdt[3]
	dfdt[3]=2*GSL_IMAG(resdF1_F4[1])*y[3]-2*GSL_IMAG(resdF1_F4[3])*y[1]-2*GSL_REAL(resdF1_F4[3])*y[2]
	return GSL_SUCCESS

#Now we can write a function which calculates x0
cdef gsl_matrix_complex * x0(SI_CI Sys,double u,double x,double z,gsl_complex d_up,gsl_complex d_down,gsl_complex D_up,gsl_complex E_up,gsl_complex t_up,gsl_complex D_down,gsl_complex E_down,gsl_complex t_down,gsl_matrix_complex *P,gsl_complex detP) nogil:
	#We first create a pointer to a set of parameters
	cdef my_params p
	p.u,p.d_up,p.d_down,p.D_up,p.E_up,p.t_up,p.D_down,p.E_down,p.t_down,p.P,p.detP=u,d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,P,detP
	cdef void *params = <void *> &p
	#Now we set up the system
	cdef gsl_odeiv2_system sys
	sys.function = &func
	sys.jacobian = &jac
	sys.dimension = 4
	sys.params = params
	#Set up the driver
	cdef gsl_odeiv2_driver * d
	#Note the las three parameters: (hi,atol,rtol), hi-initial step size, atol/rtol-absolute/relative tolerance
	d = gsl_odeiv2_driver_alloc_yp_new(&sys,gsl_odeiv2_step_msbdf,1e-3, 1e-5, 1e-4)
	#Set up the initital condition
	cdef double zz=0.0
	cdef double y[4]
	cdef gsl_matrix_complex * m
	m=x0_i(Sys,x,t_up,t_down)
	cdef gsl_complex a00,a11,a01
	a00=gsl_matrix_complex_get(m,0,0)
	a01=gsl_matrix_complex_get(m,0,1)
	a11=gsl_matrix_complex_get(m,1,1)
	y[0],y[1],y[2],y[3]=GSL_REAL(a00),GSL_REAL(a01),GSL_IMAG(a01),GSL_REAL(a11)
	#Now we apply our driver to find the solution y[i] i=1,4 at z
	cdef int status
	status = gsl_odeiv2_driver_apply(d, &zz, z, y)
	if (status != GSL_SUCCESS):
		printf("error, return value=%d\n", status)
	#Now as the calculation ended successfully we use the result to restore the x0 matrix
	cdef gsl_matrix_complex * res = gsl_matrix_complex_calloc(2,2)
	gsl_matrix_complex_set(res,0,0,gsl_complex_rect(y[0],0.0))
	gsl_matrix_complex_set(res,0,1,gsl_complex_rect(y[1],y[2]))
	gsl_matrix_complex_set(res,1,0,gsl_complex_rect(y[1],-y[2]))
	gsl_matrix_complex_set(res,1,1,gsl_complex_rect(y[3],0.0))
	#Before we return the result we have to release memory to avoid leakage
	gsl_matrix_complex_free(m)
	gsl_odeiv2_driver_free(d)
	#Now we are safe to return the result
	return res

#And we write down the analogous function, but with fewer arguments
cdef gsl_matrix_complex * x0_full(SI_CI Sys,double u,double x,double z) nogil:
	cdef gsl_complex d_up,d_down,D_up,D_down,E_up,E_down,t_up,t_down,detP
	cdef gsl_matrix_complex * P
	cdef gsl_complex x_bar_up,x_bar_down
	x_bar_up=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	x_bar_down=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	E_up,E_down=_Sigma(Sys,x_bar_up),_Sigma(Sys,x_bar_down)
	D_up,D_down=_Delta(Sys,x_bar_up),_Delta(Sys,x_bar_down)
	t_up,t_down=_gamma(D_up,E_up),_gamma(D_down,E_down)
	P=_P_matrix(Sys,t_up,t_down)
	detP=det(P)
	d_up=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_up,2),gsl_complex_pow_real(E_up,2))),1.0))
	d_down=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_down,2),gsl_complex_pow_real(E_down,2))),1.0))
	return x0(Sys,u,x,z,d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,P,detP)

cpdef np.ndarray[DTYPE_t, ndim=2] find_x0(SI_CI Sys,double u,double x,double z):
	cdef gsl_matrix_complex *m
	m=x0_full(Sys,u,x,z)
	cdef np.ndarray[DTYPE_t, ndim=2] res
	res=GSL_to_Numpy_mat(m)
	gsl_matrix_complex_free(m)
	return res

##############################################################################################################
## Now we are going to define the integrands for the anomalous part of magnetization M_ax,y,z_u_x and Q_u_x ##
##############################################################################################################

#M_ax_u_x
cdef double _M_ax_u_x(SI_CI Sys,double u, double x, double z) nogil:
	cdef gsl_complex d_up,d_down,D_up,D_down,E_up,E_down,t_up,t_down,detP,V_up,V_down,valDD
	cdef gsl_complex Gr_out_up_up,Gr_out_up_down,Gr_out_down_up,Gr_out_down_down,Gr_out_tilde_up_down,Gr_out_tilde_down_up
	#calculate X0 and X0_tilde
	cdef gsl_complex x_bar_up,x_bar_down
	cdef gsl_matrix_complex *P,*G_r_out,*G_r_out_tilde,*G_r_in,*G_r_in_tilde,*X0,*X0_tilde,*gamma_r,*gamma_r_tilde
	x_bar_up=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	x_bar_down=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	E_up,E_down=_Sigma(Sys,x_bar_up),_Sigma(Sys,x_bar_down)
	D_up,D_down=_Delta(Sys,x_bar_up),_Delta(Sys,x_bar_down)
	t_up,t_down=_gamma(D_up,E_up),_gamma(D_down,E_down)
	P=_P_matrix(Sys,t_up,t_down)
	detP=det(P)
	d_up=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_up,2),gsl_complex_pow_real(E_up,2))),1.0))
	d_down=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_down,2),gsl_complex_pow_real(E_down,2))),1.0))
	V_up,V_down=_V(D_up,E_up,u,z),_V(D_down,E_down,u,z)
	valDD=_DD(t_up,t_down,P,detP)
	#Elements of G_r_out matrices
	Gr_out_up_up=_G_r_out_up_up(t_up,t_down,valDD,V_up,P,detP)
	Gr_out_up_down=_G_r_out_up_down(t_down,V_up,V_down,valDD,P)
	Gr_out_down_up=_G_r_out_down_up(t_up,V_up,V_down,valDD,P)
	Gr_out_down_down=_G_r_out_down_down(t_up,t_down,valDD,V_down,P,detP)
	Gr_out_tilde_up_down=_G_r_out_tilde_up_down(t_down,V_up,V_down,valDD,P)
	Gr_out_tilde_down_up=_G_r_out_tilde_down_up(t_up,V_up,V_down,valDD,P)
	#Now we can define all the GSL matrices
	X0=x0(Sys,u,x,z,d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,P,detP) #X0(E,u,z)
	X0_tilde=x0_full(Sys,u,-x,z)
	gsl_matrix_complex_transpose(X0_tilde) #the X0_tilde is the x0_full(Sys,u,-x,z).T
	#G_r_out, G_r_out_tilde, G_r_in, G_r_in_tilde
	G_r_out=Els_to_Mat(Gr_out_up_up,Gr_out_up_down,Gr_out_down_up,Gr_out_down_down)
	G_r_out_tilde=Els_to_Mat(Gr_out_down_down,Gr_out_tilde_up_down,Gr_out_tilde_down_up,Gr_out_up_up)
	G_r_in=Els_to_Mat(Gr_out_up_up,gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,-2*Sys.zeta)),Gr_out_down_up),gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,2*Sys.zeta)),Gr_out_up_down),Gr_out_down_down)
	G_r_in_tilde=Els_to_Mat(Gr_out_down_down,gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,2*Sys.zeta)),Gr_out_tilde_down_up),gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,-2*Sys.zeta)),Gr_out_tilde_up_down),Gr_out_up_up)
	#gamma_r,gamma_r_tilde
	gamma_r=gsl_matrix_complex_calloc(2,2)
	gamma_r_tilde=gsl_matrix_complex_calloc(2,2)
	gsl_matrix_complex_set(gamma_r,0,1,t_up)
	gsl_matrix_complex_set(gamma_r,1,0,gsl_complex_negative(t_down))
	gsl_matrix_complex_set(gamma_r_tilde,0,1,gsl_complex_negative(t_down))
	gsl_matrix_complex_set(gamma_r_tilde,1,0,t_up)
	#Now as we have all matrices defined we construct Chi_out,Chi_out_tilde,Chi_in,Chi_in_tilde
	cdef gsl_complex alpha,beta #set alpha and beta constants for zgemm routine from BLAS
	alpha=gsl_complex_rect(1,0)
	beta=gsl_complex_rect(0,0)
	#Chi_out
	cdef gsl_matrix_complex * tmp1 = gsl_matrix_complex_calloc(2,2) #supplementary matrix
	cdef gsl_matrix_complex * Chi_out = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_out,X0,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp1,G_r_out,beta,Chi_out)
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	#Chi_out_tilde
	cdef gsl_matrix_complex * tmp2 = gsl_matrix_complex_calloc(2,2) #another supplementary matrix
	cdef gsl_matrix_complex * Chi_out_tilde = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_out_tilde,gamma_r_tilde,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,tmp1,X0,beta,tmp2)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp2,tmp1,beta,Chi_out_tilde)
	gsl_matrix_complex_scale(Chi_out_tilde,gsl_complex_rect(-1.0,0.0))
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	gsl_matrix_complex_set_zero(tmp2) #reset another supplementary function for later use
	#Chi_in
	cdef gsl_matrix_complex * Chi_in = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_in,gamma_r,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,tmp1,X0_tilde,beta,tmp2)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp2,tmp1,beta,Chi_in)
	gsl_matrix_complex_scale(Chi_in,gsl_complex_rect(-1.0,0.0))
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	gsl_matrix_complex_free(tmp2) #delete the second supplementary function since it is not anymore  needed
	#Chi_in_tilde
	cdef gsl_matrix_complex * Chi_in_tilde = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_in_tilde,X0_tilde,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp1,G_r_in_tilde,beta,Chi_in_tilde)
	gsl_matrix_complex_free(tmp1) #delete the first supplementary function since it is not anymore  needed
	#Now we have all 4 Chi matrices and are ready to return the final value. Note that we include the 1/2 in (168)
	cdef gsl_complex Chi_out01,Chi_out_tilde01,Chi_in01,Chi_in_tilde01
	Chi_out01=gsl_matrix_complex_get(Chi_out,0,1)
	Chi_out_tilde01=gsl_matrix_complex_get(Chi_out_tilde,0,1)
	Chi_in01=gsl_matrix_complex_get(Chi_in,0,1)
	Chi_in_tilde01=gsl_matrix_complex_get(Chi_in_tilde,0,1)
	cdef double res
	res=GSL_REAL(Chi_out01)+GSL_REAL(Chi_out_tilde01)+GSL_REAL(Chi_in01)+GSL_REAL(Chi_in_tilde01)
	#Finally we release the memory taken by all matrices defined inside this function
	gsl_matrix_complex_free(P)
	gsl_matrix_complex_free(G_r_out)
	gsl_matrix_complex_free(G_r_out_tilde)
	gsl_matrix_complex_free(G_r_in)
	gsl_matrix_complex_free(G_r_in_tilde)
	gsl_matrix_complex_free(X0)
	gsl_matrix_complex_free(X0_tilde)
	gsl_matrix_complex_free(gamma_r)
	gsl_matrix_complex_free(gamma_r_tilde)
	gsl_matrix_complex_free(Chi_out)
	gsl_matrix_complex_free(Chi_out_tilde)
	gsl_matrix_complex_free(Chi_in)
	gsl_matrix_complex_free(Chi_in_tilde)
	#and return the result
	return 0.5*res
	
#M_ay_u_x
cdef double _M_ay_u_x(SI_CI Sys,double u, double x, double z) nogil:
	cdef gsl_complex d_up,d_down,D_up,D_down,E_up,E_down,t_up,t_down,detP,V_up,V_down,valDD
	cdef gsl_complex Gr_out_up_up,Gr_out_up_down,Gr_out_down_up,Gr_out_down_down,Gr_out_tilde_up_down,Gr_out_tilde_down_up
	#calculate X0 and X0_tilde
	cdef gsl_complex x_bar_up,x_bar_down
	cdef gsl_matrix_complex *P,*G_r_out,*G_r_out_tilde,*G_r_in,*G_r_in_tilde,*X0,*X0_tilde,*gamma_r,*gamma_r_tilde
	x_bar_up=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	x_bar_down=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	E_up,E_down=_Sigma(Sys,x_bar_up),_Sigma(Sys,x_bar_down)
	D_up,D_down=_Delta(Sys,x_bar_up),_Delta(Sys,x_bar_down)
	t_up,t_down=_gamma(D_up,E_up),_gamma(D_down,E_down)
	P=_P_matrix(Sys,t_up,t_down)
	detP=det(P)
	d_up=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_up,2),gsl_complex_pow_real(E_up,2))),1.0))
	d_down=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_down,2),gsl_complex_pow_real(E_down,2))),1.0))
	V_up,V_down=_V(D_up,E_up,u,z),_V(D_down,E_down,u,z)
	valDD=_DD(t_up,t_down,P,detP)
	#Elements of G_r_out matrices
	Gr_out_up_up=_G_r_out_up_up(t_up,t_down,valDD,V_up,P,detP)
	Gr_out_up_down=_G_r_out_up_down(t_down,V_up,V_down,valDD,P)
	Gr_out_down_up=_G_r_out_down_up(t_up,V_up,V_down,valDD,P)
	Gr_out_down_down=_G_r_out_down_down(t_up,t_down,valDD,V_down,P,detP)
	Gr_out_tilde_up_down=_G_r_out_tilde_up_down(t_down,V_up,V_down,valDD,P)
	Gr_out_tilde_down_up=_G_r_out_tilde_down_up(t_up,V_up,V_down,valDD,P)
	#Now we can define all the GSL matrices
	X0=x0(Sys,u,x,z,d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,P,detP) #X0(E,u,z)
	X0_tilde=x0_full(Sys,u,-x,z)
	gsl_matrix_complex_transpose(X0_tilde) #the X0_tilde is the x0_full(Sys,u,-x,z).T
	#G_r_out, G_r_out_tilde, G_r_in, G_r_in_tilde
	G_r_out=Els_to_Mat(Gr_out_up_up,Gr_out_up_down,Gr_out_down_up,Gr_out_down_down)
	G_r_out_tilde=Els_to_Mat(Gr_out_down_down,Gr_out_tilde_up_down,Gr_out_tilde_down_up,Gr_out_up_up)
	G_r_in=Els_to_Mat(Gr_out_up_up,gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,-2*Sys.zeta)),Gr_out_down_up),gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,2*Sys.zeta)),Gr_out_up_down),Gr_out_down_down)
	G_r_in_tilde=Els_to_Mat(Gr_out_down_down,gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,2*Sys.zeta)),Gr_out_tilde_down_up),gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,-2*Sys.zeta)),Gr_out_tilde_up_down),Gr_out_up_up)
	#gamma_r,gamma_r_tilde
	gamma_r=gsl_matrix_complex_calloc(2,2)
	gamma_r_tilde=gsl_matrix_complex_calloc(2,2)
	gsl_matrix_complex_set(gamma_r,0,1,t_up)
	gsl_matrix_complex_set(gamma_r,1,0,gsl_complex_negative(t_down))
	gsl_matrix_complex_set(gamma_r_tilde,0,1,gsl_complex_negative(t_down))
	gsl_matrix_complex_set(gamma_r_tilde,1,0,t_up)
	#Now as we have all matrices defined we construct Chi_out,Chi_out_tilde,Chi_in,Chi_in_tilde
	cdef gsl_complex alpha,beta #set alpha and beta constants for zgemm routine from BLAS
	alpha=gsl_complex_rect(1,0)
	beta=gsl_complex_rect(0,0)
	#Chi_out
	cdef gsl_matrix_complex * tmp1 = gsl_matrix_complex_calloc(2,2) #supplementary matrix
	cdef gsl_matrix_complex * Chi_out = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_out,X0,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp1,G_r_out,beta,Chi_out)
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	#Chi_out_tilde
	cdef gsl_matrix_complex * tmp2 = gsl_matrix_complex_calloc(2,2) #another supplementary matrix
	cdef gsl_matrix_complex * Chi_out_tilde = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_out_tilde,gamma_r_tilde,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,tmp1,X0,beta,tmp2)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp2,tmp1,beta,Chi_out_tilde)
	gsl_matrix_complex_scale(Chi_out_tilde,gsl_complex_rect(-1.0,0.0))
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	gsl_matrix_complex_set_zero(tmp2) #reset another supplementary function for later use
	#Chi_in
	cdef gsl_matrix_complex * Chi_in = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_in,gamma_r,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,tmp1,X0_tilde,beta,tmp2)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp2,tmp1,beta,Chi_in)
	gsl_matrix_complex_scale(Chi_in,gsl_complex_rect(-1.0,0.0))
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	gsl_matrix_complex_free(tmp2) #delete the second supplementary function since it is not anymore  needed
	#Chi_in_tilde
	cdef gsl_matrix_complex * Chi_in_tilde = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_in_tilde,X0_tilde,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp1,G_r_in_tilde,beta,Chi_in_tilde)
	gsl_matrix_complex_free(tmp1) #delete the first supplementary function since it is not anymore  needed
	#Now we have all 4 Chi matrices and are ready to return the final value. Note that we include the 1/2 in (168)
	cdef gsl_complex Chi_out10,Chi_out_tilde10,Chi_in10,Chi_in_tilde10
	Chi_out10=gsl_matrix_complex_get(Chi_out,1,0)
	Chi_out_tilde10=gsl_matrix_complex_get(Chi_out_tilde,1,0)
	Chi_in10=gsl_matrix_complex_get(Chi_in,1,0)
	Chi_in_tilde10=gsl_matrix_complex_get(Chi_in_tilde,1,0)
	cdef double res
	res=GSL_IMAG(Chi_out10)-GSL_IMAG(Chi_out_tilde10)+GSL_IMAG(Chi_in10)-GSL_IMAG(Chi_in_tilde10)
	#Finally we release the memory taken by all matrices defined inside this function
	gsl_matrix_complex_free(P)
	gsl_matrix_complex_free(G_r_out)
	gsl_matrix_complex_free(G_r_out_tilde)
	gsl_matrix_complex_free(G_r_in)
	gsl_matrix_complex_free(G_r_in_tilde)
	gsl_matrix_complex_free(X0)
	gsl_matrix_complex_free(X0_tilde)
	gsl_matrix_complex_free(gamma_r)
	gsl_matrix_complex_free(gamma_r_tilde)
	gsl_matrix_complex_free(Chi_out)
	gsl_matrix_complex_free(Chi_out_tilde)
	gsl_matrix_complex_free(Chi_in)
	gsl_matrix_complex_free(Chi_in_tilde)
	#and return the result
	return 0.5*res

#M_az_u_x
cdef double _M_az_u_x(SI_CI Sys,double u, double x, double z) nogil:
	cdef gsl_complex d_up,d_down,D_up,D_down,E_up,E_down,t_up,t_down,detP,V_up,V_down,valDD
	cdef gsl_complex Gr_out_up_up,Gr_out_up_down,Gr_out_down_up,Gr_out_down_down,Gr_out_tilde_up_down,Gr_out_tilde_down_up
	#calculate X0 and X0_tilde
	cdef gsl_complex x_bar_up,x_bar_down
	cdef gsl_matrix_complex *P,*G_r_out,*G_r_out_tilde,*G_r_in,*G_r_in_tilde,*X0,*X0_tilde,*gamma_r,*gamma_r_tilde
	x_bar_up=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	x_bar_down=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	E_up,E_down=_Sigma(Sys,x_bar_up),_Sigma(Sys,x_bar_down)
	D_up,D_down=_Delta(Sys,x_bar_up),_Delta(Sys,x_bar_down)
	t_up,t_down=_gamma(D_up,E_up),_gamma(D_down,E_down)
	P=_P_matrix(Sys,t_up,t_down)
	detP=det(P)
	d_up=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_up,2),gsl_complex_pow_real(E_up,2))),1.0))
	d_down=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_down,2),gsl_complex_pow_real(E_down,2))),1.0))
	V_up,V_down=_V(D_up,E_up,u,z),_V(D_down,E_down,u,z)
	valDD=_DD(t_up,t_down,P,detP)
	#Elements of G_r_out matrices
	Gr_out_up_up=_G_r_out_up_up(t_up,t_down,valDD,V_up,P,detP)
	Gr_out_up_down=_G_r_out_up_down(t_down,V_up,V_down,valDD,P)
	Gr_out_down_up=_G_r_out_down_up(t_up,V_up,V_down,valDD,P)
	Gr_out_down_down=_G_r_out_down_down(t_up,t_down,valDD,V_down,P,detP)
	Gr_out_tilde_up_down=_G_r_out_tilde_up_down(t_down,V_up,V_down,valDD,P)
	Gr_out_tilde_down_up=_G_r_out_tilde_down_up(t_up,V_up,V_down,valDD,P)
	#Now we can define all the GSL matrices
	X0=x0(Sys,u,x,z,d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,P,detP) #X0(E,u,z)
	X0_tilde=x0_full(Sys,u,-x,z)
	gsl_matrix_complex_transpose(X0_tilde) #the X0_tilde is the x0_full(Sys,u,-x,z).T
	#G_r_out, G_r_out_tilde, G_r_in, G_r_in_tilde
	G_r_out=Els_to_Mat(Gr_out_up_up,Gr_out_up_down,Gr_out_down_up,Gr_out_down_down)
	G_r_out_tilde=Els_to_Mat(Gr_out_down_down,Gr_out_tilde_up_down,Gr_out_tilde_down_up,Gr_out_up_up)
	G_r_in=Els_to_Mat(Gr_out_up_up,gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,-2*Sys.zeta)),Gr_out_down_up),gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,2*Sys.zeta)),Gr_out_up_down),Gr_out_down_down)
	G_r_in_tilde=Els_to_Mat(Gr_out_down_down,gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,2*Sys.zeta)),Gr_out_tilde_down_up),gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,-2*Sys.zeta)),Gr_out_tilde_up_down),Gr_out_up_up)
	#gamma_r,gamma_r_tilde
	gamma_r=gsl_matrix_complex_calloc(2,2)
	gamma_r_tilde=gsl_matrix_complex_calloc(2,2)
	gsl_matrix_complex_set(gamma_r,0,1,t_up)
	gsl_matrix_complex_set(gamma_r,1,0,gsl_complex_negative(t_down))
	gsl_matrix_complex_set(gamma_r_tilde,0,1,gsl_complex_negative(t_down))
	gsl_matrix_complex_set(gamma_r_tilde,1,0,t_up)
	#Now as we have all matrices defined we construct Chi_out,Chi_out_tilde,Chi_in,Chi_in_tilde
	cdef gsl_complex alpha,beta #set alpha and beta constants for zgemm routine from BLAS
	alpha=gsl_complex_rect(1,0)
	beta=gsl_complex_rect(0,0)
	#Chi_out
	cdef gsl_matrix_complex * tmp1 = gsl_matrix_complex_calloc(2,2) #supplementary matrix
	cdef gsl_matrix_complex * Chi_out = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_out,X0,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp1,G_r_out,beta,Chi_out)
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	#Chi_out_tilde
	cdef gsl_matrix_complex * tmp2 = gsl_matrix_complex_calloc(2,2) #another supplementary matrix
	cdef gsl_matrix_complex * Chi_out_tilde = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_out_tilde,gamma_r_tilde,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,tmp1,X0,beta,tmp2)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp2,tmp1,beta,Chi_out_tilde)
	gsl_matrix_complex_scale(Chi_out_tilde,gsl_complex_rect(-1.0,0.0))
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	gsl_matrix_complex_set_zero(tmp2) #reset another supplementary function for later use
	#Chi_in
	cdef gsl_matrix_complex * Chi_in = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_in,gamma_r,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,tmp1,X0_tilde,beta,tmp2)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp2,tmp1,beta,Chi_in)
	gsl_matrix_complex_scale(Chi_in,gsl_complex_rect(-1.0,0.0))
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	gsl_matrix_complex_free(tmp2) #delete the second supplementary function since it is not anymore  needed
	#Chi_in_tilde
	cdef gsl_matrix_complex * Chi_in_tilde = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_in_tilde,X0_tilde,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp1,G_r_in_tilde,beta,Chi_in_tilde)
	gsl_matrix_complex_free(tmp1) #delete the first supplementary function since it is not anymore  needed
	#Now we have all 4 Chi matrices and are ready to return the final value. Note that we include the 1/2 in (168)
	cdef gsl_complex Chi_out00,Chi_out_tilde00,Chi_in00,Chi_in_tilde00,Chi_out11,Chi_out_tilde11,Chi_in11,Chi_in_tilde11
	Chi_out00=gsl_matrix_complex_get(Chi_out,0,0)
	Chi_out_tilde00=gsl_matrix_complex_get(Chi_out_tilde,0,0)
	Chi_in00=gsl_matrix_complex_get(Chi_in,0,0)
	Chi_in_tilde00=gsl_matrix_complex_get(Chi_in_tilde,0,0)
	Chi_out11=gsl_matrix_complex_get(Chi_out,1,1)
	Chi_out_tilde11=gsl_matrix_complex_get(Chi_out_tilde,1,1)
	Chi_in11=gsl_matrix_complex_get(Chi_in,1,1)
	Chi_in_tilde11=gsl_matrix_complex_get(Chi_in_tilde,1,1)
	cdef double res
	res=GSL_REAL(Chi_out00)-GSL_REAL(Chi_out11)+GSL_REAL(Chi_out_tilde00)-GSL_REAL(Chi_out_tilde11)+GSL_REAL(Chi_in00)-GSL_REAL(Chi_in11)+GSL_REAL(Chi_in_tilde00)-GSL_REAL(Chi_in_tilde11)
	#Finally we release the memory taken by all matrices defined inside this function
	gsl_matrix_complex_free(P)
	gsl_matrix_complex_free(G_r_out)
	gsl_matrix_complex_free(G_r_out_tilde)
	gsl_matrix_complex_free(G_r_in)
	gsl_matrix_complex_free(G_r_in_tilde)
	gsl_matrix_complex_free(X0)
	gsl_matrix_complex_free(X0_tilde)
	gsl_matrix_complex_free(gamma_r)
	gsl_matrix_complex_free(gamma_r_tilde)
	gsl_matrix_complex_free(Chi_out)
	gsl_matrix_complex_free(Chi_out_tilde)
	gsl_matrix_complex_free(Chi_in)
	gsl_matrix_complex_free(Chi_in_tilde)
	#and return the result
	return 0.25*res

#Q_u_x
cdef double _Q_u_x(SI_CI Sys,double u, double x, double z) nogil:
	cdef gsl_complex d_up,d_down,D_up,D_down,E_up,E_down,t_up,t_down,detP,V_up,V_down,valDD
	cdef gsl_complex Gr_out_up_up,Gr_out_up_down,Gr_out_down_up,Gr_out_down_down,Gr_out_tilde_up_down,Gr_out_tilde_down_up
	#calculate X0 and X0_tilde
	cdef gsl_complex x_bar_up,x_bar_down
	cdef gsl_matrix_complex *P,*G_r_out,*G_r_out_tilde,*G_r_in,*G_r_in_tilde,*X0,*X0_tilde,*gamma_r,*gamma_r_tilde
	x_bar_up=gsl_complex_rect(x-Sys.Ez,Sys.reg_c)
	x_bar_down=gsl_complex_rect(x+Sys.Ez,Sys.reg_c)
	E_up,E_down=_Sigma(Sys,x_bar_up),_Sigma(Sys,x_bar_down)
	D_up,D_down=_Delta(Sys,x_bar_up),_Delta(Sys,x_bar_down)
	t_up,t_down=_gamma(D_up,E_up),_gamma(D_down,E_down)
	P=_P_matrix(Sys,t_up,t_down)
	detP=det(P)
	d_up=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_up,2),gsl_complex_pow_real(E_up,2))),1.0))
	d_down=gsl_complex_sub(gsl_complex_rect(x,Sys.reg_c),gsl_complex_mul_imag(gsl_complex_sqrt(gsl_complex_sub(gsl_complex_pow_real(D_down,2),gsl_complex_pow_real(E_down,2))),1.0))
	V_up,V_down=_V(D_up,E_up,u,z),_V(D_down,E_down,u,z)
	valDD=_DD(t_up,t_down,P,detP)
	#Elements of G_r_out matrices
	Gr_out_up_up=_G_r_out_up_up(t_up,t_down,valDD,V_up,P,detP)
	Gr_out_up_down=_G_r_out_up_down(t_down,V_up,V_down,valDD,P)
	Gr_out_down_up=_G_r_out_down_up(t_up,V_up,V_down,valDD,P)
	Gr_out_down_down=_G_r_out_down_down(t_up,t_down,valDD,V_down,P,detP)
	Gr_out_tilde_up_down=_G_r_out_tilde_up_down(t_down,V_up,V_down,valDD,P)
	Gr_out_tilde_down_up=_G_r_out_tilde_down_up(t_up,V_up,V_down,valDD,P)
	#Now we can define all the GSL matrices
	X0=x0(Sys,u,x,z,d_up,d_down,D_up,E_up,t_up,D_down,E_down,t_down,P,detP) #X0(E,u,z)
	X0_tilde=x0_full(Sys,u,-x,z)
	gsl_matrix_complex_transpose(X0_tilde) #the X0_tilde is the x0_full(Sys,u,-x,z).T
	#G_r_out, G_r_out_tilde, G_r_in, G_r_in_tilde
	G_r_out=Els_to_Mat(Gr_out_up_up,Gr_out_up_down,Gr_out_down_up,Gr_out_down_down)
	G_r_out_tilde=Els_to_Mat(Gr_out_down_down,Gr_out_tilde_up_down,Gr_out_tilde_down_up,Gr_out_up_up)
	G_r_in=Els_to_Mat(Gr_out_up_up,gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,-2*Sys.zeta)),Gr_out_down_up),gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,2*Sys.zeta)),Gr_out_up_down),Gr_out_down_down)
	G_r_in_tilde=Els_to_Mat(Gr_out_down_down,gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,2*Sys.zeta)),Gr_out_tilde_down_up),gsl_complex_mul(gsl_complex_exp(gsl_complex_rect(0.0,-2*Sys.zeta)),Gr_out_tilde_up_down),Gr_out_up_up)
	#gamma_r,gamma_r_tilde
	gamma_r=gsl_matrix_complex_calloc(2,2)
	gamma_r_tilde=gsl_matrix_complex_calloc(2,2)
	gsl_matrix_complex_set(gamma_r,0,1,t_up)
	gsl_matrix_complex_set(gamma_r,1,0,gsl_complex_negative(t_down))
	gsl_matrix_complex_set(gamma_r_tilde,0,1,gsl_complex_negative(t_down))
	gsl_matrix_complex_set(gamma_r_tilde,1,0,t_up)
	#Now as we have all matrices defined we construct Chi_out,Chi_out_tilde,Chi_in,Chi_in_tilde
	cdef gsl_complex alpha,beta #set alpha and beta constants for zgemm routine from BLAS
	alpha=gsl_complex_rect(1,0)
	beta=gsl_complex_rect(0,0)
	#Chi_out
	cdef gsl_matrix_complex * tmp1 = gsl_matrix_complex_calloc(2,2) #supplementary matrix
	cdef gsl_matrix_complex * Chi_out = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_out,X0,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp1,G_r_out,beta,Chi_out)
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	#Chi_out_tilde
	cdef gsl_matrix_complex * tmp2 = gsl_matrix_complex_calloc(2,2) #another supplementary matrix
	cdef gsl_matrix_complex * Chi_out_tilde = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_out_tilde,gamma_r_tilde,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,tmp1,X0,beta,tmp2)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp2,tmp1,beta,Chi_out_tilde)
	gsl_matrix_complex_scale(Chi_out_tilde,gsl_complex_rect(-1.0,0.0))
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	gsl_matrix_complex_set_zero(tmp2) #reset another supplementary function for later use
	#Chi_in
	cdef gsl_matrix_complex * Chi_in = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_in,gamma_r,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,tmp1,X0_tilde,beta,tmp2)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp2,tmp1,beta,Chi_in)
	gsl_matrix_complex_scale(Chi_in,gsl_complex_rect(-1.0,0.0))
	gsl_matrix_complex_set_zero(tmp1) #reset the supplementary function for later use
	gsl_matrix_complex_free(tmp2) #delete the second supplementary function since it is not anymore  needed
	#Chi_in_tilde
	cdef gsl_matrix_complex * Chi_in_tilde = gsl_matrix_complex_calloc(2,2)
	gsl_blas_zgemm (CblasNoTrans,CblasNoTrans,alpha,G_r_in_tilde,X0_tilde,beta,tmp1)
	gsl_blas_zgemm (CblasNoTrans,CblasConjTrans,alpha,tmp1,G_r_in_tilde,beta,Chi_in_tilde)
	gsl_matrix_complex_free(tmp1) #delete the first supplementary function since it is not anymore  needed
	#Now we have all 4 Chi matrices and are ready to return the final value. Note that we include the 1/2 in (168)
	cdef gsl_complex Chi_out00,Chi_out_tilde00,Chi_in00,Chi_in_tilde00,Chi_out11,Chi_out_tilde11,Chi_in11,Chi_in_tilde11
	Chi_out00=gsl_matrix_complex_get(Chi_out,0,0)
	Chi_out_tilde00=gsl_matrix_complex_get(Chi_out_tilde,0,0)
	Chi_in00=gsl_matrix_complex_get(Chi_in,0,0)
	Chi_in_tilde00=gsl_matrix_complex_get(Chi_in_tilde,0,0)
	Chi_out11=gsl_matrix_complex_get(Chi_out,1,1)
	Chi_out_tilde11=gsl_matrix_complex_get(Chi_out_tilde,1,1)
	Chi_in11=gsl_matrix_complex_get(Chi_in,1,1)
	Chi_in_tilde11=gsl_matrix_complex_get(Chi_in_tilde,1,1)
	cdef double res
	res=GSL_REAL(Chi_out00)+GSL_REAL(Chi_out11)+GSL_REAL(Chi_out_tilde00)+GSL_REAL(Chi_out_tilde11)+GSL_REAL(Chi_in00)+GSL_REAL(Chi_in11)+GSL_REAL(Chi_in_tilde00)+GSL_REAL(Chi_in_tilde11)
	#Finally we release the memory taken by all matrices defined inside this function
	gsl_matrix_complex_free(P)
	gsl_matrix_complex_free(G_r_out)
	gsl_matrix_complex_free(G_r_out_tilde)
	gsl_matrix_complex_free(G_r_in)
	gsl_matrix_complex_free(G_r_in_tilde)
	gsl_matrix_complex_free(X0)
	gsl_matrix_complex_free(X0_tilde)
	gsl_matrix_complex_free(gamma_r)
	gsl_matrix_complex_free(gamma_r_tilde)
	gsl_matrix_complex_free(Chi_out)
	gsl_matrix_complex_free(Chi_out_tilde)
	gsl_matrix_complex_free(Chi_in)
	gsl_matrix_complex_free(Chi_in_tilde)
	#and return the result
	return 0.25*res

#####################################################
## Python wrappers to the integrands defined above ##
#####################################################
cpdef double M_ax_u_x(double u, double x, SI_CI Sys, double z):
	return _M_ax_u_x(Sys,u,x,z)

cpdef double M_ay_u_x(double u, double x, SI_CI Sys, double z):
	return _M_ay_u_x(Sys,u,x,z)
	
cpdef double M_az_u_x(double u, double x, SI_CI Sys, double z):
	return _M_az_u_x(Sys,u,x,z)

cpdef double Q_u_x(double u, double x, SI_CI Sys, double z):
	return _Q_u_x(Sys,u,x,z)
	
############################################################################################################
## Now we just have to integrate over (u,x) the previously defined functions to obtain the M_ax,y,z and Q ##
############################################################################################################

#NOTE: Since we pass as an argument a Python object of type 'SI_CI', we cannot use the CQUAD routine from GSL because in this case we sacrifice the speed on integration stage in favor of flexibility. Flexibility means that we can change the parameters 'on flight' without recompiling the .pyx file with a different set of parameters.

###############################
## Now we can integrate in u ##
###############################
cpdef double M_ax_x(double x, double u_cutoff, SI_CI Sys, double z):
	cdef double eabs,erel,res
	cdef int Nlim=1000
	eabs,erel=100*Sys.reg_c,1e-8
	res=0
	tmp=integrate.quad(M_ax_u_x,u_cutoff,1.0,args=(x,Sys,z),full_output=0,epsabs=eabs,epsrel=erel,limit=Nlim)
	res+=tmp[0]
	return res

cpdef double M_ay_x(double x, double u_cutoff, SI_CI Sys, double z):
	cdef double eabs,erel,res
	cdef int Nlim=1000
	eabs,erel=100*Sys.reg_c,1e-8
	res=0
	tmp=integrate.quad(M_ay_u_x,u_cutoff,1.0,args=(x,Sys,z),full_output=0,epsabs=eabs,epsrel=erel,limit=Nlim)
	res+=tmp[0]
	return res

cpdef double M_az_x(double x, double u_cutoff, SI_CI Sys, double z):
	cdef double eabs,erel,res
	cdef int Nlim=1000
	eabs,erel=100*Sys.reg_c,1e-8
	res=0
	tmp=integrate.quad(M_az_u_x,u_cutoff,1.0,args=(x,Sys,z),full_output=0,epsabs=eabs,epsrel=erel,limit=Nlim)
	res+=tmp[0]
	return res

cpdef double Q_x(double x, double u_cutoff, SI_CI Sys, double z):
	cdef double eabs,erel,res
	cdef int Nlim=1000
	eabs,erel=100*Sys.reg_c,1e-8
	res=0
	tmp=integrate.quad(Q_u_x,u_cutoff,1.0,args=(x,Sys,z),full_output=0,epsabs=eabs,epsrel=erel,limit=Nlim)
	res+=tmp[0]
	return res

###############################
## Finally we integrate in x ##
###############################
cpdef np.ndarray[DTYPE_t_r, ndim=1] M_ax(double u_cutoff, double x_cutoff, SI_CI Sys, double z):
	cdef double eabs,erel,x_left,x_right,xdiv_l,xdiv_r,res,error,Nevals,Nwarn
	cdef int Nlim=5000
	cdef np.ndarray[DTYPE_t_r, ndim=1] pts
	eabs,erel=100*Sys.reg_c,1e-8
	res,error,Nevals,Nwarn=0,0,0,0
	x_left=cos(Sys.theta/2)-Sys.T_up-Sys.T_down+Sys.Ez
	x_right=cos(Sys.theta/2)+Sys.T_up+Sys.T_down+Sys.Ez
	xdiv_l=1-Sys.Ez
	xdiv_r=1+Sys.Ez
	if (x_left<=0):
		pts=np.array([x_right,xdiv_l,xdiv_r],dtype=np.double)
	else:
		pts=np.array([x_left,x_right,xdiv_l,xdiv_r],dtype=np.double)
	tmp=integrate.quad(M_ax_x,0.0,x_cutoff,args=(u_cutoff,Sys,z),full_output=1,epsabs=eabs,epsrel=erel,limit=Nlim,points=pts)
	res+=tmp[0]
	error+=tmp[1]
	Nevals+=tmp[2]['neval']
	if (len(tmp)>3):
		Nwarn+=1
	return np.array([res,error,Nevals,Nwarn],dtype=np.double)

cpdef np.ndarray[DTYPE_t_r, ndim=1] M_ay(double u_cutoff, double x_cutoff, SI_CI Sys, double z):
	cdef double eabs,erel,x_left,x_right,xdiv_l,xdiv_r,res,error,Nevals,Nwarn
	cdef int Nlim=5000
	cdef np.ndarray[DTYPE_t_r, ndim=1] pts
	eabs,erel=100*Sys.reg_c,1e-8
	res,error,Nevals,Nwarn=0,0,0,0
	x_left=cos(Sys.theta/2)-Sys.T_up-Sys.T_down+Sys.Ez
	x_right=cos(Sys.theta/2)+Sys.T_up+Sys.T_down+Sys.Ez
	xdiv_l=1-Sys.Ez
	xdiv_r=1+Sys.Ez
	if (x_left<=0):
		pts=np.array([x_right,xdiv_l,xdiv_r],dtype=np.double)
	else:
		pts=np.array([x_left,x_right,xdiv_l,xdiv_r],dtype=np.double)
	tmp=integrate.quad(M_ay_x,0.0,x_cutoff,args=(u_cutoff,Sys,z),full_output=1,epsabs=eabs,epsrel=erel,limit=Nlim,points=pts)
	res+=tmp[0]
	error+=tmp[1]
	Nevals+=tmp[2]['neval']
	if (len(tmp)>3):
		Nwarn+=1
	return np.array([res,error,Nevals,Nwarn],dtype=np.double)

cpdef np.ndarray[DTYPE_t_r, ndim=1] M_az(double u_cutoff, double x_cutoff, SI_CI Sys, double z):
	cdef double eabs,erel,x_left,x_right,xdiv_l,xdiv_r,res,error,Nevals,Nwarn
	cdef int Nlim=5000
	cdef np.ndarray[DTYPE_t_r, ndim=1] pts
	eabs,erel=100*Sys.reg_c,1e-8
	res,error,Nevals,Nwarn=0,0,0,0
	x_left=cos(Sys.theta/2)-Sys.T_up-Sys.T_down+Sys.Ez
	x_right=cos(Sys.theta/2)+Sys.T_up+Sys.T_down+Sys.Ez
	xdiv_l=1-Sys.Ez
	xdiv_r=1+Sys.Ez
	if (x_left<=0):
		pts=np.array([x_right,xdiv_l,xdiv_r],dtype=np.double)
	else:
		pts=np.array([x_left,x_right,xdiv_l,xdiv_r],dtype=np.double)
	tmp=integrate.quad(M_az_x,0.0,x_cutoff,args=(u_cutoff,Sys,z),full_output=1,epsabs=eabs,epsrel=erel,limit=Nlim,points=pts)
	res+=tmp[0]
	error+=tmp[1]
	Nevals+=tmp[2]['neval']
	if (len(tmp)>3):
		Nwarn+=1
	return np.array([res,error,Nevals,Nwarn],dtype=np.double)

cpdef np.ndarray[DTYPE_t_r, ndim=1] Q(double u_cutoff, double x_cutoff, SI_CI Sys, double z):
	cdef double eabs,erel,x_left,x_right,xdiv_l,xdiv_r,res,error,Nevals,Nwarn
	cdef int Nlim=5000
	cdef np.ndarray[DTYPE_t_r, ndim=1] pts
	eabs,erel=100*Sys.reg_c,1e-8
	res,error,Nevals,Nwarn=0,0,0,0
	x_left=cos(Sys.theta/2)-Sys.T_up-Sys.T_down+Sys.Ez
	x_right=cos(Sys.theta/2)+Sys.T_up+Sys.T_down+Sys.Ez
	xdiv_l=1-Sys.Ez
	xdiv_r=1+Sys.Ez
	if (x_left<=0):
		pts=np.array([x_right,xdiv_l,xdiv_r],dtype=np.double)
	else:
		pts=np.array([x_left,x_right,xdiv_l,xdiv_r],dtype=np.double)
	tmp=integrate.quad(Q_x,0.0,x_cutoff,args=(u_cutoff,Sys,z),full_output=1,epsabs=eabs,epsrel=erel,limit=Nlim,points=pts)
	res+=tmp[0]
	error+=tmp[1]
	Nevals+=tmp[2]['neval']
	if (len(tmp)>3):
		Nwarn+=1
	return np.array([res,error,Nevals,Nwarn],dtype=np.double)
