#include "QueryParser.h"
#include "Query.h"
#include "htString.h"
#include "WordType.h"


FuzzyExpander *
QueryParser::expander = 0;

//
// parse a query string
//
//
Query *
QueryParser::Parse(const String &query_string)
{
	error = "";
	Token().Set(query_string);

	Query *result = ParseExpression();
	if(result && !Token().IsEnd())
	{
		Expected("end of query");
	//	delete result;
		result = 0;
	}
	return result;
}

// parse one word
// return a fuzzy word query
//
Query *
QueryParser::ParseWord()
{
	Query *result = 0;
	if(expander)
	{
		result = expander->MakeQuery(Token().Value());
	}
	else
	{
		result = new ExactWordQuery(Token().Value());
	}
	Token().Next();
	return result;
}

//
// parse one word
// return an exact query
//
Query *
QueryParser::ParseExactWord()
{
	Query *result = new ExactWordQuery(Token().Value());
	Token().Next();
	return result;
}

// 
// phrase == word { word }
//
Query *
QueryParser::ParsePhrase()
{
	Query *result = 0;
	Query *word = 0;
	if(!Token().IsEnd() && !Token().IsQuote())
	{
		word = ParseExactWord();
	}
	if(word)
	{
		result = new PhraseQuery;
		result->Add(word);
		while(word && !Token().IsEnd() && !Token().IsQuote())
		{
			word = ParseExactWord();
			if(word)
			{
				result->Add(word);
			}
		}
	}
	if(!word && result)
	{
		delete result;
		result = 0;
	}
	if(!result)
	{
		Expected("at least one word after \"");
	}
	return result;
}

void
QueryParser::Expected(const String &what)
{
	error << "Expected " << what;
	if(Token().IsEnd())
	{
		error << " at the end";
	}
	else
	{
		error << " instead of '" << Token().Value() << "'";
	}
}

//
// expr == term { term }
//
Query *
SimpleQueryParser::ParseExpression()
{
	Query *result = 0;
	Query *term = ParseTerm();
	if(term)
	{
		if(token.IsEnd())
		{
			result = term;
		}
		else
		{
			result = MakeQuery();
			result->Add(term);
			while(!token.IsEnd())
			{
				term = ParseTerm();
				if(term)
				{
					result->Add(term);
				}
			}
		}
	}
	if(!term)
	{
		delete result;
		result = 0;
	}
	return result;
}


//
// term == word | '"' phrase '"'
//
Query *
SimpleQueryParser::ParseTerm()
{
	Query *result = 0;

	if(token.IsQuote())
	{
		token.Next();
		result = ParsePhrase();
		if(result)
		{
			if(token.IsQuote())
			{
				token.Next();
			}
			else
			{
				Expected("closing \"");
				delete result;
				result = 0;
			}
		}
	}
	else if(token.IsWord())
	{
		// don't advance token here!
		result = ParseWord();
	}
	else
	{
		Expected("a word or a quoted phrase");
	}
	return result;
}


//
// expr == andlist { 'or' andlist }
//
Query *
BooleanQueryParser::ParseExpression()
{
	Query *result = 0;
	Query *term = ParseAnd();
	if(term)
	{
		if(token.IsOr())
		{
			result = new OrQuery;
			result->Add(term);
			while(term && token.IsOr())
			{
				token.Next();
				term = ParseAnd();
				if(term)
				{
					result->Add(term);
				}
			}
		}
		else
		{
			result = term;
		}
	}
	if(!term && result)
	{
		delete result;
		result = 0;
	}
	return result;
}

Query *
GParser::ParseExpression()
{
	List factors;
	Query *result = 0;
	String op = "";

	Query *factor = ParseFactor();
	if(factor)
	{
		result = factor;
	}
	while(factor && (token.IsOr() || token.IsAnd() || token.IsNot() || token.IsNear()))
	{
		if(op != token.Value())
		{
			Query *previous = result;
			result = MakeOperatorQuery(token.Value());
			result->Add(previous);
			op = token.Value();
		}
		token.Next();
		factor = ParseFactor();
		if(factor)
		{
			result->Add(factor);
		}
	}
	if(!factor && result)
	{
		delete result;
		result = 0;
	}
	return result;
}

OperatorQuery *
GParser::MakeOperatorQuery(const String &op) const
{
cerr << "Making operator for " << op << endl;
	OperatorQuery *result = 0;
	if(op == String("or"))
	{
		result = new OrQuery;
	}
	else if(op == String("and"))
	{
		result = new AndQuery;
	}
	else if(op == String("not"))
	{
		result = new NotQuery;
	}
	else if(op == String("near"))
	{
		result = new NearQuery;
	}
	return result;
}

//
// andlist = notlist { 'and' notlist }
//
Query *
BooleanQueryParser::ParseAnd()
{
	Query *result = 0;
	Query *not = ParseNot();
	if(not)
	{
		if(token.IsAnd())
		{
			result = new AndQuery();
			result->Add(not);
			while(not && token.IsAnd())
			{
				token.Next();
				not = ParseNot();
				if(not)
				{
					result->Add(not);
				}
			}
		}
		else
		{
			result = not;
		}
	}
	if(!not && result)
	{
		delete result;
		result = 0;
	}
	return result;
}

//
// notlist = nearlist { 'not' nearlist }
//
Query *
BooleanQueryParser::ParseNot()
{
	Query *result = 0;
	Query *near = ParseNear();
	if(near)
	{
		if(token.IsNot())
		{
			result = new NotQuery();
			result->Add(near);
			while(near && token.IsNot())
			{
				token.Next();
				near = ParseNear();
				if(near)
				{
					result->Add(near);
				}
			}
		}
		else
		{
			result = near;
		}
	}
	if(!near && result)
	{
		delete result;
		result = 0;
	}
	return result;
}

//
// near == factor { 'near'  [ '/' number ] factor }
// 'near' query is binary
//
Query *
BooleanQueryParser::ParseNear()
{
	Query *result = ParseFactor();
	while(result && token.IsNear())
	{
		token.Next();
		int distance = 10; // config["default_near_distance"];
		if(token.IsSlash())
		{
			distance = 0;
			token.Next();
			if(token.IsWord())
			{
				distance = token.Value().as_integer();
				token.Next();
			}
		}
		if(distance > 0)
		{
			Query *right = ParseFactor();
			if(right)
			{
				Query *tmp = new NearQuery(distance);
				tmp->Add(result);
				tmp->Add(right);
				result = tmp;
			}
			else
			{
				delete result;
				result = 0;
			}
		}
		else
		{
			Expected("a distance > 0 for 'Near'");
			delete result;
			result = 0;
		}
	}
	return result;
}

//
// factor == word | '"' phrase '"' | '(' expression ')'
//
Query *
BooleanQueryParser::ParseFactor()
{
	Query *result = 0;

	if(token.IsWord())
	{
		result = ParseWord();
	}
	else if(token.IsQuote())
	{
		token.Next();
		result = ParsePhrase();
		if(result)
		{
			if(token.IsQuote())
			{
				token.Next();
			}
			else
			{
				Expected("closing \"");
				delete result;
				result = 0;
			}
		}
	}
	else if(token.IsLeftParen())
	{
		token.Next();
		result = ParseExpression();
		if(result)
		{
			if(token.IsRightParen())
			{
				token.Next();
			}
			else
			{
				Expected(")");
				delete result;
				result = 0;
			}
		}
	}
	else
	{
		Expected("'(', '\"', or a word");
	}
	return result;
}


void
QueryLexer::Set(const String &query_string)
{
	query = query_string;
	current_char = 0;
	Next();
}

void
QueryLexer::Next()
{
	unsigned char	text = query[current_char];
	current = "";

	while (text
	&& !current.length()
	&& !WordType::Instance()->IsStrictChar(text))
	{
		if (text == '(' || text == ')' || text == '\"' || text == '/')
		{
	    		current << text;
			//cout << "lexer symbol: " << current << endl;
		}
		text = query[++current_char];
	}

	if (!current.length() && text)
	{
		while (text && WordType::Instance()->IsChar(text) && text != '/')
		{
			current << text;
			text = query[++current_char];
		}
	}
	current.lowercase();
	//cout << "lexer current word: " << current << endl;
}

bool
QueryLexer::IsEnd() const
{
	return current == String("");
}

bool
QueryLexer::IsQuote() const
{
	return current == String("\"");
}

bool
BooleanLexer::IsOr() const
{
	return current == String("or");
}
	
bool
BooleanLexer::IsAnd() const
{
	return current == String("and");
}

bool
BooleanLexer::IsNot() const
{
	return current == String("not");
}

bool
BooleanLexer::IsNear() const
{
	return current == String("near");
}

bool
BooleanLexer::IsSlash() const
{
	return current == String("/");
}

bool
BooleanLexer::IsLeftParen() const
{
	return current == String("(");
}

	
bool
BooleanLexer::IsRightParen() const
{
	return current == String(")");
}

bool
BooleanLexer::IsWord() const
{
	return !IsEnd()
	&& !IsQuote()
	&& !IsRightParen()
	&& !IsLeftParen()
	&& !IsSlash()
	&& !IsAnd()
	&& !IsOr()
	&& !IsAnd()
	&& !IsNot()
	&& !IsNear();
}
	
	
	
	

Query *
GParser::ParseFactor()
{
	Query *result = 0;

	if(token.IsWord())
	{
		result = ParseWord();
	}
	else if(token.IsQuote())
	{
		token.Next();
		result = ParsePhrase();
		if(result)
		{
			if(token.IsQuote())
			{
				token.Next();
			}
			else
			{
				Expected("closing \"");
				delete result;
				result = 0;
			}
		}
	}
	else if(token.IsLeftParen())
	{
		token.Next();
		result = ParseExpression();
		if(result)
		{
			if(token.IsRightParen())
			{
				token.Next();
			}
			else
			{
				Expected(")");
				delete result;
				result = 0;
			}
		}
	}
	else
	{
		Expected("'(', '\"', or a word");
	}
	return result;
}
