// Copyright (C) 2001 Gianni Mariani
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// As a special exception to the GNU General Public License, permission is
// granted for additional uses of the text contained in its release
// of Common C++.
//
// The exception is that, if you link the Common C++ library with other
// files to produce an executable, this does not by itself cause the
// resulting executable to be covered by the GNU General Public License.
// Your use of that executable is in no way restricted on account of
// linking the Common C++ library code into it.
//
// This exception does not however invalidate any other reasons why
// the executable file might be covered by the GNU General Public License.
//
// This exception applies only to the code released under the
// name Common C++.  If you copy code from other releases into a copy of
// Common C++, as the General Public License permits, the exception does
// not apply to the code that you add in this way.  To avoid misleading
// anyone as to the status of such modified files, you must delete
// this exception notice from them.
//
// If you write modifications of your own for Common C++, it is your choice
// whether to permit this exception to apply to your modifications.
// If you do not wish that, delete this exception notice.
//

//
// cmdoptns.cpp
//

#include "cmdoptns.h"

#ifdef HAVE_GETOPT_LONG

#include <string>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>
#include <iostream.h>


//
// In most cases, users will use this default option list.
//
CommandOption * DefaultCommandOptionList = 0;

CommandOption::CommandOption(
	const char		* in_option_name,
	const char		* in_option_letter,
	const char		* in_description,
	OptionType		  in_option_type,
	bool			  in_required = false,
	CommandOption  ** pp_next = & DefaultCommandOptionList
)
	:	option_name( in_option_name ),
		option_letter( in_option_letter ),
		description( in_description ),
		option_type( in_option_type ),
		required( in_required ),
		next( * pp_next )
{
	* pp_next = this;
}

CommandOptionWithArg::CommandOptionWithArg(
	const char		* in_option_name,
	const char		* in_option_letter,
	const char		* in_description,
	OptionType		  in_option_type,
	bool			  in_required = false,
	CommandOption  ** pp_next = & DefaultCommandOptionList
)
	: CommandOption(
		in_option_name,
		in_option_letter,
		in_description,
		in_option_type,
		in_required,
		pp_next
	), values( 0 ), numvalue( 0 )
{
}


CommandOptionArg::CommandOptionArg(
	const char		* in_option_name,
	const char		* in_option_letter,
	const char		* in_description,
	bool			  in_required = false,
	CommandOption  ** pp_next = & DefaultCommandOptionList
)
	: CommandOptionWithArg(
		in_option_name,
		in_option_letter,
		in_description,
		HasArg,
		in_required,
		pp_next
	)
{
}

CommandOptionRest::CommandOptionRest(
	const char		* in_option_name,
	const char		* in_option_letter,
	const char		* in_description,
	bool			  in_required = false,
	CommandOption  ** pp_next = & DefaultCommandOptionList
)
	: CommandOptionWithArg(
		in_option_name,
		in_option_letter,
		in_description,
		Trailing,
		in_required,
		pp_next
	)
{
}

CommandOptionCollect::CommandOptionCollect(
	const char		* in_option_name,
	const char		* in_option_letter,
	const char		* in_description,
	bool			  in_required = false,
	CommandOption  ** pp_next = & DefaultCommandOptionList
)
	: CommandOptionWithArg(
		in_option_name,
		in_option_letter,
		in_description,
		Collect,
		in_required,
		pp_next
	)
{
}

CommandOptionNoArg::CommandOptionNoArg(
	const char		* in_option_name,
	const char		* in_option_letter,
	const char		* in_description,
	bool			  in_required = false,
	CommandOption  ** pp_next = & DefaultCommandOptionList
)
	: CommandOption(
		in_option_name,
		in_option_letter,
		in_description,
		NoArg,
		in_required,
		pp_next
	), numset( 0 )
{
}

// ======== CommandOption =============================================
// PURPOSE:
//	CommandOption dummy methods ..
//

void CommandOption::ParseDone( CommandOptionParse * cop )
{
}

void CommandOption::FoundOption( CommandOptionParse * cop, const char * value = 0 )
{
}

void CommandOption::FoundOption( CommandOptionParse * cop, const char ** value, int num )
{
}

void CommandOption::PerformTask( CommandOptionParse * cop )
{
}
	
bool CommandOption::HasValue()
{
	return true;
}

CommandOption::~CommandOption()
{
}

CommandOptionWithArg::~CommandOptionWithArg()
{
	if ( values ) {
		free( values );
		values = 0;
	}
}

CommandOptionParse::~CommandOptionParse(void)
{
}

// ======== CommandOptionArg ==========================================
// PURPOSE:
//	Methods for CommandOptionArg
//

bool CommandOptionWithArg::HasValue()
{
	return numvalue > 0;
}

CommandOptionArg::~CommandOptionArg()
{
}	

//
//
static void my_alloc( char *** vals, int num, int incr )
{
	int	num_alloc = 0;
	if ( * vals ) {
		num_alloc = num | 3;
	}

	if ( ( incr + num ) > num_alloc ) {
		int newsiz = ( incr + num ) | 3;
		* vals = ( char ** ) realloc( * vals, sizeof( ** vals ) * newsiz );
	}
}

void CommandOptionWithArg::FoundOption( CommandOptionParse * cop, const char * value = 0 )
{
	if ( value ) {
		my_alloc( ( char *** ) & values, numvalue ? numvalue + 1 : 0, 1 );
		values[ numvalue ++ ] = value;
		values[ numvalue ] = 0;
	}
}

void CommandOptionWithArg::FoundOption( CommandOptionParse * cop, const char ** value, int num )
{
	my_alloc( ( char *** ) & values, numvalue ? numvalue + 1 : 0, num + 1 );

	int	j = 0;
	for ( int i = numvalue; j < num; i ++, j ++ ) {
		values[ i ] = value[ j ];
	}
	numvalue += num;
	values[ numvalue ] = 0;
}


void CommandOptionNoArg::FoundOption( CommandOptionParse * cop, const char * value = 0 )
{
	numset ++;
}

// ======== CommandOptionParse ========================================
// PURPOSE:
//	methods for CommandOptionParse
//

class CommandOptionParse_impl : public CommandOptionParse {
public:

	char			* comment;
	int				  num_options;
	struct option   * long_options;
	CommandOption  ** opt_list;
	CommandOption  ** co_list;
	char		    * optstring;
	CommandOption	* trailing;

	int				  argc;
	char		   ** argv;

	bool			  has_err;
	char			* fail_arg;

	bool			  usage_string_set;
	string			  usage_string;
	bool			  required_errors_set;
	string			  error_msgs;

	CommandOption	* fail_option;
	
	virtual ~CommandOptionParse_impl()
	{
		delete[] opt_list;
		delete[] co_list;
		delete[] optstring;
		delete[] long_options;
	}
		
	CommandOptionParse_impl(
		int				in_argc,
		char		 ** in_argv,
		char		  * in_comment,
		CommandOption * options
	) :
		comment( in_comment ),
		argc( in_argc ),
		argv( in_argv ),
		has_err( false ),
		fail_arg( 0 ),
		usage_string_set( false ),
		required_errors_set( false ),
		error_msgs( "" ),
		fail_option( 0 )
	{

		// First need to count all options.

		CommandOption		* to = options;
		int					  ocnt = 0;
		int					  ccnt = 0;
		int					  flag;

		while ( to ) {
			if ( to->option_name ) ocnt ++;
			ccnt ++;
			to = to->next;
		}

		num_options = ccnt;
		co_list = new (CommandOption *)[ocnt];
		opt_list = new (CommandOption *)[ccnt];
		long_options = new option[ccnt+1];
		optstring = new char[ 2*ccnt+2 ];

		// initialize the last option count
		long_options[ ocnt ].name = 0;
		long_options[ ocnt ].has_arg = 0;
		long_options[ ocnt ].flag = 0;
		long_options[ ocnt ].val = 0;
		
		char	* tos = optstring;
		* tos++ = '+';
		to = options;
		while ( to ) {

			if ( to->option_type == CommandOption::Trailing ) {
				if ( ! trailing ) {
					trailing = to;
				}
			} else if ( to->option_type == CommandOption::Collect ) {
				trailing = to;
			}

			opt_list[ -- ccnt ] = to;

			if ( to->option_name ) {
				-- ocnt;
				co_list[ ocnt ] = to;
				long_options[ ocnt ].name = to->option_name;
				long_options[ ocnt ].has_arg = to->option_type == CommandOption::HasArg;
				long_options[ ocnt ].flag = & flag;
				long_options[ ocnt ].val = ocnt;
			}

			if (  to->option_letter && to->option_letter[ 0 ] ) {
				* tos ++ = to->option_letter[ 0 ];
				if ( to->option_type == CommandOption::HasArg ) {
					* tos ++ = ':';
				}
			}

			
			to = to->next;
		}
		* tos = 0;

		int c;
		int option_index;

		opterr = 0;	// tell getopt_long not to print any errors
		flag = -1;
		while ( optind < argc ) {

			if (
				(
					c = getopt_long(
						argc, argv, optstring, long_options, &option_index
					)
				) == -1
			) {
				if ( ! trailing ) {
					break;
				} else if ( trailing->option_type == CommandOption::Trailing ) {
					break;
				} else {
					optarg = argv[ optind ];
					optind ++;
					to = trailing;
				}

			} else if ( flag != -1 ) {
				to = co_list[ flag ];
				flag = -1;
			} else if ( c == '?' ) {

				if ( optind < 2 ) {
					fail_arg = argv[ optind ];
				} else {
					fail_arg = argv[ optind - 1 ];
				}
				
				has_err = true;

				return;
				
			} else {

				// need to search through the options.

				for ( int i = 0; i < num_options; i ++ ) {

					to = opt_list[ i ];
					if ( ! to->option_letter ) continue;
					
					if ( c == to->option_letter[ 0 ] ) {

						break;
					}
				}
				// assert( to );
			}

			// do we terminate here ?
			if ( to->option_type == CommandOption::Trailing ) {
				break;
			}

			if ( c != ':' ) {
				to->FoundOption( this, optarg );
			} else {
				has_err = true;
				fail_option = to;
				break;
			}
				
		}

		if ( optind < argc ) {
			if ( trailing ) {
				trailing->FoundOption(
					this,
					( const char ** ) ( argv + optind ),
					argc - optind
				);
			} else {
				has_err = true;
				fail_arg = argv[ optind ];
			}
		}

		// Now check to see that all required args made it !

		for ( int i = 0; i < num_options; i ++ ) {
			CommandOption	* to = opt_list[ i ];

			// Tell this parameter that it's done now.
			to->ParseDone( this );
			
			if ( to->required && ! to->HasValue() ) {
				has_err = true;
				break;
			}
		}
		
	}
	
	bool	ArgsHaveError();

	virtual const char * PrintUsage();
	virtual const char * PrintErrors();

	void MakePrintErrors()
	{
		if ( required_errors_set ) return;
		required_errors_set = true;

		if ( fail_arg ) {
			error_msgs = error_msgs + "Unknown/malformed option '" + fail_arg + "' \n";
		} else if ( fail_option ) {
			string name;
			bool name_msg;
			if ( fail_option->option_name ) {
				name_msg = true;
				name = fail_option->option_name;
			} else if ( fail_option->option_letter ) {
				name_msg = true;
				name = fail_option->option_letter;
			} else if ( fail_option == trailing ) {
				name_msg = false;
			} else {
				name = "--option with no name--";
				name_msg = true;
			}
			if ( name_msg ) {
				error_msgs = error_msgs + "Option '" + name + "' requires value\n";
			}
		} else if ( has_err ) {

			// loop thru all required args
			
			for ( int i = 0; i < num_options; i ++ ) {
				CommandOption	* to = opt_list[ i ];
	
				if ( to->required && ! to->HasValue() ) {
					error_msgs = error_msgs + "Value required for option '";
					
					if ( to->option_name ) {
						error_msgs = error_msgs + "--" + to->option_name;
					} else if ( to->option_letter && to->option_letter[ 0 ] ) {
						error_msgs = error_msgs + '-' + to->option_letter[ 0 ];
					} else {
						error_msgs = error_msgs + to->description;
					}

					error_msgs = error_msgs + "' is missing\n";
				}
			}
		
		}

	}
	

	void MakePrintUsage()
	{
		if ( usage_string_set ) return;

		string	str( "" );
		
		string	str_argv0 = argv[ 0 ];

		str = str + "Usage : ";

		string::size_type slashpos = str_argv0.rfind('/');
		if ( slashpos > str_argv0.length() ) {
			slashpos = 0;
		} else {
			slashpos ++;
		}

		str.append( str_argv0, slashpos, str_argv0.length() - slashpos );

		str = str + ' ' + comment + '\n';
		
		for ( int i = 0; i < num_options; i ++ ) {

			CommandOption	* to = opt_list[ i ];
			char			* begin = "\t";
			char			* obegin = "\t";

			to = opt_list[ i ];

			if ( to->option_letter && to->option_letter[ 0 ] ) {
				str = str + begin + '-' + to->option_letter[ 0 ];
				begin = ", ";
				obegin = " - ";
			}
			
			if ( to->option_name ) {
				str = str + begin + "--" + to->option_name;
				begin = ", ";
				obegin = " - ";
			}

			if ( to->option_type == CommandOption::HasArg ) {
				str = str + begin + " <value>";
			} else if ( to->option_type == CommandOption::Trailing ) {
				str = str + begin + " <rest of command...>";
			} else if ( to->option_type == CommandOption::Collect ) {
				str = str + begin + " <...>";
			}

			str = str + obegin + to->description + "\n";
		}

		usage_string = str;
	}

    virtual void RegisterError( const char * err_msg )
	{
		error_msgs = error_msgs + err_msg + '\n';
		has_err = true;
	}

	virtual void PerformTask()
	{
		for ( int i = 0; i < num_options; i ++ ) {
			CommandOption	* to = opt_list[ i ];

			// Each parameter has this invoked
			to->PerformTask( this );
			
		}
	}

};

CommandOptionParse * MakeCommandOptionParse(
	int				   argc,
	char			** argv,
	char			 * comment,
	CommandOption	 * options = DefaultCommandOptionList
) {
	return new CommandOptionParse_impl( argc, argv, comment, options );
}

bool	CommandOptionParse_impl::ArgsHaveError()
{
	return has_err;
}

const char * CommandOptionParse_impl::PrintUsage()
{
	MakePrintUsage();
	return usage_string.c_str();
}

const char * CommandOptionParse_impl::PrintErrors()	
{
	MakePrintErrors();
	return error_msgs.c_str();
}

#endif
/** EMACS **
 * Local variables:
 * mode: c++
 * c-basic-offset: 8
 * End:
 */

