/*
 * Create tables for change set information.
 */

/*
 * ChangeLogs contains information from the change.
 */

drop table ChangeLogs;

CREATE TABLE ChangeLogs (
	ChangeLogGuid	CHAR(32)
		CONSTRAINT PK_ChangeLogGuid PRIMARY KEY,
	ChangeSetGuid	CHAR(32) NULL,
	CVSROOT			VARCHAR(1024),
	Module			VARCHAR(1024),
	Directory		VARCHAR(1024),
	UserId			VARCHAR(64),
	UnixTime		ABSTIME,
	LogMessageMD5	CHAR(32),
	MessageGuid		CHAR(32)
);

CREATE INDEX Idx_ChangeLogsMerge ON ChangeLogs ( UserId, LogMessageMD5, UnixTime );
CREATE INDEX Idx_ChangeLogsXS ON ChangeLogs ( ChangeSetGuid, UserId, LogMessageMD5, UnixTime );
CREATE INDEX Idx_ChangeLogsMessages ON ChangeLogs ( MessageGuid );
CREATE INDEX Idx_ChangeLogsDirectory ON ChangeLogs ( Directory, UnixTime );

/*
 * LogMessages are large.  These are broken into multiple records.
 * Once post-entry processing is done (ChangeSet is discovered),
 * duplicate records may be deleted.
 */

drop table Messages;

CREATE TABLE Messages (
	MessageGuid		CHAR(32),
	MessageSeq		INT4,
	MessageText		TEXT
);

CREATE INDEX Idx_Messages ON Messages ( MessageGuid, MessageSeq );

/*
 * BugDispositions are records extracted from the log message.
 * Once post-entry processing is done (ChangeSet is discovered),
 * duplicate records may be deleted.
 */

drop table BugDispositions;

CREATE TABLE BugDispositions (
	ChangeLogGuid	CHAR(32),
	BugId			VARCHAR(32),
	BugType			VARCHAR(32),
	BugNum			INT4,
	Disposition		VARCHAR(64)
);

CREATE INDEX Idx_BugDispositions ON BugDispositions ( ChangeLogGuid, BugId );
CREATE INDEX Idx_BugDispositionsXFR ON BugDispositions ( BugId, Disposition );

/*
 * FileRevisions describes the changes to the repository for each file.
 * FileRevisions are specific to each ChangeLog.
 */

drop table FileRevisions;

CREATE TABLE FileRevisions (
	ChangeLogGuid   CHAR(32),
	FileName		VARCHAR(1024),
	BranchName		VARCHAR(1024),
	OldRevision		VARCHAR(32),
	NewRevision		VARCHAR(32)
);

CREATE INDEX Idx_FileRevisionsG ON FileRevisions ( ChangeLogGuid, FileName, BranchName );
CREATE INDEX Idx_FileRevisionsF ON FileRevisions ( FileName, BranchName, ChangeLogGuid );
CREATE INDEX Idx_FileRevisionsB ON FileRevisions ( BranchName, FileName, ChangeLogGuid );

/*
 * ChangeSet records describe a collection of ChangeLogs.
 * ChangeSet records are discovered during entry processing.
 */

drop table ChangeSets;

CREATE TABLE ChangeSets (
	ChangeSetGuid   CHAR(32)
		CONSTRAINT PK_ChangeSetGuid PRIMARY KEY,
	BugsExported	VARCHAR(4),
	EmailsDone		VARCHAR(4)
);

/*
 * AppliedChangeSets describes which change-sets were described in
 * a change log.
 */

drop table AppliedChangeSets;

CREATE TABLE AppliedChangeSets (
	ChangeLogGuid			CHAR(32),
	AppliedChangeSetGuid	CHAR(32)
);

CREATE INDEX Idx_AppliedChangeSets ON AppliedChangeSets ( ChangeLogGuid, AppliedChangeSetGuid );
CREATE INDEX Idx_AppliedChangeSetsXchg ON AppliedChangeSets ( AppliedChangeSetGuid, ChangeLogGuid );

/*
 * AutoChangeSets
 * Automatically apply change sets from one branch to another branch.
 * This is to record which operations need to be applied automatically.
 */

DROP TABLE AutoChangeSets;

CREATE TABLE AutoChangeSets (
	AutoChangeSetId	CHAR(32)
		CONSTRAINT PK_AutoChangeSetId PRIMARY KEY,
	UnixTime		ABSTIME,
	Module			VARCHAR(1024),
	BranchFrom		VARCHAR(1024),
	BranchTo		VARCHAR(1024),
	IsActive		VARCHAR(4)	
);

/*
 * AutoChangeSetApplied
 * Once a change set has been applied, a record of it is placed here.
 * It should be followed by a new AppliedChangeSets record once the
 * changes have been checked into CVS.  AutoChangeSetApplied is used
 * to discriminate the phase difference of
 */

DROP TABLE AutoChangeSetApplied;

CREATE TABLE AutoChangeSetApplied (
	AutoChangeSetId		CHAR(32),
	UnixTime			ABSTIME,		/* The time this record was generated */
	ChangeSetGuid		CHAR(32),	/* The change set that is applied	*/
	AppliedChangeSet	CHAR(32)	/* The change set as applied to the to */
									/* branch							*/
);


/*
 * ListXS
 * This is a view on the main joins between tables and files.
 */


DROP VIEW ListXS;

CREATE VIEW
	ListXS
AS SELECT
	ChangeSets.ChangeSetGuid,
	ChangeSets.BugsExported,
	ChangeSets.EmailsDone,
	ChangeLogs.CVSROOT,
	Module,
	Directory,
	UserId,
	UnixTime,
	FileName,
	BranchName,
	OldRevision,
	NewRevision
FROM
	ChangeSets, ChangeLogs, FileRevisions
WHERE
	( ChangeSets.ChangeSetGuid = ChangeLogs.ChangeSetGuid )
	and ( FileRevisions.ChangeLogGuid = ChangeLogs.ChangeLogGuid )
;


/*
 * BugXS
 * This is a view on the main joins between tables and files.
 */


DROP VIEW BugXS;

CREATE VIEW
	BugXS
AS SELECT
	ChangeSets.ChangeSetGuid,
	BugId,
	Disposition
FROM
	ChangeSets, BugDispositions
WHERE
	( BugDispositions.ChangeLogGuid = ChangeSets.ChangeSetGuid )
;


