//
// loginfofilter.cpp
//
// This is a CVS "loginfo" filter.
//
// This creates a sql script to load the extracted values into the
// database of choice.  It also optionally creates an email with the
// log message that is sent to everyone.
//
// It expects to be executed with the %{eTVv} format.
// This format expects that the following patch is applied to
// CVS.  This was designed for CVS version 1.11.
//
//------------------------------------------------------------------------
#if 0
--- logmsg.c~	Wed Mar  1 08:33:39 2000
+++ logmsg.c	Sun Mar 25 15:14:57 2001
@@ -585,6 +585,50 @@
 	    {
 		switch (*c)
 		{
+		/* perform escaped format file name */
+		case 'e':
+		    {
+			int len = strlen (str_list);
+			char * addstr = p->key;
+			char * send;
+			str_list =
+			    xrealloc (str_list,
+				  len + strlen (addstr) * 2 + 5);
+			send = str_list + len; /* String end */
+			while ( * addstr ) {
+			    switch ( * addstr ) {
+				case ' ' :
+				case ',' :
+				case '\\' :
+				    * send++ = '\\';
+			    }
+			    * send++ = * addstr++;
+			}
+			* send = 0;
+		    }
+		    break;
+		/* the tag format */
+		case 'T':
+		    {
+			int len = strlen (str_list);
+			char * send;
+			char * addstr = li->tag ? li->tag : "";
+			str_list =
+			    xrealloc (str_list,
+				  len + strlen (addstr) * 2 + 5);
+			send = str_list + len; /* String end */
+			while ( * addstr ) {
+			    switch ( * addstr ) {
+				case ' ' :
+				case ',' :
+				case '\\' :
+				    * send++ = '\\';
+			    }
+			    * send++ = * addstr++;
+			}
+			* send = 0;
+		    }
+		    break;
 		case 's':
 		    str_list =
 			xrealloc (str_list,
#endif
//------------------------------------------------------------------------
//
//

#include <ostream.h>
#include <stdlib.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#include <time.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <fstream.h>
#include <multimap.h>

#define gg_guid_define
#include "gguid.h"
#include "bug_type.h"
#include "md5.h"
#include "cmdoptns.h"

char log_message_header[] = "\nLog Message:\n";
char new_dir_str[] = "- New directory";
char imported_str[] = "- Imported sources";

#define BUF_INITSIZ		(1024*2)

#define Xalloc ( char * ) ::malloc
#define Xrealloc ( char * ) ::realloc

class loginfofilter;


// ======== readfile ==================================================
// PURPOSE:
//	Read a file into a buffer.
//
// RETURNS:
// 	Pointer to a buffer containing the contents read in. plen points
//	to the length of the buffer read.
//  nil is returned on failure.
//

char * readfile( int fd, int * plen )
{
	char * buf = Xalloc( BUF_INITSIZ );
	int capacity = BUF_INITSIZ;
	int	rlen = 0;
	int	retval;

	while ( ( retval = ::read( fd, buf + rlen, capacity - rlen - 1 ) ) ) {

		if ( retval == -1 ) {
			if ( errno == EINTR ) {
				continue;
			}
			free( buf );
			return 0;
		}

		rlen += retval;

		if ( capacity < ( rlen + BUF_INITSIZ/4 ) ) {
			capacity += capacity;
			buf = Xrealloc( buf, capacity );
			if ( !buf ) {
				return 0;
			}
		}
	}

	* plen = rlen;

	buf = Xrealloc( buf, rlen + 1 );
	buf[ rlen ] = 0;
	
	return buf;
	
} // end readfile



// ======== findmem ===================================================
// PURPOSE:
//	Find a duplicate of a buffer within a buffer.
//
// RETURNS:
// 	Pointer to the first duplicate byte in the buffer pointed to
//	by 'data'.
//

const char * findmem(
	const char * data, size_t datalen, const char * str, size_t len
) {
	const char	* pos;
	char	  lastch = str[ len - 1 ];
	
	if ( len > datalen ) {
		return 0;
	}

	size_t	  left = datalen - ( len - 1 );
	
	pos = data + len -1;

	while ( left > 0 ) {
		if ( pos[ 0 ] == lastch ) {
			if ( ! ::memcmp( pos - len + 1, str, len - 1 ) ) {
				return pos - len + 1;
			}
		}
		pos ++;
		left --;
	}

	return 0;

} // end findmem


// ======== md5_if ====================================================
// PURPOSE:
//	Helper class for using the md5 interface.
//

class md5_if {
	struct MD5Context a_ctxt[1];

public:

	// default constructor
	md5_if()
	{
		MD5Init( a_ctxt );
	}

	// compute the md5sum of the buffer.
	void Update( unsigned char const *buf, unsigned len )
	{
		MD5Update( a_ctxt,  buf, len );
	}

	// construct and compute the md5 sum of the buffer
	md5_if( unsigned char const *buf, unsigned len )
	{
		MD5Init( a_ctxt );

		Update( buf, len );
	}

	// Get the md5 digest from the context.
	void GetDigest( unsigned char digest[16] )
	{
		MD5Final( digest, a_ctxt );
	}
};

// ======== SQLescape =================================================
// PURPOSE:
//	Escape a string for SQL
//

void	SQLescape( ostream &fo, const char * s )
{

	bool	bol = false;
	
	while ( *s ) {
		if ( *s == '\'' ) {
			fo << "\\'";
		} else if ( bol && *s == '\n' ) {
			fo << "\\n";
		} else {
			fo << *s;
		}
		bol = ( *s == '\n' );

		s ++;
	}	

    return;

} // end SQLescape

// print exactly len bytes
void	SQLescape( ostream &fo, const char * s, int len )
{

	bool	bol = false;
	
	while ( len ) {
		if ( *s == '\'' ) {
			fo << "\\'";
		} else if ( bol && *s == '\n' ) {
			fo << "\\n";
		} else {
			fo << *s;
		}
		bol = ( *s == '\n' );

		s ++;
		len --;
	}	

    return;

} // end SQLescape



// ======== log_message_t =============================================
// PURPOSE:
//	A class to manage the log message and file content.
//

class log_message_t {
public:

	gg_guid			  msgguid;
	char			* file_content;
	int				  file_content_len;
	const char		* log_message;
	int				  log_message_len;
	gg_guid			  md5_digest;

	// Construct the log message from an input fril descriptor
	log_message_t( int fd )
	{
		msgguid.make();

		file_content = readfile( fd, & file_content_len );

		if ( ! file_content ) {
			file_content = "";
			file_content_len = 0;
		}
		
		log_message = findmem(
			file_content, file_content_len,
			log_message_header, sizeof( log_message_header ) -1
		);

		if ( ! log_message ) {
			log_message = "";
			log_message_len = 0;
		} else {
			log_message += sizeof( log_message_header ) -1;
			log_message_len =
				file_content_len - ( log_message - file_content );
		}

		// Compute the md5sum
		md5_if md5sum(( unsigned char const * ) log_message, log_message_len);
		md5sum.GetDigest( md5_digest.guid );
		
	}

	// Print the content to standard output
	void dump_content( ostream &fo )
	{
		fo.write( log_message, log_message_len );
	}

	// need to dump content into one or more tables
	void dump_sql_content( ostream &fo, gg_guid &i_guid )
	{
		// write sql statements with log_message broken out in chunks
		int chunk = 6000;		// size o chunks
		int j = 0;

		// Need to handle 0 length messages.
		for ( int i = 0; (!j) || ( i < log_message_len ); i += chunk ) {
			fo << "\n"
				"INSERT INTO Messages (\n"
				"	MessageGuid,\n"
				"	MessageSeq,\n"
				"	MessageText\n"
				") VALUES (\n"
				"	'"; msgguid.print_hex( fo ); fo <<"',\n"
				"	" << j++ << ",\n"
				"	'";
	
			SQLescape(
				fo, log_message + i,
				(
					( ( log_message_len - i ) > chunk )
					? chunk
					: ( log_message_len - i )
				)
			);
	
			fo << "'\n);\n\n";
		}
					
	}

};


// ======== parse_string ==============================================
// PURPOSE:
//	Parse a string containing escape sequences ( pairs beginning with \).
//	The second character in the sequence is the wanted character.
//	This parses file names and tag names from the CVS log information.
//	The parse stops on the first character equal to ch or then end of
//	the string.
//
// RETURNS:
// 	A pointer to the first unescaped "ch" byte.
//

char * parse_string( char * s, char ch, int * plen, bool strend = false )
{
	char	* sin = s;
	int		i = 0;
	if ( plen ) * plen = 0;
	
	// skip characters after a '\'
	while ( * s && ( * s != ch ) ) {
		if ( * s == '\\' ) {
			s ++;
		}
		i ++;
		s ++;
	}

	if ( !strend && ( * s != ch ) ) {
		cerr << "Failed to parse log entry : '" << sin << "'\n";
		::exit( 1 );
	}
	
	if ( plen ) * plen = i;
	return s;
}


// ======== parse_dup =================================================
// PURPOSE:
//	Create a duplicate of a "\" escaped string of size i.
//
// RETURNS:
// 	A pointer to the buffer containing a malloc'd duplicate.
//

char * parse_dup( char * sin, int i )
{
	char * str1 = Xalloc( i + 1 );
	char * retstr = str1;
	
	for ( const char * s = sin; i; s ++, i -- ) {
		if ( * s == '\\' ) {
			s ++;
		}
		* str1++ = * s;
	}
	
	* str1 = 0;
	return retstr;
}


// ======== log_entry_t ===============================================
// PURPOSE:
//	Manage a change log entry.
//

class log_entry_t {
public:
	const char	* filename;
	const char	* branch_tag;
	const char	* rev_old;
	const char	* rev_new;

	log_entry_t * next;

	log_entry_t( char ** pstr )
	{
		char * str = *pstr;

		char * comma;
		int	   i = 0;

		// Parse out the file name
		comma = parse_string( str, ',', &i );
		filename = parse_dup( str, i );
		
		// Parse out the branch tag

		str = ++ comma;
		comma = parse_string( str, ',', &i );
		branch_tag = parse_dup( str, i );
		
		// Parse out the old revision

		str = ++ comma;
		comma = parse_string( str, ',', &i );
		rev_old = parse_dup( str, i );
				
		// Parse out the new revision
		
		str = ++ comma;
		char * space = parse_string( str, ' ', &i, true );
		rev_new = parse_dup( str, i );

		// Point to the new spot !
		* pstr = space;
	}

	log_entry_t(
		loginfofilter	* lif,
		const char		* beginstr,
		const char		* end_dir,
		const char		* end_file
	);
	
	log_entry_t(
		const char * nfilename,
		const char * nbranch_tag,
		const char * nrev_old,
		const char * nrev_new
	) {
		filename = nfilename;
		branch_tag = nbranch_tag;
		rev_old = nrev_old;
		rev_new = nrev_new;
	}
	
	void dump_content( ostream &fo )
	{
		fo << "File '" << filename << "' ";
		fo << "Branch '" << branch_tag << "' ";
		fo << "Old rev '" << rev_old << "' ";
		fo << "New rev '" << rev_new << "'\n";
	}

	void dump_sql_content( ostream &fo, gg_guid &i_guid )
	{
		fo <<
			"INSERT INTO FileRevisions (\n"
			"	ChangeLogGuid,\n"
			"	FileName,\n"
			"	BranchName,\n"
			"	OldRevision,\n"
			"	NewRevision\n"
			") VALUES (\n"
			"	'"; i_guid.print_hex( fo ); fo << "',\n"
			"	'"; SQLescape(fo, filename ); fo << "',\n"
			"	'"; SQLescape(fo, branch_tag ); fo << "',\n"
			"	'"; SQLescape(fo, rev_old ); fo << "',\n"
			"	'"; SQLescape(fo, rev_new ); fo << "'\n"
			");\n\n";
	}
	
};


class ImportFile {
public:
	ImportFile		* next;
	char			* filename;
	
	ImportFile(
		loginfofilter	* lif,
		const char		* beginstr,
		const char		* end_dir,
		const char		* end_file
	);
};

// Multimap compare routine
struct ltstr
{
	bool operator()(const char* s1, const char* s2) const
	{
		return strcmp(s1, s2) < 0;
	}
};

typedef multimap<const char*, loginfofilter *, ltstr>	lif_map_t;
typedef pair<const char*, loginfofilter *>				lif_pair_t;

class loginfofilter {
public:
	log_message_t	* lm;
	char			* cvsroot;
	char			* user;
	log_entry_t		* log_entry;	// linked list of entries
	char			* directory;
	char			* module;
	time_t			  now;
	bug_list		  bl;
	gg_guid			  my_guid;
	lif_map_t		* map_o_lifs;
	loginfofilter	* lif_parent;

private:
	void init_loginfofilter(
		const char * arg, CommandOptionParse * args
	)
	{
		int		arglen = strlen( arg );

		// if this is a New Directory log - give up.
		if ( findmem( arg, arglen, new_dir_str, sizeof( new_dir_str ) -1 ) ) {
			::exit( 0 );
		}

		// get the time it is now
		::time( & now );

		// Get the user name string 
		struct passwd * pw_ent = ::getpwuid( ::getuid() );

		if ( ! pw_ent ) {
			user = Xalloc( 14 );
			::sprintf( user, "%ud", ::getuid() );
		} else {
			user = ::strdup( pw_ent->pw_name );
		}
		
		// Need a working area above CVSROOT - so we need a CVSROOT environment
		// variable in directory form only.
		cvsroot = ::getenv( "CVSROOT" );

		// We must have a CVSROOT
		if ( ! cvsroot ) {
			cerr << "*** must have environment variable CVSROOT set\n";
			cerr << args->PrintUsage();
			::exit( 1 );
		}

		// We must have a CVSROOT starting at '/'
		if ( cvsroot[ 0 ] != '/' ) {
			cerr << "*** CVSROOT must be a directory '" << cvsroot << "'\n";
			cerr << args->PrintUsage();
			::exit( 1 );
		}

		// Need to check on how sources are imported.
		if ( findmem( arg, arglen, imported_str, sizeof( imported_str ) -1 ) ) {

			// Importing files.
			// Need to parse the log message which contains a list of all imported files.
			// Files that are imported start with a 'N'.  Files that have already been
			// imported start with a 'U'.

			const char * t_str = findmem(
				lm->log_message, lm->log_message_len,
				"\nStatus:\n\n", sizeof( "\nStatus:\n\n" ) - 1
			);

			const char * new_end = t_str;

			const char * lm_end = lm->log_message + lm->log_message_len;

			// bugs can only be in the initial part of the message - don't parse the
			// file names.
			parse_bugs( &bl, lm->log_message, t_str - lm->log_message );
			
			while ( t_str = findmem( t_str, lm_end - t_str, "\nN ", 3 ) ) {

				// skip over the "\nN "
				t_str += 3;

				// find the end of the line
				char * endline = ( char * ) ::memchr( t_str, '\n', lm_end - t_str );

				if ( ! endline ) {
					cerr << "Import log message is broken - new line is missing\n";
					::exit( 1 );
				}

				// Look for the last '/' - this delimits the directory name
				char * end_file = endline;

				while ( ( end_file > t_str ) && ( * end_file != '/' ) ) end_file --;

				if ( * end_file != '/' ) {
					cerr << "Import log message is broken - no '/' in file name :'";
					cerr.write( t_str - 3, 3 + endline - t_str );
					cerr << "'\n";
					::exit( 1 );
				}

				// Add the file to the list o imported files.
				new log_entry_t( this, t_str, end_file, endline );

				t_str = endline;

			}

			// Remove all the log message nonsese
			lm->log_message_len = new_end - lm->log_message;

			
		} else {

			// Find the directory name - since the name is not escaped
			// we need to head over to the first "," and go back to a
			// space character.
	
			char * t_str = strchr( arg, ',' );
	
			if ( ! t_str ) {
				// very strange error
				cerr << arg << " - contains no comma separated fields - badly formatted args\n";
				::exit( 1 );
			}
			
			char * end_dir = 0;
	
			while ( arg < t_str ) {
	
				if ( * t_str == ' ' ) {
					if ( ( t_str - arg ) > 1 ) {
						if ( t_str[ -1 ] == '\\' ) {
							t_str -= 2;
							continue;
						} else {
							end_dir = t_str;
							break;
						}
					}
				}
	
				t_str --;
			}
	
			if ( ! end_dir ) {
				cerr << arg << " - found no leading directory - badly formatted args\n";
				::exit( 1 );
			}
	
			directory = Xalloc( end_dir - arg + 1 );
	
			memcpy( directory, arg, end_dir - arg );
			directory[ end_dir - arg ] = 0;
	
			char * dt_str = strchr( directory, '/' );
	
			// find the module
			if ( !dt_str ) {
				module = strdup( directory );
			} else {
				module = Xalloc( dt_str - directory + 1 );
				memcpy( module, directory, dt_str - directory );
				module[ dt_str - directory ] = 0;
			}
	
			
			// go get all the files.
			while ( * t_str ) {
	
				t_str ++; // skip over the space
	
				log_entry_t * entry = new log_entry_t( & t_str );
	
				entry->next = log_entry;
				log_entry = entry;
	
			}

			parse_bugs( &bl, lm->log_message, lm->log_message_len );
		}
		
	}

public:
	// constructor
	loginfofilter( log_message_t * ilm, const char * arg, CommandOptionParse * args )
		:
			lm( ilm ),
			log_entry( 0 ),
			bl(),
			map_o_lifs(0),
			lif_parent(0)
	{
		my_guid.make();
		init_loginfofilter( arg, args );
	}
	
	loginfofilter(
		log_message_t * ilm, const char * arg, CommandOptionParse * args, gg_guid & gd
	)
		:
			lm( ilm ),
			log_entry( 0 ),
			bl(),
			map_o_lifs(0),
			lif_parent(0)
	{
		my_guid = gd;
		init_loginfofilter( arg, args );
	}
	

	//
	// Constructor for a copy of a loginfofilter for imported files.
	//
	loginfofilter(
		loginfofilter	* parent,
		const char		* inmodule,
		const char		* indirectory
	)
		:
			lm( parent->lm ),
			log_entry( 0 ),
			bl(),
			map_o_lifs(0),
			lif_parent( parent )
	{

		// Initialize the basic stuff
		now = parent->now + 1;

		my_guid.make();		
		cvsroot = lif_parent->cvsroot;
		user = lif_parent->user;
		
		directory = strdup( indirectory );
		module = strdup( inmodule );

		// Insert this new loginfofilter into the parent's map
		lif_parent->map_o_lifs->insert( lif_pair_t( directory, this ) );
		
	}
	

	void dump_content_raw( ostream &fo )
	{
		fo << "Dump of log-info-filter\n";

		fo << "GUID      : '";
		my_guid.print_hex( fo );
		fo << "'\n";
		fo << "CVSROOT   : '" << cvsroot << "'\n";
		fo << "directory : '" << directory << "'\n";
		fo << "user      : '" << user << "'\n";
		fo << "Unix time : '" << now << "'\n";
		fo << "Mesg.guid : '";
		lm->msgguid.print_hex( fo );
		fo << "'\n";
		fo << "Log md5sum: '";
		lm->md5_digest.print_hex( fo );
		fo << "'\n";
		fo << "Log message:\n";
		lm->dump_content( fo );
		fo << "------\n";
		
		for ( log_entry_t * entry = log_entry; entry; entry = entry->next ) {
			fo << "Entry : ";
			entry->dump_content( fo );
		}

		bl.dump_content( fo );
	}

	void dump_content( ostream &fo )
	{
		if ( map_o_lifs ) {
			for (
				lif_map_t::iterator	it = map_o_lifs->begin();
				it != map_o_lifs->end();
				++ it
			) {
				(*it).second->dump_content( fo );
			}
		} else {

			dump_content( fo );

		}

	}

	
	void dump_sql_content_raw( ostream &fo )
	{

		fo <<
			"INSERT INTO ChangeLogs (\n"
			"	ChangeLogGuid,\n"
			"	CVSROOT,\n"
			"	Module,\n"
			"	Directory,\n"
			"	UserId,\n"
			"	UnixTime,\n"
			"	LogMessageMD5,\n"
			"	MessageGuid\n"
			") VALUES (\n"
			"	'"; my_guid.print_hex( fo ); fo <<"',\n"
			"	'"; SQLescape(fo, cvsroot); fo <<"',\n"
			"	'"; SQLescape(fo, module); fo <<"',\n"
			"	'"; SQLescape(fo, directory); fo <<"',\n"
			"	'"; SQLescape(fo, user); fo <<"',\n"
			"	" << now << ",\n"
			"	'"; lm->md5_digest.print_hex( fo ); fo <<"',\n"
			"	'"; lm->msgguid.print_hex( fo ); fo <<"'\n"
			");\n\n";

		for ( log_entry_t * entry = log_entry; entry; entry = entry->next ) {
			entry->dump_sql_content( fo, my_guid );
		}

		bl.dump_sql_content( fo, my_guid );
		
	}	

	void dump_sql_content( ostream &fo )
	{
		fo << "/* CVS Loginfo SQL output */\n\n\n";
		fo << "BEGIN;\n\n";

		// Dump the log message only once !
		lm->dump_sql_content( fo, my_guid );

		// Now we need to dump any other log info structs

		if ( map_o_lifs ) {
			for (
				lif_map_t::iterator	it = map_o_lifs->begin();
				it != map_o_lifs->end();
				++ it
			) {
				(*it).second->dump_sql_content_raw( fo );
			}
		} else {

			dump_sql_content_raw( fo );

		}

		fo << "\nCOMMIT;\n\n/* End of SQL output */\n";
		
	}	

};

log_entry_t::log_entry_t(
	loginfofilter	* lif,
	const char		* beginstr,
	const char		* end_dir,
	const char		* end_file
) {

	bool			  first_time = false;
	
	// Find the map of lifs
	lif_map_t * lifmap = lif->map_o_lifs;
	if ( ! lifmap ) {
		lifmap = lif->map_o_lifs = new lif_map_t();
		first_time = true;
	}

	// create a key to find an lif.
	int len = end_dir - beginstr;
	char sdirectory[ len + 1 ];
	::memcpy( sdirectory, beginstr, len );
	sdirectory[ len ] = 0;

	// Find the lif object for this directory
	lif_map_t::const_iterator itr = lifmap->find( sdirectory );

	// Finding the correct loginfofilter
	loginfofilter * nlif;
	if ( itr == lifmap->end() ) {

		const char * first_slash = ( const char * ) ::memchr( beginstr, '/', end_dir - beginstr + 1 );
		int	mlen = first_slash - beginstr;
		char smodule[ mlen + 1 ];
		::memcpy( smodule, beginstr, mlen );
		smodule[ mlen ] = 0;

		// Need to create a new one - or for the first time use this one.
		if ( first_time ) {
			lif->directory = strdup( sdirectory );
			lif->module = strdup( smodule );
			lifmap->insert( lif_pair_t( lif->directory, lif ) );
			nlif = lif;
		} else {
			nlif = new loginfofilter( lif, smodule, sdirectory );
		}
	} else {
		// Found it !
		nlif = (* itr).second;
	}


	len = end_file - end_dir;
	char * fn = new char[ len ];
	::memcpy( fn, ( char * ) (end_dir+1), len-1 );
	fn[ len -1 ] = 0;
	
	// Now we need to initialize this
	filename = fn ;
	branch_tag = "";
	rev_old = "NONE";
	rev_new = "1.1";

	// Link this entry....
	this->next = nlif->log_entry;
	nlif->log_entry = this;

}


//
// Define command line arguments.
//

CommandOptionNoArg  helparg(
	"help", "?", "Print help usage"
);

CommandOptionArg	sql_out(
	"sql_out", "S", "Create SQL data in the directory specified"
);

CommandOptionArg	cvsxsroot(
	"cvsxs", "d", "the cvsxs root directory"
);

CommandOptionArg	log_history(
	"log_history", "h", "Append the command input to the \n"
	"\t\tfile specified."
);

CommandOptionNoArg	debug_out(
	"dump", "i", "dump contents to standard output."
);

CommandOptionCollect    restoargs(
	0, 0, "CVS loginfo format %{eTVv} argument. (cvs must have %{eT} patch)",
	true
);


// ======== go_do_command =============================================
// PURPOSE:
//	Execute a command specified
//
// RETURNS:
// 	
//

void	go_do_command( const char * command, const char * filename )
{


    return;

} // end go_do_command


int main( int argc, char ** argv )
{
	// Parse command line arguments
	CommandOptionParse * args = MakeCommandOptionParse(
		argc, argv,
		"CVS loginfo filter."
	);

    // If the user requested help then suppress all the usage error
	// messages.
	if ( helparg.numset ) {
		cerr << args->PrintUsage();
		::exit(1);
	}
	
    // Print usage your way.
	if ( args->ArgsHaveError() ) {
		cerr << args->PrintErrors();
		cerr << args->PrintUsage();
		::exit(1);
	}
		
	// check to see the number of arguments.
	if ( restoargs.numvalue != 1 ) {
		// Can only have one argument
		cerr << "Too many arguments\n";
		cerr << "Expecting single argument supplied by CVS\n";
		cerr << args->PrintUsage();
		::exit(1);
	}

	// make the guid here
	gg_guid gd;
	gd.make();

	// Read the log message from standard input
	log_message_t	blm( 0 );

	// For every history file - dump the content
	for ( int i = 0; i < log_history.numvalue; i ++ ) {
		const char * filename = log_history.values[ i ];
		
		int fd;

		if ( -1 == ( fd = open( filename, O_WRONLY | O_CREAT | O_NOCTTY, 0666 ) ) ) {
			char	* errstr = strerror( errno );
			cerr
				<< "Failed to open history file: '"
				<< filename << "' : " << errstr << "\n";
			continue;
		}

		struct flock a_flock[ 1 ];

		a_flock->l_type = F_WRLCK;
		a_flock->l_whence = SEEK_CUR;
		a_flock->l_start = 0;
		a_flock->l_len = 0;
		a_flock->l_pid = 0;
		
		while ( -1 == fcntl( fd, F_SETLKW, a_flock ) ) {
			if ( errno != EINTR ) {
				char * errstr = strerror( errno );
				cerr << "Failed to take lock on history file: '"
				<< filename << "' : " << errstr << "\n";
				break;
			}
		}

		// Seek to the end of the file for appending.
		::lseek( fd, 0, SEEK_END );

		ofstream xout( fd );

		if ( ! xout.is_open() ) {
			cerr << "Failed to open history: '" << filename << "'\n";
			continue;
		}		
		
		xout << "--------"; gd.print_hex( xout ); xout << "\n";
		
		xout << " " << restoargs.values[ 0 ] << "\n";

		xout << "++++++++"; gd.print_hex( xout ); xout << "\n";
		
		char * cvsroot = ::getenv( "CVSROOT" );

		if ( cvsroot ) {
			xout << "CVSROOT='" << cvsroot << "'\n";
		} else {
			xout << "CVSROOT=''\n";
		}
		
		xout << "++++++++"; gd.print_hex( xout ); xout << "\n";
		
		// get the time it is now
		xout << "TIME=" << ::time( 0 ) << "\n";

		xout << "++++++++"; gd.print_hex( xout ); xout << "\n";
		
		// Get the user name string 
		struct passwd * pw_ent = ::getpwuid( ::getuid() );

		char * user;
		if ( ! pw_ent ) {
			user = Xalloc( 14 );
			::sprintf( user, "%ud", ::getuid() );
		} else {
			user = ::strdup( pw_ent->pw_name );
		}

		xout << "USER='" << user << "'\n";

		xout << "++++++++"; gd.print_hex( xout ); xout << "\n";
		
		xout.write( blm.log_message, blm.log_message_len );

		xout << "--------"; gd.print_hex( xout ); xout << "\n";

		xout.close();

	}
	
	// Filter all the data sent
	loginfofilter *lif = new loginfofilter( &blm, restoargs.values[ 0 ], args, gd );

	// Print debug dump
	if ( debug_out.numset ) {
		lif->dump_content( cout );
	}

	// Dump sql commands into directories - files named by the
	// guid followed by .sql
	for ( int i = 0; i < sql_out.numvalue; i ++ ) {

		// Find the length of the directory name
		int		dirlen = strlen( sql_out.values[ i ] );

		// normalize on the trailing '/'
		if ( sql_out.values[ i ][ dirlen - 1 ] != '/' ) dirlen ++;

		// create a buffer - dir len + guid + .sql + nul
		char filename[ dirlen + 32 + sizeof( ".sql" ) ];
		char * ts = filename;

		// copy directory contents
		memcpy( ts, sql_out.values[ i ], dirlen - 1 );
		ts += (dirlen - 1);

		// add the '/'
		* ( ts ++ ) = '/';

		// place the guid in buffer
		gd.print_hex( ts );
		ts += 32;

		// Copy the string
		memcpy( ts, ".sql", sizeof( ".sql" ) );
		// It should be null terminated now !

		// open the file
		ofstream xout( filename );

		// bomb is the file did not open
		if ( ! xout.is_open() ) {
			cerr << "Failed to open SQL out file: '" << filename << "'\n";
			continue;
		}
		
		lif->dump_sql_content( xout );
		xout.close();

	}

	return 0;
}
