/* parallel init
 * rc.c
 *
 * Copyright (C) 2005 Olivier Sessink
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
/* #define DEBUG */
/*
 * indenting is done with
 * indent --line-length 100 --k-and-r-style --tab-size 4 -bbo --ignore-newlines rc.c
 */

#ifdef DEBUG
#define DEBUG_MSG printf
#else							/* not DEBUG */
#ifdef __GNUC__
#define DEBUG_MSG(...)
 /**/
#else							/* notdef __GNUC__ */
void empty(char *arg, ...);
#define DEBUG_MSG empty
#endif							/* __GNUC__ */
#endif							/* DEBUG */

#define PRINTTIME

#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/wait.h>
#include <string.h>
#include <glob.h>
#include <stdio.h>
#include <errno.h>

#ifdef PRINTTIME
#include <time.h>
#endif

#ifndef __GNUC__
void empty(char *arg, ...)
{
}
#endif

#define BUF_GROW_SIZE 1024
typedef struct {
	char *buf;
	size_t len;
	size_t size;
} Tbuf;

static void append2buf(Tbuf * buf, char *data, size_t len)
{
	DEBUG_MSG("append2buf, buflen=%d, bufsize=%d, len=%d\n",buf->len,buf->size,len);
	if (buf->len + len + 1 > buf->size) {
		buf->size += (len +1 < BUF_GROW_SIZE) ? BUF_GROW_SIZE : len +1;
		buf->buf = realloc(buf->buf, buf->size);
	}
	memcpy(buf->buf + buf->len, data, len);
	buf->len += len;
	buf->buf[buf->len] = '\0';
	DEBUG_MSG("append2buf, buf has: '%s'\n",buf->buf);
}

typedef struct {
#ifdef DEBUG
	char *name;
#endif
	int timeout_counter; /* number of select() timeouts before we continue 
								we don't use a time here, because the system time
								might change during boot
								*/
	pid_t pid;
	int stdout;
	Tbuf bufout;
} Tchild;

static void start_child(Tchild * child, char *script, char *arg1)
{
	int pout[2];
	pid_t pid;
	/* filedes[0]  is  for reading, filedes[1] is for writing. */
#ifdef DEBUG
	child->name = strdup(script);
	printf("start_child, set name to %s\n",child->name);
#endif
	pipe(pout);
	child->stdout = pout[0];
	pid = fork();
	if (0 == pid) {
		char *args[3];
		args[0] = script;
		args[1] = arg1;
		args[2] = NULL;
		printf("executing %s %s\n", script, arg1);
		dup2(pout[1], STDOUT_FILENO);
		dup2(pout[1], STDERR_FILENO);
		close(pout[0]);

		if (execv(script, args) == -1) {
			printf("error %d occurred when starting %s\n", errno, script);
			perror(NULL);
		}
		exit(errno);
	}
	close(pout[1]);
	child->pid = pid;
}
/* single childs do not need buffering of their output, and this
way they can be interactive as well */
static void start_single_child(char *script, char *arg1) {
	pid_t pid;
	int status;		
	pid = fork();
	if (0 == pid) {
		char *args[3];
		args[0] = script;
		args[1] = arg1;
		args[2] = NULL;
		printf("executing %s %s\n",script,arg1);
		if (execv(script, args) == -1) {
			printf("error %d occurred starting %s\n",errno,script);
			printf(strerror(errno));
		}
		exit(errno);
	}

	DEBUG_MSG("waiting for %s with pid=%d\n",scripts[i],children[i]);
	waitpid(pid, &status, 0);
}
static int child_still_running(Tchild children[], int numchilds)
{
	int i;
#ifdef DEBUG
	printf("running ");
	for (i = 0; i < numchilds; i++) {
		if (children[i].stdout > 0 || children[i].pid > 0)
			printf("%s %s%s ", children[i].name, children[i].stdout > 0 ? "s":"", children[i].pid > 0?"p":"");
	}
	printf("\n");
#endif
	
	for (i = 0; i < numchilds; i++) {
		if ((children[i].stdout > 0 || children[i].pid > 0) && children[i].timeout_counter < 100)
			return 1;
	}
	return 0;
}

static int run_children(Tchild * child, int numchilds)
{
	int i;
	fd_set inputs;
	DEBUG_MSG("run_children, have %d children\n",numchilds);
	
	while (child_still_running(child, numchilds)) {
		/* as long as at least one child is running */
		int res;
		struct timeval timeout;
		timeout.tv_sec = 0;
		timeout.tv_usec = 100000;	/* 0.1 seconds */

		FD_ZERO(&inputs);
		for (i = 0; i < numchilds; i++) {
			if (child[i].stdout > 0) {
				FD_SET(child[i].stdout, &inputs);
			}
		}
		res = select(FD_SETSIZE, &inputs, NULL, NULL, &timeout);
		if (res == 0) {
			/* timeout, test for stopped children, and re-run loop */
			int status;
			for (i = 0; i < numchilds; i++) {
				child[i].timeout_counter++;
				if (child[i].pid > 0) {
					DEBUG_MSG("select timeout, testing exit status of child %d at pid %d\n",i,child[i].pid);
					if (0 != waitpid(child[i].pid, &status, WNOHANG)) {
						/* a return value other than 0 means that the child has exited */
						child[i].pid = -1;
						child[i].timeout_counter = 90;
						DEBUG_MSG("child %d has exited\n", i);
					}
				}
			}
		} else if (res == -1) {
			/* error condition */
			printf("select returns an error\n");
			perror(NULL);
		} else {
			DEBUG_MSG("we should have data on %d descriptors!\n", res);
			for (i = 0; i < numchilds; i++) {
				char buffer[255];
				if (child[i].stdout > 0 && FD_ISSET(child[i].stdout, &inputs)) {
					size_t num = read(child[i].stdout, buffer, 255);
					if (((int)num) == -1) {
						DEBUG_MSG("read child %d returned an error: %s\n",i,strerror(errno));
					} else if (num == 0) {
						int status;
						DEBUG_MSG("closing stdout for child %d\n", i);
						close(child[i].stdout);
						child[i].stdout = -1;
						if (child[i].pid > 0 && 0 != waitpid(child[i].pid, &status, WNOHANG)) {
							/* a return value other than 0 means that the child has exited */
							child[i].pid = -1;
							DEBUG_MSG("child %d has exited\n", i);
						}
					} else {
						DEBUG_MSG("process %d, appending %d bytes from stdout\n", i, num);
						append2buf(&child[i].bufout, buffer, num);
					}
				}
				if (child[i].stdout == -1 && child[i].bufout.buf != NULL) {
					DEBUG_MSG("print child %d output, %d bytes:\n",i,child[i].bufout.len);
					printf("%s", child[i].bufout.buf);
					free(child[i].bufout.buf);
					child[i].bufout.buf = NULL;
				}
			}
		}
	}
	for (i = 0; i < numchilds; i++) {
		if (child[i].stdout > 0) {
			DEBUG_MSG("timeout, closing stdin for child %d\n",i);
			close(child[i].stdout);
		}
	}
	return 1;
}
/* free a NULL terminated array of strings, and all strings */
static void free_strarr(char **arr)
{
	if (arr) {
		char **tmp = arr;
		while (*tmp) {
			free(*tmp);
			tmp++;
		}
		free(arr);
	}
}
/* count the number of strings in a NULL terminated array */
static unsigned int count_strarr(char **arr)
{
	char **tmp = arr;
	while (*tmp != NULL)
		tmp++;
	return (tmp - arr);
}

/* return the suffix (the actual command name) from
a string; does NOT allocate new memory !!!!!!!!! */
static char *get_suffix(char *string)
{
	/* the suffix is everything after the /etc/rc?.d/??? */
	return string + 14;
}
/* tests if the provided suffix is present in the provided NULL terminated array */
static unsigned short int strarr_has_suffix(char **arr, const char *suffix)
{
	char **tmp = arr;
	while (*tmp) {
		if (strcmp(get_suffix(*tmp), suffix) == 0)
			return 1;
		tmp++;
	}
	return 0;
}
/* removes any occurence of the suffix from the NULL terminated array */
static char **remove_suffix_from_strarr(char **arr, char *suffix)
{
	int i = 0, j = 0;
	while (arr[i] != NULL) {
		if (strcmp(get_suffix(arr[i]), suffix) == 0) {
			free(arr[i]);
		} else {
			if (i != j)
				arr[j] = arr[i];
			j++;
		}
		i++;
	}
	if (i != j) {
		arr[j] = NULL;
	}
	return arr;
}

/*static unsigned short int strarr_has_string(char **arr, const char *string) {
	char **tmp = arr;
	while (*tmp) {
		if (strcmp(*tmp, string)==0) return 1;
		tmp++;
	}
	return 0;
}*/

/* if max == -1 means unlimited */
static char **duplicate_strarr(char **arr, signed int max)
{
	char **newarr, **tmp1 = arr, **tmp2;
	unsigned int count;

	count = count_strarr(arr);
	if (max > 0 && max < (int) count) {
		count = max;
	}
	newarr = tmp2 = malloc((count + 1) * sizeof(char *));
	while (tmp1 - arr < (int) count) {
/*		DEBUG_MSG("duplicated %s\n", *tmp1);*/
		*tmp2 = strdup(*tmp1);
		tmp2++;
		tmp1++;
	}
	*tmp2 = NULL;
	return newarr;
}

static char **return_scripts_sorted(const char *runleveldir, const char *startchar)
{
	glob_t globbuf;
	int ret;
	char *pattern = malloc(14 * sizeof(char));
	pattern[0] = '\0';
	pattern = strncat(pattern, runleveldir, 14);
	pattern = strncat(pattern, startchar, 14);
	pattern = strncat(pattern, "*", 14);
	DEBUG_MSG("searching for %s\n", pattern);
	ret = glob(pattern, 0, NULL, &globbuf);
	free(pattern);
	if (0 == ret) {
		char **retval = duplicate_strarr(globbuf.gl_pathv, -1);
		globfree(&globbuf);
		return retval;
	}
	return NULL;
}

static unsigned short int dir_exists(const char *path)
{
	struct stat sb;
	if (0 == stat(path, &sb) && S_ISDIR(sb.st_mode)) {
		return 1;
	}
	return 0;
}

static void run_batch(char **scripts, int len,char *arg1)
{
	if (len==1) {
		DEBUG_MSG("start single child %s\n",scripts[0]);
		start_single_child(scripts[0], arg1);
	} else {
		unsigned short int i,r=0;
		Tchild *children;
		DEBUG_MSG("run_batch with %d children and argument %s\n",len,arg1);
		children = malloc(len * sizeof(Tchild));
		memset(children, 0, len * sizeof(Tchild));
		/* first start all scripts */
		for (i = 0; i < len; i++) {
			if (scripts[i]) {
				DEBUG_MSG("r=%d, i=%d, about to call start_child for %s %s\n",r,i,scripts[i],arg1);
				start_child(&children[r], scripts[i], arg1);
				r++;
			}
		}
	
		/* now wait for all scripts to finish */
		run_children(children, r);
	#ifdef DEBUG
		DEBUG_MSG("finished batch, free children\n");
		for (i = 0; i < r; i++) {
			free(children[i].name);
		}
	#endif
		free(children);
	}
}

static char *build_runleveldir(char *level)
{
	if (strlen(level) == 1) {
		char *retval = malloc(12);
		retval[0] = '\0';
		snprintf(retval, 12, "/etc/rc%s.d/", level);
		return retval;
	}
	printf("ERROR: runlevel had strlen() > 1\n !!");
	return NULL;
}

int main(int argc, char **argv)
{
	char *cur_runlevel, *prev_runlevel;
	char **Sscripts = NULL, **Kscripts = NULL, **prevSscripts = NULL;
	char *runleveldir, *arg1;
#ifdef PRINTTIME
	time_t tm;
	tm = time(NULL);
	printf("started at %s", ctime(&tm));
#endif
	cur_runlevel = getenv("RUNLEVEL");
	prev_runlevel = getenv("PREVLEVEL");	/* NULL means there was no previous level  */
	printf("from environment: runlevel=%s, prevlevel=%s\n", cur_runlevel, prev_runlevel);

	/* test if the runlevel was passed as the first argument */
	if (argc >= 2) {
		cur_runlevel = argv[1];
	}

	/* test if runlevel is valid 0-9 or S */
	/* needs to be done */

	/* find the runleveldir */
	runleveldir = build_runleveldir(cur_runlevel);
	if (!dir_exists(runleveldir)) {
		printf("ERROR: runleveldir %s does not exist!\n", runleveldir);
	}
	/* return 'start' script for this runlevel */
	Sscripts = return_scripts_sorted(runleveldir, "S");
	/* return 'stop' scripts for this runlevel, sort by sequence number 
	if there was no previous level, there is nothing to stop
	*/
	if (prev_runlevel && strcmp(prev_runlevel,"N")!=0) {
		Kscripts = return_scripts_sorted(runleveldir, "K");
	}

	/* run batches of scripts with the same sequence number parallel */
	if (Kscripts) {
		unsigned short int start = 0, len = 1, i;
		for (i = 1; Kscripts[i] != NULL; i++) {
			DEBUG_MSG("comparing sequence number for %s and %s\n", Kscripts[start], Kscripts[i]);
			if (strncmp(Kscripts[start], Kscripts[i], 14) == 0) {
				len++;
				DEBUG_MSG("the same sequence number! len=%d\n", len);
			} else {
				run_batch(&Kscripts[start], len, "stop");
				start = i;
				len = 1;
			}
		}
		run_batch(&Kscripts[start], len, "stop");
	}

	if (prev_runlevel && strcmp(prev_runlevel,"S")!=0 && strcmp(prev_runlevel,"N")!=0) {
		/* if this suffix is already started in the previous runlevel, and not 
		   stopped in this runlevel, we don't need to start it anymore. previous
		   runlevel NULL or "S" or "N" don't need checking */
		char **tmp;
		char *prevrunleveldir = build_runleveldir(prev_runlevel);
		prevSscripts = return_scripts_sorted(prevrunleveldir, "S");
		if (prevSscripts) {
			tmp = prevSscripts;
			while (*tmp) {
				char *suffix = get_suffix(*tmp);
				if (strarr_has_suffix(Sscripts, suffix) && !strarr_has_suffix(Kscripts, suffix)) {
					DEBUG_MSG("removing %s from Sscripts, it was already started by the previous runlevel, and not stopped by this level\n",suffix);
					Sscripts = remove_suffix_from_strarr(Sscripts, suffix);
				}
				tmp++;
			}
		}
	}

	/* run batches of scripts with the same sequence number parallel
	for runlevel 0 or 6 we should 'stop' everything */
	if (strcmp(cur_runlevel,"0")==0 || strcmp(cur_runlevel,"6")==0) {
		arg1 = "stop";
	} else {
		arg1 = "start";
	}
	
	if (Sscripts) {
		unsigned short int start = 0, len = 1, i;
		for (i = 1; Sscripts[i] != NULL; i++) {
			DEBUG_MSG("comparing sequence number for %s and %s\n", Sscripts[start], Sscripts[i]);
			if (strncmp(Sscripts[start], Sscripts[i], 14) == 0) {
				len++;
				DEBUG_MSG("sequence number the same! len=%d\n", len);
			} else {
				run_batch(&Sscripts[start],len, arg1);
				start = i;
				len = 1;
			}
		}
		run_batch(&Sscripts[start], len, arg1);
	}
	free_strarr(Sscripts);
	free_strarr(Kscripts);
	free_strarr(prevSscripts);
	free(runleveldir);
#ifdef PRINTTIME
	tm = time(NULL);
	printf("finished at %s", ctime(&tm));
#endif
	exit(0);
}
