/*
 * $Header: /home/cvspublic/jakarta-commons/collections/src/java/org/apache/commons/collections/Bag.java,v 1.1 2001/08/29 15:28:07 jstrachan Exp $
 * $Revision: 1.1 $
 * $Date: 2001/08/29 15:28:07 $
 *
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Commons", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 */

package org.apache.commons.collections;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * Makes a {@link Map} look like a {@link Collection}.
 *
 * @author Chuck Burdick
 **/
public class MapCollection implements Collection {
   private Map _map;

   public MapCollection() {
      setMap(null);
   }

   public MapCollection(Map m) {
      setMap(m);
   }

   /**
    * If the underlying {@link Map} does not already contain the given
    * object as a key, it will map the given object to null. The
    * object will appear in the keySet of the map.
    *
    * @return <code>true</code> if the map changed as a result of this
    * call
    **/
   public boolean add(Object o) {
      boolean result = false;
      if (!getMap().containsKey(o)) {
         getMap().put(o, null);
         result = true;
      }
      return result;
   }

   /**
    * If the underlying {@link Map} does not already contain the given
    * objects as keys, it will map them to null. The objects will
    * appear in the keySet of the map.
    *
    * @return <code>true</code> if the map changed as a result of this
    * call
    **/
   public boolean addAll(Collection c) {
      boolean result = false;
      boolean changed = false;
      Iterator i = c.iterator();
      while (i.hasNext()) {
         changed = add(i.next());
         result = result || changed;
      }
      return result;
   }

   public void clear() {
      getMap().clear();
   }

   /**
    * Returns <code>true</code> if the map contains a mapping for the
    * specified key.
    **/
   public boolean contains(Object o) {
      return getMap().containsKey(o);
   }

   /**
    * Returns <code>true</code> if the map contains a mapping for the
    * all keys specified in the given {@link Collection}.
    **/
   public boolean containsAll(Collection c) {
      return getMap().keySet().containsAll(c);
   }

   public boolean equals(Object o) {
      boolean result = false;
      if (o instanceof Map) {
         result = getMap().equals(o);
      } else if (o instanceof MapCollection) {
         Map other = ((MapCollection)o).getMap();
         result = getMap().equals(other);
      } else {
         result = super.equals(o);
      }
      return result;
   }

   public int hashCode() {
      return getMap().hashCode();
   }

   public boolean isEmpty() {
      return getMap().isEmpty();
   }

   /**
    * Returns an {@link Iterator} over the keys of the map.
    **/
   public Iterator iterator() {
      return getMap().keySet().iterator();
   }

   /**
    * Removes the mapping from the underlying {@link Map}.
    *
    * @return <true> if the map changed as a result of the call
    **/
   public boolean remove(Object o) {
      return getMap().keySet().remove(o);
   }

   /**
    * Removes all mappings from the underlying {@link Map} that are
    * specified by the provided {@link Collection}.
    *
    * @return <true> if the map changed as a result of the call
    **/
   public boolean removeAll(Collection c) {
      return getMap().keySet().removeAll(c);
   }

   public boolean retainAll(Collection c) {
      return getMap().keySet().retainAll(c);
   }

   public int size() {
      return getMap().size();
   }

   /**
    * Returns an array of the keys in the map.
    **/
   public Object[] toArray() {
      return getMap().keySet().toArray();
   }

   /**
    * Returns an array containing all of the keys of the map whose
    * runtime type is that of the specified array.
    **/
   public Object[] toArray(Object[] a) {
      return getMap().keySet().toArray(a);
   }

   /**
    * Returns the underlying {@link Map}.
    *
    * @see #setMap
    **/
   public Map getMap() {
      return _map;
   }

   /**
    * Sets the {@link Map} that backs this adapter.
    *
    * @see #MapCollection(Map)
    **/
   public void setMap(Map m) {
      _map = m;
   }
}






