/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE file.
 */
package org.apache.james.testing;

import org.apache.avalon.framework.configuration.*;
import org.apache.james.userrepository.*;
import junit.framework.*;
import java.sql.*;

/**
 * A valid JUnit test case for testing a UsersJDBCRepository.
 * This test-case tests the implementation against a number of
 * database servers, using different drivers.
 *
 * The actual database configuration will need to be configured
 * in the {@link #suite() } method.
 *
 * The actual tests are defined in the TestUsersRepository
 *
 * <br>TODO - should load the database configuration from a properties file.
 *
 * @author Darrell DeBoer <dd@bigdaz.com>
 */
public class TestUsersJDBCRepository extends TestUsersRepository
{
    protected String m_driver;
    protected String m_dbUrl;
    protected String m_user;
    protected String m_pwd;
    protected boolean m_drop;

    /**
     * Builds a suite of tests for JDBC Repository, connecting to different databases,
     * using different drivers.
     * The tests are run twice for each database/driver combo, the first time
     * the JamesUsers table is dropped prior to running the test, representing
     * a new installation, and the second time the existing table is used.
     */
    public static Test suite() 
    {
        TestSuite suite = new TestSuite();
        
        // MySQL odbc connection
        suite.addTest(new TestUsersJDBCRepository("testBasic", "sun.jdbc.odbc.JdbcOdbcDriver", "jdbc:odbc:mail-mysql", "sa", "password", true));
        suite.addTest(new TestUsersJDBCRepository("testBasic", "sun.jdbc.odbc.JdbcOdbcDriver", "jdbc:odbc:mail-mysql", "sa", "password", false));
        
        // MySQL native jdbc connection
        suite.addTest(new TestUsersJDBCRepository("testBasic", "org.gjt.mm.mysql.Driver", "jdbc:mysql://192.168.0.1/mail", "sa", "password", true));
        suite.addTest(new TestUsersJDBCRepository("testBasic", "org.gjt.mm.mysql.Driver", "jdbc:mysql://192.168.0.1/mail", "sa", "password", false));

        // Microsoft SQL Server connection ( using ODBC).
        /*
        These tests fail due to some strange padding issue with 
        the jdbc/odbc interaction... (Column data is padded to full width),
        so the user names come back like 
            "User2                           "
        This is how the values end up in the database - who knows why...?
        suite.addTest(new TestUsersJDBCRepository("testBasic", "sun.jdbc.odbc.JdbcOdbcDriver", "jdbc:odbc:mail", "sa", "password", true));
        suite.addTest(new TestUsersJDBCRepository("testBasic", "sun.jdbc.odbc.JdbcOdbcDriver", "jdbc:odbc:mail", "sa", "password", false));
        */
        
        // Microsoft SQL Server connection ( using Inet Sprinta driver).
        suite.addTest(new TestUsersJDBCRepository("testBasic", "com.inet.tds.TdsDriver", "jdbc:inetdae7:192.168.0.1?database=mail", "sa", "password", true));
        suite.addTest(new TestUsersJDBCRepository("testBasic", "com.inet.tds.TdsDriver", "jdbc:inetdae7:192.168.0.1?database=mail", "sa", "password", false));
        
        // Oracle 8i with "thin" driver.
        suite.addTest(new TestUsersJDBCRepository("testBasic", "oracle.jdbc.driver.OracleDriver", "jdbc:oracle:thin:@192.168.0.1:1521:csl", "sa", "password", true));
        suite.addTest(new TestUsersJDBCRepository("testBasic", "oracle.jdbc.driver.OracleDriver", "jdbc:oracle:thin:@192.168.0.1:1521:csl", "sa", "password", false));

        return suite;
    }

    public TestUsersJDBCRepository(String name) {
        super(name);
    }

    /**
     * Builds a JDCB repository test for a specified database configuration.
     */
    TestUsersJDBCRepository(String name, String driver, String dbUrl, String user, String pwd, boolean drop)
    {
        super(name);
        m_driver = driver;
        m_dbUrl = dbUrl;
        m_user = user;
        m_pwd = pwd;
        m_drop = drop;
    }

    /**
     * Initialises the UserRepository using the specified database configuration.
     * If <code>_drop = false</code>, the JamesUser table will be dropped before
     * running the test.
     */
    protected void setUp() throws Exception 
    {
        // Drop the JamesUsers table, if required.
        if ( m_drop ) {
            Connection conn;
            Class.forName(m_driver);
            if ( m_user == null ) {
                conn = DriverManager.getConnection(m_dbUrl);
            }
            else {
                conn = DriverManager.getConnection(m_dbUrl, m_user, m_pwd);
            }

            try {
                conn.createStatement().execute("DROP TABLE JamesUsers");
                System.out.println("Dropped \'JamesUsers\' table.");
            }
            catch ( SQLException sqlExc ) {
                // ignore exceptions in dropping table.
            }
            finally {
                conn.close();
            }
        }

        // Create the driver config.
        DefaultConfiguration driverConfig = new DefaultConfiguration("driver", "UserRepositoryTest.destination.driver");
        driverConfig.setAttribute("class", m_driver);

        // Create the datasource config.
        DefaultConfiguration dburlConfig = new DefaultConfiguration("dburl", "UserRepositoryTest.destination.datasource.dburl");
        dburlConfig.setValue(m_dbUrl);
        DefaultConfiguration userConfig = new DefaultConfiguration("user", "UserRepositoryTest.destination.datasource.user");
        userConfig.setValue(m_user);
        DefaultConfiguration pwdConfig = new DefaultConfiguration("password", "UserRepositoryTest.destination.datasource.password");
        pwdConfig.setValue(m_pwd);

        DefaultConfiguration datasourceConfig = new DefaultConfiguration("datasource", "UserRepository.destination.datasource");
        datasourceConfig.addChild(dburlConfig);
        datasourceConfig.addChild(userConfig);
        datasourceConfig.addChild(pwdConfig);

        // Create the destination config to hold these
        DefaultConfiguration destConfig = new DefaultConfiguration("destination", "UserRepositoryTest.destination");
        destConfig.addChild(driverConfig);
        destConfig.addChild(datasourceConfig);

        DefaultConfiguration config = new DefaultConfiguration("UserRepositoryTest", "UserRepositoryTest");
        config.addChild(destConfig);

        UsersJDBCRepository jdbcRepos = new UsersJDBCRepository();
        jdbcRepos.configure(config);
        jdbcRepos.initialize();
        m_userRepos = jdbcRepos;

    }
}
