/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE file.
 */
package org.apache.james.testing;

import junit.framework.*;
import org.apache.james.userrepository.*;
import org.apache.james.services.*;
import java.util.*;

/**
 * A JUnit TestCase for test a UsersRepository implementation.
 * Subclasses should configure and initialise the _userRepos
 * with a valid UsersRepository implementation in their <code>setup</code>
 * method.
 *
 * @author Darrell DeBoer <dd@bigdaz.com>
 */
public abstract class TestUsersRepository extends TestCase
{
    protected UsersRepository m_userRepos;

    public TestUsersRepository(String name) {
        super(name);
    }

    public void testBasic() {
        // Clean out users, and make sure they've gone.
        int count = m_userRepos.countUsers();
        int removed = removeAllUsers();
        assertEquals(count, removed);

        checkCount(0);

        // Add a bunch of new users.
        List userList = new LinkedList();
        userList.add("User1");
        userList.add("User2");
        userList.add("User3");
        
        Iterator userIterator = userList.iterator();
        while ( userIterator.hasNext() ) {
            assertTrue(addUser((String)userIterator.next(), "password"));
        }

        // Make sure they are there.
        checkCount( userList.size() );

        // Do some checks on every user.
        userIterator = userList.iterator();
        while ( userIterator.hasNext() ) {
            String username = (String)userIterator.next();
            checkContains(username);

            assertNotNull( m_userRepos.getUserByName(username));
        }

        // Check case sensitivity on lookups.
        checkContains("uSEr2", true);
        checkNotContains("uSEr2", false);

        // Make sure we can't add another user the same.
        //assertTrue( ! (addUser("User1", "password")) );
        // Can't duplicate users with different case.
        //assertTrue( ! (addUser("uSEr1", "password")) );

        // Check the list of users.
        Iterator userNames = m_userRepos.list();
        while ( userNames.hasNext() ) {
            assertTrue(userList.contains((String)userNames.next()));
        }

        // Check the password test.
        assertTrue(m_userRepos.test("User1", "password"));
        assertTrue(! m_userRepos.test("User2", "wrong"));

        // Check adding, getting, modifying, getting, removing, getting.
        // Add and test.
        User user = null;
        addUser("updateUser", "password");
        assertTrue(m_userRepos.test("updateUser", "password"));

        // Get, update and test
        user = m_userRepos.getUserByName("updateUser");
        assertNotNull(user);
        user.setPassword("newPassword");
        m_userRepos.updateUser(user);

        checkContains("updateUser");
        assertTrue(m_userRepos.test("updateUser", "newPassword"));
        user = null;

        // Remove and test.
        m_userRepos.removeUser("updateUser");
        user = m_userRepos.getUserByName("udpateUser");
        checkNotContains("updateUser");
        assertNull(user);
        assertTrue(! m_userRepos.test("updateUser", "newPassword"));
    }

    private boolean addUser(String name, String pwd) {
        DefaultJamesUser user = new DefaultJamesUser(name, "SHA");
        user.setPassword(pwd);
        return m_userRepos.addUser(user);
    }

    private int removeAllUsers()
    {
        int count = 0;
        Iterator userList = m_userRepos.list();
        while ( userList.hasNext() ) {
            String userName = (String)userList.next();
            m_userRepos.removeUser(userName);
            count++;
        }

        return count;
    }

    private void checkCount(int expected)
    {
        int count = m_userRepos.countUsers();
        assertEquals(expected, count);
    }

    private void checkContains(String name)
    {
        checkContains(name, false, false);
    }

    private void checkNotContains(String name)
    {
        checkContains(name, false, true);
    }

    private void checkContains(String name, boolean ignoreCase)
    {
        checkContains(name, ignoreCase, false);
    }

    private void checkNotContains(String name, boolean ignoreCase)
    {
        checkContains(name, ignoreCase, true);
    }

    private void checkContains(String name, boolean ignoreCase, boolean expectMissing)
    {
        boolean found;
        if ( ignoreCase ) {
            found = m_userRepos.containsCaseInsensitive(name);
        }
        else {
            found = m_userRepos.contains(name);
        }

        if ( expectMissing ) {
            assertTrue(!found);
        }
        else {
            assertTrue(found);
        }
    }

}
