/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE file.
 */
package org.apache.james.transport.mailets;
/**
 * ForwardAsAttachment.java
 *
 *
 * Created: Thu Oct 11 16:24:23 2001
 *
 * @author <a href="mailto:okidz@pindad.com">Oki DZ</a>
 * @version 1.0
 *
 * <mailet match="All" class="ForwardAsAttachment">
 *   <!-- default: -->
 *   <sendingAddress>mail-server@yourdomain.com</sendingAddress>
 *   <!-- default: -->
 *   <receivingAddress>postmaster@yourdomain.com</receivingAddress>
 *   <subject></subject>
 *   <notice></notice>
 * </mailet>
 *
 */
import java.lang.*;
import java.io.*;
import java.util.*;
import java.net.*;

import javax.mail.*;
import javax.mail.internet.*;

import org.apache.avalon.framework.component.*;

import org.apache.mailet.*;
import org.apache.james.*;
import org.apache.james.core.*;
import org.apache.james.util.*;
import org.apache.james.services.*;

public class ForwardAsAttachment extends GenericMailet {

    protected MailAddress sender;
    protected MailAddress recipient; 
    protected String subjectText;
    protected String noticeText;
    protected MailServer mailServer;

    private InetAddress hostAddress;
    private MimeMessage message;
    
    public void init() throws MessagingException {
	try {
	    hostAddress = InetAddress.getLocalHost();
	    ComponentManager componentManager = (ComponentManager)
		getMailetContext()
		.getAttribute(Constants.AVALON_COMPONENT_MANAGER);
	    mailServer = (MailServer)
		componentManager.lookup("org.apache.james.services.MailServer");
	    if (getInitParameter("sendingAddress") == null) {
		sender = new 
		    MailAddress("mail-server@" + hostAddress.getHostName());
	    } else {
		sender = new MailAddress(getInitParameter("sendingAddress"));
	    }
	    if (getInitParameter("receivingAddress") != null) {
		recipient = new
		    MailAddress(getInitParameter("receivingAddress"));
	    } else {
		recipient = getMailetContext().getPostmaster();
	    }
	    if (getInitParameter("subject") == null) {
		subjectText = "Forwarded message.";
	    } else {
		subjectText = getInitParameter("subject");
	    }
	    if (getInitParameter("notice") == null) {
		noticeText = "[Notice text here]";
	    } else {
		noticeText = getInitParameter("notice");
	    }
	} catch(Exception e) {
	    throw new MessagingException(e.getMessage());
	}
    }  

    public void service(Mail mail) throws MessagingException {
	try {	
	    Mail forward = newForward(mail);
	    mailServer.sendMail(forward);
	    mail.setState(Mail.GHOST);
	} catch(Exception e) {
	    throw new MessagingException(e.getMessage());
	}  
    }

    protected Mail newForward(Mail mail) throws MessagingException {
	message = mail.getMessage();
	Collection recipients = new Vector();
	recipients.add(recipient.toString());
	Session session = Session
	    .getDefaultInstance(System.getProperties(), null);

	String dateString = RFC822DateFormat.toString(new Date());
	StringWriter headbuff = new StringWriter();
	PrintWriter headprint = new PrintWriter(headbuff, true);
	headprint.println("ReturnPath: <" + sender.toString() + ">");
	headprint.println("Received: from " + hostAddress.getHostName() +
			  " ([" + hostAddress.getHostAddress() + "])");
	headprint.println("          by " + hostAddress.getHostName() +
			  " (" + getMailetContext().getServerInfo() + ")");
	headprint.println("          for <" + recipient.toString() + ">; " +
			  dateString);
	headprint.println("Date: " + dateString);
	headprint.println("From: " + sender.toString());
	headprint.println("To: " + recipient.toString());
	headprint.println("Subject: " + subjectText);
	headprint.println("MIME-Version: 1.0");
	//
	StringWriter sout = new StringWriter();
	PrintWriter out = new PrintWriter(sout, true);
	out.println("Notice: ");
	out.println(noticeText);
	out.println();
	out.println("Original message details: ");
	if (message.getSubject() != null) {
	    out.println("\tSubject: " + message.getSubject());
	} 
	if (message.getSentDate() != null) {
	    out.println("\tDate: " + message.getSentDate());
	} 
	out.println("\tFrom: " + mail.getSender());
	out.print("\tTo: ");
	for (Iterator i = mail.getRecipients().iterator(); i.hasNext();) {
	    out.print(i.next().toString());
	    if (i.hasNext()) out.print(", ");
	} 
	out.println();
	int size = message.getSize();
	out.println("\tSize: " + size);
	MailImpl mc = null;
	MimeMessage msg = new MimeMessage(session);
	
	StringBuffer msgbuff = new StringBuffer();

	String headfilename, origfilename;
	File headfile = null;
	File file = null;
	OutputStream ostream = null;
	InputStream istream = null;

	MimeMultipart multipart;
	MimeBodyPart bodypart;
	PrintStream printstream;
	LineNumberReader lineReader;
	//
	String header = "";
	String body = "";
	String textInsert = sout.toString();
	Object o;

	try {
	    multipart = new MimeMultipart();
	    // new insert
	    bodypart = new MimeBodyPart();
	    bodypart.setContent(textInsert, "text/plain");
	    multipart.addBodyPart(bodypart);
	    o = message.getContent();
	    if (o instanceof Multipart) {
		// orig. attachment(s)
		MimeMultipart mp = (MimeMultipart) o;
		for (int i = 0, n = mp.getCount(); i < n; i++) {
		    MimeBodyPart bp = (MimeBodyPart) mp.getBodyPart(i);
		    multipart.addBodyPart(bp);
		} 
		file = File.createTempFile("forwardAsAttachment", null);
		origfilename = file.getAbsolutePath();
		ostream = new FileOutputStream(file);
		multipart.writeTo(ostream);
		ostream.close();
		// get boundary line
		istream = new FileInputStream(origfilename);
		lineReader = new 
		    LineNumberReader(new InputStreamReader(istream));
		headprint.println("Content-Type: MULTIPART/MIXED; BOUNDARY=\"" +
				  lineReader.readLine().substring(2) + "\"");
		headprint.println();
      		header = headbuff.toString();
		istream.close();
	    } else {		
		// orig. text
		bodypart = new MimeBodyPart();
		bodypart.setContent((String) message.getContent(),"text/plain");
		multipart.addBodyPart(bodypart);
		// 
		file = File.createTempFile("forwardAsAttachment", null);
		origfilename = file.getAbsolutePath();
		ostream = new FileOutputStream(file);
		multipart.writeTo(ostream);
		ostream.close();
		// get boundary line
		istream = new FileInputStream(origfilename);
		lineReader = new 
		    LineNumberReader(new InputStreamReader(istream));
		headprint.println("Content-Type: MULTIPART/MIXED; BOUNDARY=\"" +
				  lineReader.readLine().substring(2) + "\"");
		headprint.println();
		header = headbuff.toString();
		istream.close();
	    }
	    msgbuff.append(header);
	    istream = new BufferedInputStream(new FileInputStream(origfilename));
	    for (int i = 0; (i = istream.read()) != -1;) {
		msgbuff.append((char) i);
	    } 
	    istream.close();
	    msg.setText(msgbuff.toString());
	    mc = new MailImpl(mailServer.getId(), sender, recipients);
	    mc.setMessage(msg);
	} catch(Exception e) {
	    log(e.getMessage());
	    throw new MessagingException(e.getMessage());
	} finally {
	    try {
		file.delete();
	    } catch(Exception e) {
		e.printStackTrace();
	    }
	}
	return mc;
    }

    public String getMailetInfo() {
	return "A mailet for forwarding messages; converting them to attachments.";
    }

    public String getMailetName() {
	return getClass().getName();
    }

}
