/*
 *  Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 *  This software is published under the terms of the Apache Software License
 *  version 1.1, a copy of which has been included with this distribution in
 *  the LICENSE file.
 */
package org.apache.james.transport.mailets;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import javax.mail.MessagingException;
import javax.mail.internet.MimeMessage;

import org.apache.mailet.GenericMailet;
import org.apache.mailet.Mail;

/**
 *  Removes a list of specified headers from a message.
 *
 * Example:
 *
 * <mailet match="All" class="org.apache.james.transport.mailets.RemoveHeader">
 *     <header-1>Return-Path</header-1>
 *     <header-2>X-A-Custom-Header</header-2>
 * </mailet>
 *
 */
public class RemoveHeader extends GenericMailet
{

    /**  The unique list of headers to remove from messages. */
    protected Set headers;


    /**
     * Return a string describing this mailet.
     *
     * @return    a string describing this mailet
     */
    public String getMailetInfo() {
        return "RemoveHeader Mailet";
    }


    /**
     *  Initalize the mailet.
     */
    public void init() {
        headers = new HashSet();
        String h ;
        int i = 1 ;
        while( ( h = getInitParameter( "header-"+i ) ) != null ) {
            headers.add( h );
            i++ ;
        }
    }


    /**
     *  Removes the <code>headers</code> from the given message.
     *
     * @param  mail                    the mail being processed
     * @exception  MessagingException  if an error arises during message processing
     */
    public void service( Mail mail ) throws MessagingException {
        MimeMessage mesg = mail.getMessage();
        String header;
        try {
            for ( Iterator i = headers.iterator(); i.hasNext();  ) {
                header = (String)i.next();
                mesg.removeHeader( header );
                log( "Removed Header '"+header+"' from mail.") ;
            }
            mesg.saveChanges();
        }
        catch ( MessagingException me ) {
            log( me.getMessage() );
        }
    }

}

