/*
 * Copyright (c) 2000, Niklas Mehner 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * 
 *   - Redistributions of source code must retain the above copyright 
 *     notice, this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in 
 *     the documentation and/or other materials provided with the 
 *     distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */ 
package org.j3de.behavior.impl;

import java.util.Enumeration;

import javax.media.j3d.BoundingSphere;
import javax.media.j3d.TransformGroup;       
import javax.media.j3d.Transform3D; 
import javax.media.j3d.WakeupOnElapsedFrames;

import javax.vecmath.AxisAngle4d;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;
import javax.vecmath.Matrix3d;

import org.w3c.dom.Node; 
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import org.j3de.behavior.MovementBehavior;    
import org.j3de.util.Configurable; 
import org.j3de.util.ConfigHelper;   
import org.j3de.util.ConfigurationException;

public class DefaultMovementBehavior extends MovementBehavior implements Configurable { 
  private transient Transform3D    moveTransform;
  private transient Transform3D    sensorTransform;
  private transient TransformGroup transformGroup;
  private transient long timestamp;
  private transient WakeupOnElapsedFrames wakeup;
  private transient Vector3d       translation;
  private transient Matrix3d       rotationMatrix;
  private transient AxisAngle4d    axisAngle;
  
  private double    speedFactor;   
  
  public DefaultMovementBehavior() {
    speedFactor = 5.0;
  }
  
  public void setTransformGroup(TransformGroup group) {
    this.transformGroup = group;   
  }   
  
  public void initialize() {
    setSchedulingBounds(new BoundingSphere(new Point3d(0.0,0.0,0.0), 100.0));
    
    moveTransform   = new Transform3D();
    sensorTransform = new Transform3D(); 
    translation    = new Vector3d();
    axisAngle      = new AxisAngle4d();
    rotationMatrix = new Matrix3d();
    
    wakeup = new WakeupOnElapsedFrames(0);
    wakeupOn(wakeup);
    timestamp = System.currentTimeMillis();
  }
                                                           
  private int frames = 0;
  private long frametimestamp = System.currentTimeMillis();
                                                          
  public void processStimulus(Enumeration criteria) {  
    double scale = (System.currentTimeMillis() - timestamp) / 1000.0;
    
    // read current transofrmations                                       
    getSensor().getRead(sensorTransform);   
    transformGroup.getTransform(moveTransform);
     
    // get values for translation and axisAngle
    sensorTransform.get(translation);       
    sensorTransform.get(rotationMatrix);
    axisAngle.set(rotationMatrix);
    
    // scale translation
    translation.scale(scale * speedFactor);
    sensorTransform.setTranslation(translation);
    
    // if angle is not zero : scale angle and set Rotation
    if (axisAngle.angle != 0) {
      axisAngle.angle *= scale;
      sensorTransform.set(axisAngle);
    }
    
    // apply transformation
    moveTransform.mul(sensorTransform);
                          
    transformGroup.setTransform(moveTransform);       
    
    timestamp = System.currentTimeMillis();    
     
    
    frames++;
    
    if (frames >= 100) {
      System.out.println("fps : " + (100000.0 / (double)(timestamp - frametimestamp) ));
      frametimestamp = timestamp;
      frames = 0;
    }
    wakeupOn(wakeup);
  }                                                       
  
  //////////////////////////////////////////////////////////
  // Implementation of Configurable interface ..          //
  //////////////////////////////////////////////////////////
  protected transient ConfigHelper helper;

  public void configure(Node node, Document nodeFactory) throws ConfigurationException  {
    if (helper == null)
      helper = new ConfigHelper(node, nodeFactory, false);       
      
    speedFactor = helper.getPropertyValue("speedfactor", 1.0, true);
  }    
  
  public Node configure(Document nodeFactory, String componentName) throws ConfigurationException {  
    Element component = ConfigHelper.newComponent(nodeFactory, componentName, this); 
    helper = new ConfigHelper(component, nodeFactory, true);
    configure(component, nodeFactory);
    return component;
  }
   
  public boolean isConfigurationChanged() {
    return helper.isConfigurationChanged();
  }   
   
  public void configurationSaved() { 
    helper.configurationSaved();
  }       

  
}