/*
 * Copyright (c) 2000, Niklas Mehner 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * 
 *   - Redistributions of source code must retain the above copyright 
 *     notice, this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in 
 *     the documentation and/or other materials provided with the 
 *     distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */ 
package org.j3de.input;

import javax.media.j3d.InputDevice;
import javax.media.j3d.Sensor;
import javax.media.j3d.Transform3D;

import javax.vecmath.Point3d;  
import javax.vecmath.Vector3d;  

import org.w3c.dom.Node;
import org.w3c.dom.Document;   

import org.j3de.exception.ExceptionHandler;
import org.j3de.util.AbstractComponent;   
import org.j3de.util.ConfigurationException; 

import Joystick;

/** This implements a Joystick-Input-Device using the joystick component by   
  * Satoshi Konno (available from <a href=http://www.cyber.koganei.tokyo.jp/vr/index.html"> 
  * http://www.cyber.koganei.tokyo.jp/vr/index.html </a>) unfortunatly the library is only available for
  * Windows. If anyone is willing to write this for Linux (or any other OS) 
  * or has already done so, please <a href="mailto:wombat@uni.de"> drop me a line </a> !
  * 
  * Currently the Joystick only allows to turn up/down, left/right and to move forward
  * or backward using the first two buttons. Someday I'll rewrite this, to make the
  * function-assignment more flexible ...
  */
public class JoystickInputDevice extends AbstractComponent implements InputDevice {  
  private int[]        BUTTONS = new int[0];
  private Sensor       sensor;
  private int          processingMode; 
  private Transform3D  rotY;  
  private Transform3D  rotZ;  
  private Transform3D  transform;  
  private Vector3d     translation;
  private int          deviceNumber;
  
  /** Turning-angle per second when value of joystick is at maximum. */
  private double       turnAngle;           
  
  /** error represents the minimum value the device is reacting to, else
    * it would be moving all the time ... */
  private double       error;  
   
  /** The Win32 function returns -1 for the z-Axis-Value of a 2-axis joysticks. This is value
    * should be ignored. */
  private boolean      ignoreZ; 
  private Joystick     joystick;  
  
  public JoystickInputDevice() {  
    this.sensor  = new Sensor(this);         
    processingMode = InputDevice.DEMAND_DRIVEN;   
    translation    = new Vector3d();
    transform      = new Transform3D();
    rotY           = new Transform3D();
    rotZ           = new Transform3D();
  }
    
  public boolean initialize() {  
    try {     
      joystick = new Joystick(deviceNumber);
      return true;
    } catch (Exception e) {
      ExceptionHandler.handleException(e);
      return false;
    }
  }
  
  public void setNominalPositionAndOrientation() {
    // TODO implement 
  }
  
  public void pollAndProcessInput() {             
    double xpos = joystick.getXPos();
    double ypos = joystick.getYPos();
    double zpos = joystick.getZPos();
    
    if (Math.abs(xpos) > error)  
      transform.rotY(-1.0 * xpos * turnAngle);     
    else
      transform.setIdentity();     
    
    if (Math.abs(ypos) > error) {  
      rotY.rotX(ypos * turnAngle);
      transform.mul(rotY);       
    }
   
    if ((Math.abs(zpos) > error) & (!ignoreZ)) { 
      rotZ.rotZ(zpos * turnAngle);
      transform.mul(rotZ); 
    }
    
    int buttons = joystick.getButtons();    
    
    if ((buttons & Joystick.BUTTON1) > 0) {
      translation.z = -1.0;
    } else if ((buttons & Joystick.BUTTON2) > 0) { 
      translation.z = 1.0;
    } else {
      translation.z =  0.0;
    }      
    
    transform.setTranslation(translation);                                     
    
    sensor.setNextSensorRead(System.currentTimeMillis(), transform, BUTTONS);
  }
  
  public void processStreamInput() {  
    // According to j3d documentation this should remain an empty method ...
  }
  
  public void close() {  
    // nothing to do ...
  }
  
  public int getProcessingMode() {                           
    return processingMode;
  }
  
  public void setProcessingMode(int mode) {   
    this.processingMode = mode; 
  }
  
  public int getSensorCount() {
    return 1;
  }
  
  public Sensor getSensor(int sensorIndex) {  
    return sensor;
  }      
  
  public void configure(Node node, Document nodeFactory)  throws ConfigurationException {  
    super.configure(node, nodeFactory);    
    
    turnAngle    = (Double.parseDouble(helper.getPropertyValue("TurnDegrees", "15", true)) / 360.0) * 2.0 * Math.PI;

    deviceNumber = helper.getPropertyValue("device", 0, true);  
    
    ignoreZ      = helper.getPropertyValue("ignoreZAxis", "true", true).equals("true"); 
    
    error        = Double.parseDouble(helper.getPropertyValue("calibrationError", "0.0", true)); 
    
  }
} 