/*
 * Copyright (c) 2000, Niklas Mehner 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions 
 * are met:
 * 
 *   - Redistributions of source code must retain the above copyright 
 *     notice, this list of conditions and the following disclaimer.
 * 
 *   - Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in 
 *     the documentation and/or other materials provided with the 
 *     distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR 
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR 
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS 
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */ 
package org.j3de.input;  

import java.awt.AWTEvent;
import java.awt.Dimension;
import java.awt.Toolkit;
import java.awt.event.MouseEvent;
import java.util.Enumeration;  
  
import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.InputDevice;
import javax.media.j3d.Sensor;
import javax.media.j3d.Transform3D;
import javax.media.j3d.WakeupCondition;
import javax.media.j3d.WakeupCriterion;
import javax.media.j3d.WakeupOnAWTEvent;
import javax.media.j3d.WakeupOr;

import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import org.w3c.dom.Node;
import org.w3c.dom.Document;  

import org.j3de.util.AbstractComponent;   
import org.j3de.util.ConfigurationException;  

/** MouseInputDevice implement an InputDevice, that uses the position of the MousePointer to
  * determine the offset of the x- and y-axis. The left and right mouse-button are used to emulate
  * movement on the z-axis. The middle mouse-button is returned as a button. */  
public class MouseInputDevice extends AbstractComponent implements InputDevice, HasBehavior {  
  private static int[] BUTTONS = new int[1];
  
  private Sensor      sensor;
  private int         processingMode;   
  private Transform3D transform;   
  private Vector3d    currentPosition; 
  
  private double      maxXAmplitude;
  private double      maxYAmplitude;
  private double      maxZAmplitude;    
  
  private double      movementSpeed;
  
  private double      screenX;
  private double      screenY;
  
  private long        leftButtonDownTime;
  private long        middleButtonDownTime;
  private long        rightButtonDownTime;
   
  private WakeupCondition mouseCriterion;

  public MouseInputDevice() {  
    this.sensor  = new Sensor(this, 30, 1);   
    this.sensor.setPredictionPolicy(Sensor.NO_PREDICTOR);
          
    processingMode = InputDevice.DEMAND_DRIVEN;  
  } 
  
  public boolean initialize() {  
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    screenX = d.getWidth() / 2.0;
    screenY = d.getHeight() / 2.0;
    transform = new Transform3D();
    currentPosition = new Vector3d();
    return true;
  }
  
  public void setNominalPositionAndOrientation() {
    // TODO implement 
  }
  
  public synchronized void pollAndProcessInput() {     
    if (leftButtonDownTime == 0)
      BUTTONS[0]= 0;
    else 
      BUTTONS[0]= 1;
    
    long currentTime = System.currentTimeMillis();
                    
    if (middleButtonDownTime != 0) {
      currentPosition.z -= (currentTime - middleButtonDownTime) / 1000.0 * movementSpeed; 
      middleButtonDownTime = currentTime; 
    }

    if (rightButtonDownTime != 0) {
      currentPosition.z += (currentTime - rightButtonDownTime) / 1000.0 * movementSpeed;  
      rightButtonDownTime = currentTime; 
    }
                                        
    if (currentPosition.z > maxZAmplitude) 
      currentPosition.z = maxZAmplitude;
    
    if (currentPosition.z < -maxZAmplitude) 
      currentPosition.z = -maxZAmplitude;
     
    transform.setTranslation(currentPosition);
    
    sensor.setNextSensorRead(currentTime, transform, BUTTONS);
  }
  
  public void processStreamInput() {  
    // According to j3d documentation this should remain an empty method ...
  }
  
  public void close() {  
    // nothing to do ...
  }
  
  public int getProcessingMode() {                           
    return processingMode;
  }
  
  public void setProcessingMode(int mode) {   
    this.processingMode = mode; 
  }
  
  public int getSensorCount() {
    return 1;
  }
  
  public Sensor getSensor(int sensorIndex) {  
    return sensor;
  }
  
  public void configure(Node node, Document nodeFactory)  throws ConfigurationException {  
    super.configure(node, nodeFactory);    
    
    maxXAmplitude = Double.parseDouble(helper.getPropertyValue("maxXAmplitude", "10", true));
    maxYAmplitude = Double.parseDouble(helper.getPropertyValue("maxYAmplitude", "10", true));
    maxZAmplitude = Double.parseDouble(helper.getPropertyValue("maxZAmplitude", "10", true));
    
    movementSpeed = Double.parseDouble(helper.getPropertyValue("speed", "10", true));    
  }    
  
  private synchronized void processMouseEvent(MouseEvent event) {   
    if (event == null)
      return;
        
    if ((event.getID() == MouseEvent.MOUSE_MOVED) || (event.getID() == MouseEvent.MOUSE_DRAGGED))  {   
      // The mouse has been moved, calculate the new Position of x- and y-axis  
      currentPosition.x = (event.getX() - screenX) / screenX * maxXAmplitude;
      currentPosition.y = -(event.getY() - screenY) / screenY * maxYAmplitude;
      
    } else if (event.getID() == MouseEvent.MOUSE_PRESSED)  { 
      
      // A mouse button has been pressed. Register when this happened.
      if (((event.getModifiers() & MouseEvent.BUTTON1_MASK) > 0) && (leftButtonDownTime == 0)) {    
        leftButtonDownTime = event.getWhen();    
      } 
      
      if (((event.getModifiers() & MouseEvent.BUTTON2_MASK) > 0) && (middleButtonDownTime == 0)) {
        middleButtonDownTime = event.getWhen();
      } 
      
      if (((event.getModifiers() & MouseEvent.BUTTON3_MASK) > 0) && (rightButtonDownTime == 0)) {
        rightButtonDownTime = event.getWhen();
      }   
    
    } else if (event.getID() == MouseEvent.MOUSE_RELEASED) {
      
      // A mouse button has been released.   
      // set DownTime to 0 and calculate new position on z-axis
      if (((event.getModifiers() & MouseEvent.BUTTON1_MASK) > 0) && (leftButtonDownTime != 0)) { 
        leftButtonDownTime = 0;
      } 
      
      if (((event.getModifiers() & MouseEvent.BUTTON2_MASK) > 0) && (middleButtonDownTime != 0)) {        
        currentPosition.z -= (event.getWhen() - middleButtonDownTime) / 1000.0 * movementSpeed; 
        middleButtonDownTime = 0;
      } 
      
      if (((event.getModifiers() & MouseEvent.BUTTON3_MASK) > 0) && (rightButtonDownTime != 0)) {
        currentPosition.z += (event.getWhen() - rightButtonDownTime) / 1000.0 * movementSpeed;  
        rightButtonDownTime = 0;
      }   
    
    }
  }     
 
  
  public Behavior getBehavior() {
    return new Behavior() {  
    
      public void initialize() {
        setSchedulingBounds(new BoundingSphere(new Point3d(0.0,0.0,0.0), 100.0));
        
        WakeupCriterion[] mouseEvents = new WakeupCriterion[4];

        mouseEvents[0] = new WakeupOnAWTEvent(MouseEvent.MOUSE_MOVED);
        mouseEvents[1] = new WakeupOnAWTEvent(MouseEvent.MOUSE_PRESSED);
        mouseEvents[2] = new WakeupOnAWTEvent(MouseEvent.MOUSE_RELEASED);    
        mouseEvents[3] = new WakeupOnAWTEvent(MouseEvent.MOUSE_DRAGGED);    
        mouseCriterion = new WakeupOr(mouseEvents);
        wakeupOn(mouseCriterion); 
      } 
   
      public synchronized void processStimulus(Enumeration criteria) {     
        WakeupCriterion wakeup;
        AWTEvent[] event;
      
        while(criteria.hasMoreElements()) {
          wakeup = (WakeupCriterion) criteria.nextElement();
        
          if( !(wakeup instanceof WakeupOnAWTEvent) )
            continue;
      
          event = ((WakeupOnAWTEvent)wakeup).getAWTEvent();
          for (int i=0; i<event.length; i++) {
            if ((event[i].getID() == MouseEvent.MOUSE_MOVED) || 
                (event[i].getID() == MouseEvent.MOUSE_PRESSED) || 
                (event[i].getID() == MouseEvent.MOUSE_RELEASED) ||
                (event[i].getID() == MouseEvent.MOUSE_DRAGGED)) {
              processMouseEvent((MouseEvent)event[i]);
            } else
              System.out.println("Unknown event : " + event);
          }
        }
        wakeupOn(mouseCriterion);
      } 
    }; 
    
  } 
} 