/*
 * xdb_odbc_file.c
 *
 * This code is based on original xdb_odbc module (jabber 1.0)
 * written by Chuck Youse and/or Jon Simantov and/or Ben Kahn.
 *
 * 11/3/00 - Modified by LP (ABM), port to jabberd 1.2
 * 2/28/01 - Implemented simple ODBC support (xdbodbc_[gs]et)
 */
#include "jabberd.h"
#include "xdb_odbc.h"

/* static variables */
static char *s_hostname = NULL;  /* ODBC host name */
static char *s_user = NULL;      /* ODBC username */
static char *s_password = NULL;  /* ODBC password */
static char *s_database = NULL;  /* ODBC database name */
static HDBC s_connection;             /* the master ODBC connection */
static int s_need_connect = 1;         /* do we need to connect? */

/* static tag names used in validation functions */
static const char bindvar_name[] = "bindvar";
static const char bindcol_name[] = "bindcol";
static const char name_attr_name[] = "name";

/* definition of the table that contains XML query definition references */
struct query_table
{
    const char *query_name;
    xmlnode query;
    int (*validate_func)(xmlnode q_root);
};

/* forward declarations for query table */
static int validate_get_storage_xml(xmlnode q_root);
static int validate_set_storage_xml(xmlnode q_root);
static int validate_purge_storage_xml(xmlnode q_root);
static int validate_get_storage_count(xmlnode q_root);
static int validate_set_storage_count(xmlnode q_root);

/* Query table, used for query validation and as query storage */
static struct query_table s_query_table[] =
{
  { "get-storage-xml", NULL, validate_get_storage_xml },
  { "set-storage-xml", NULL, validate_set_storage_xml },
  { "purge-storage-xml", NULL, validate_purge_storage_xml },
  { "get-storage-count", NULL, validate_get_storage_count },
  { "set-storage-count", NULL, validate_set_storage_count },
  { NULL, NULL, NULL } /* EOT SENTINEL */
};

/***********************************************************************
 * Query validation routines
 */

static int validate_get_storage_xml(xmlnode q_root)
{
    xmlnode tmp;
    char *attr;
    int status = 0;

    for (tmp=xmlnode_get_firstchild(q_root); tmp; tmp=xmlnode_get_nextsibling(tmp))
    { /* test for the presence of all required bindings */
        if (j_strcmp(xmlnode_get_name(tmp),bindvar_name)==0)
        { /* get the "name" attribute and check it out */
            attr = xmlnode_get_attrib(tmp,name_attr_name);

            if (j_strcmp(attr,"server_jid")==0)
                status++;
	    else if (j_strcmp(attr,"username")==0)
                status++;

        } /* end if */
        else if (j_strcmp(xmlnode_get_name(tmp),bindcol_name)==0)
        { /* get the "name" attribute and check it out */
            attr = xmlnode_get_attrib(tmp,name_attr_name);

            if (j_strcmp(attr,"nxml")==0)
                status++;
            else if (j_strcmp(attr,"xml")==0)
                status++;

        } /* end else if */

    } /* end for */

    return status == 4;
}

static int validate_set_storage_xml(xmlnode q_root)
{
    xmlnode tmp;
    char *attr;
    int status = 0;

    for (tmp=xmlnode_get_firstchild(q_root); tmp; tmp=xmlnode_get_nextsibling(tmp))
    { /* test for the presence of all required bindings */
        if (j_strcmp(xmlnode_get_name(tmp),bindvar_name)==0)
        { /* get the "name" attribute and check it out */
            attr = xmlnode_get_attrib(tmp,name_attr_name);

            if (j_strcmp(attr,"server_jid")==0)
                status++;
	    else if (j_strcmp(attr,"username")==0)
                status++;
	    else if (j_strcmp(attr,"nxml")==0)
                status++;
	    else if (j_strcmp(attr,"xml")==0)
                status++;

        } /* end if */
    } /* end for */

    return status == 4;
}

static int validate_purge_storage_xml(xmlnode q_root)
{
    xmlnode tmp;
    char *attr;
    int status = 0;

    for (tmp=xmlnode_get_firstchild(q_root); tmp; tmp=xmlnode_get_nextsibling(tmp))
    { /* test for the presence of all required bindings */
        if (j_strcmp(xmlnode_get_name(tmp),bindvar_name)==0)
        { /* get the "name" attribute and check it out */
            attr = xmlnode_get_attrib(tmp,name_attr_name);

            if (j_strcmp(attr,"server_jid")==0)
                status++;
	    else if (j_strcmp(attr,"username")==0)
                status++;

        } /* end if */
    } /* end for */

    return status == 2;
}

static int validate_get_storage_count(xmlnode q_root)
{
    xmlnode tmp;
    char *attr;
    int status = 0;

    for (tmp=xmlnode_get_firstchild(q_root); tmp; tmp=xmlnode_get_nextsibling(tmp))
    { /* test for the presence of all required bindings */
        if (j_strcmp(xmlnode_get_name(tmp),bindvar_name)==0)
        { /* get the "name" attribute and check it out */
            attr = xmlnode_get_attrib(tmp,name_attr_name);

            if (j_strcmp(attr,"server_jid")==0)
                status++;
	    else if (j_strcmp(attr,"username")==0)
                status++;

        } /* end if */
        else if (j_strcmp(xmlnode_get_name(tmp),bindcol_name)==0)
        { /* get the "name" attribute and check it out */
            attr = xmlnode_get_attrib(tmp,name_attr_name);

            if (j_strcmp(attr,"xmlcount")==0)
                status++;

        } /* end else if */

    } /* end for */

    return status == 3;
}

static int validate_set_storage_count(xmlnode q_root)
{
    xmlnode tmp;
    char *attr;
    int status = 0;

    for (tmp=xmlnode_get_firstchild(q_root); tmp; tmp=xmlnode_get_nextsibling(tmp))
    { /* test for the presence of all required bindings */
        if (j_strcmp(xmlnode_get_name(tmp),bindvar_name)==0)
        { /* get the "name" attribute and check it out */
            attr = xmlnode_get_attrib(tmp,name_attr_name);

            if (j_strcmp(attr,"server_jid")==0)
                status++;
	    else if (j_strcmp(attr,"username")==0)
                status++;
	    else if (j_strcmp(attr,"xmlcount")==0)
                status++;

        } /* end if */
    } /* end for */

    return status == 3;
}


/***********************************************************************
 * External functions
 ***********************************************************************/

int is_true(const char *str)
{
    static const char *test[] = { "true", "yes", "on", "1", NULL };
    register int i;

    if (!str)
        return 0;  /* can't be true */
    for (i=0; test[i]; i++)
        if (j_strcmp(str,test[i])==0)
            return 1;  /* match found */

    return 0;  /* no match */

} /* end is_true */

int is_false(const char *str)
{
    static const char *test[] = { "false", "no", "off", "0", NULL };
    register int i;

    if (!str)
        return 0;  /* can't be false */
    for (i=0; test[i]; i++)
        if (j_strcmp(str,test[i])==0)
            return 1;  /* match found */

    return 0;  /* no match */

} /* end is_false */

int xdbodbc_config_init(xmlnode cfgroot)
{
    static char def_hostname[] = "localhost";  /* default host name */
    static char def_user_db[] = "jabber";      /* default user and database name */
    static char def_pass[] = "";               /* default password */
    char *q_name;                              /* query name */
    xmlnode conn_base, query_base, tmp;        /* pointers into the node tree */
    register int i;                            /* loop counter */

    if (!cfgroot)
        return 0;  /* configuration not present */

    /* Have a look in the XML configuration data for connection information. */
    conn_base = xmlnode_get_tag(cfgroot,"connection");
    if(conn_base == NULL)
    {
        log_error(NULL,"xdb_odbc: Missing ODBC connection configuration");
        return 0;
    }
    else
    { /* get the database connection parameters from the <connection> block */
      const char *s_tmp;

        tmp = xmlnode_get_tag(conn_base,"host");
        if (tmp) {
	  s_tmp = (const char *)xmlnode_get_data(xmlnode_get_firstchild(tmp));
	  if (s_tmp) {
	    s_hostname = (char *)malloc(strlen(s_tmp) + 1);
	    strcpy(s_hostname, s_tmp);
	  }
	}
        tmp = xmlnode_get_tag(conn_base,"user");
        if (tmp) {
	  s_tmp = (const char *)xmlnode_get_data(xmlnode_get_firstchild(tmp));
	  if (s_tmp) {
	    s_user = (char *)malloc(strlen(s_tmp) + 1);
	    strcpy(s_user, s_tmp);
	  }	  
	}
        tmp = xmlnode_get_tag(conn_base,"pass");
        if (tmp) {
	  s_tmp = (const char *)xmlnode_get_data(xmlnode_get_firstchild(tmp));
	  if (s_tmp) {
	    s_password = (char *)malloc(strlen(s_tmp) + 1);
	    strcpy(s_password, s_tmp);
	  }	  
	}
        tmp = xmlnode_get_tag(conn_base,"db");
        if (tmp) {
	  s_tmp = (const char *)xmlnode_get_data(xmlnode_get_firstchild(tmp));
	  if (s_tmp) {
	    s_database = (char *)malloc(strlen(s_tmp) + 1);
	    strcpy(s_database, s_tmp);
	  }	  
	}

	log_debug(ZONE,"configuring ODBC connection (src,user,auth): %s,%s,%s",
		  s_hostname,s_user,s_password);

    } /* end if */

    /* fill in defaults for the connection parameters as necessary */
    if (!s_hostname)
        s_hostname = def_hostname;
    if (!s_user)
        s_user = def_user_db;
    if (!s_password)
        s_password = def_pass;
    if (!s_database)
        s_database = def_user_db;

    /* look at the queries and load them into our query table */
    query_base = xmlnode_get_tag(cfgroot,"queries");
    if(query_base == NULL)
    {
        log_error(NULL,"xdb_odbc: Missing ODBC queries configuration");
        return 0;
    }
    if (query_base)
    { /* begin probing for queries in the query base section */
        for (tmp=xmlnode_get_firstchild(query_base); tmp;
	     tmp=xmlnode_get_nextsibling(tmp))
            if (   (j_strcmp(xmlnode_get_name(tmp),"querydef")==0)
                    && xmlnode_get_tag(tmp,"text"))
            { /* get the query name and check it against the list */
                q_name = xmlnode_get_attrib(tmp,name_attr_name);
                for (i=0; s_query_table[i].query_name; i++)
                    if (j_strcmp(s_query_table[i].query_name,q_name)==0)
                    { /* now run the validation function */
		      if ((*(s_query_table[i].validate_func))(tmp)) {
			s_query_table[i].query = tmp;
			log_debug(ZONE, "Configuring query %s.", s_query_table[i].query_name);
			log_debug(ZONE, "XML: %s, %d", xmlnode2str(s_query_table[i].query), s_query_table[i].query);
		      }
		      else
			{ /* query validation failed - this is an error! */
			  log_error(ZONE,"[xdbodbc_config_init] query \'%s\' validate failed",
				    q_name);
			  return 0;
			  
			} /* end else */
		      break;
		      
                    } /* end if and for */
		
            } /* end if and for */

    } /* end if */

    /* Ensure that the query table has been completely specified by now. */
    for (i=0; s_query_table[i].query_name; i++)
        if (!(s_query_table[i].query))
        { /* make sure all known queries are specified! */
            log_error(ZONE,"[xdbodbc_config_init] query \"%s\" not specified",
                      s_query_table[i].query_name);
            return 0;

        } /* end if and for */

    return 1;  /* Qa'pla! */

} /* end xdbodbc_config_init */

HDBC *xdbodbc_database(void)
{
    if (s_need_connect)
    { /* get ready to connect to the database here */
        odbc_init();
	xdbodbc_connect();
    } /* end if */

    return &s_connection;

} /* end xdbodbc_database */

xmlnode xdbodbc_query_get(const char *name)
{
    register int i;  /* loop counter */

    /* do a simple linear search */
    for (i=0; s_query_table[i].query_name; i++) {
      if (j_strcmp(s_query_table[i].query_name,name)==0) {
	return s_query_table[i].query;
      }
    }
    return NULL;  /* not found */

} /* end xdbodbc_query_get */

void xdbodbc_config_uninit(void)
{
    register int i;  /* loop counter */

    if (s_need_connect==0)
    { /* close the database connection */
        xdbodbc_disconnect();
	odbc_deinit();
    } /* end if */

    /* null out the predefined query table */
    for (i=0; s_query_table[i].query_name; i++)
        s_query_table[i].query = NULL;

    s_hostname = s_user = s_password = s_database = NULL;

} /* end xdbodbc_config_uninit */

void xdbodbc_connect(void)
{
  if (!s_need_connect&& s_connection)
    return;

  if ((s_connection = odbc_connect(s_hostname,s_user,s_password)) == -1)
    return; /* connection failed */
  s_need_connect = 0;  /* the connection is made */

    return;
}

void xdbodbc_disconnect(void)
{
  odbc_disconnect(s_connection);
  memset(&s_connection,0,sizeof(HDBC));
  s_need_connect = 1;
}

/************************** Database access ***************************

XDB information for each server and user is stored in DB (accessed using ODBC).
XML format is to store user info (as is in file based module xdb_file).

To avoid problems with variable format data like blob, text etc., the xml data
are stored in varchar chunks (each 8000 byte size), using ordinal number to
address multiple chunks.

Following tables are used:

STORAGE:                           - stores the xml chunks
	server_jid (varchar 255)   - server name (main or transport)
	username   (varchar 255)   - user name
	nxml       (int)           - ordinal number for multiple chunks
        xml        (varchar 8000)  - XML data chunk

STORAGE_COUNT:                     - stores current number of XML data chunks in STORAGE table
	server_jid (varchar 255)   - server name (main or transport)
	username   (varchar 255)   - user name
	xmlcount   (int)           - current number of XML chunks in STORAGE table

Following stored procedures are used:

GET_STORAGE_XML
SET_STORAGE_XML
GET_STORAGE_COUNT
SET_STORAGE_COUNT

***********************************************************************/

#define XML_DATA_CHUNK_SIZE	8000	/* Size of XML data chunk */

xmlnode xdbodbc_get(char *server, char *user)
{
    xmlnode query_xml, query_count;    /* query nodes */
    xmlnode xmldata;                   /* XML data node - returned */
    HDBC *db;                          /* the database connection */
    HSTMT statement;                   /* query statement */
    query_def qd;                      /* the query definition */
    int qrc;                           /* return code from query */
    col_map map;                       /* column mapping */
    int ndx_xmlcount;                  /* index of the xmlcount, nxml, xml elementa */
    int ndx_nxml;
    int ndx_xml;
    int xmlcount;                      /* xmlcount value returned after query */
    int xmlcount_len;
    int nxml;                          /* ordinal number of the XML chunk */
    int nxml_len;
    char xml_data[XML_DATA_CHUNK_SIZE+1];/* XML data chunk */
    int xml_len;
    char *buffer = 0;                  /* XML data buffer */
    int ncount;                        /* XML chunks counter */

    if (!server || !user)
    { /* the server & user was not specified - we have to bug off */
        log_error(ZONE,"[xdbodbc_get] server and/or user not specified");
        return NULL;
    }

    /* Open the database. */
    db = xdbodbc_database();
    if (!db)
    { /* the database is unavailable - we are probably f**ked no matter what we do */
        log_error(ZONE,"[xdbodbc_get] database not initialized");
        return NULL;
    }

    query_count = xdbodbc_query_get("get-storage-count");
    if (!query_count)
    { /* no query - eep! */
        log_error(ZONE,"--!!-- ERROR - get-storage-count query not found?");
        return NULL;
    }

    /* Get number of XML chunks */
    qd = xdbodbc_querydef_init(query_count);
    if (!qd) {
      log_error(ZONE,"--!!-- ERROR - get-storage-count query def failed");
      return NULL;
    }

    xdbodbc_querydef_setvar(qd,"server_jid", server);
    xdbodbc_querydef_setvar(qd,"username", user);

    /* Execute the query! */
    qrc = odbc_query(*db, &statement, xdbodbc_querydef_finalize(qd));
    xdbodbc_querydef_free(qd);
    if (!qrc)
    { /* the query failed - bail out! */
        log_error(ZONE,"[xdbodbc_get] query_count query failed");
        return NULL;
    }

    /* Get result column numbers to bind them */
    map = xdbodbc_colmap_init(query_count);
    ndx_xmlcount = xdbodbc_colmap_index(map, "xmlcount");
    xdbodbc_colmap_free(map);

    odbc_bindcol(*db, statement, ndx_xmlcount + 1, &xmlcount, sizeof(int), &xmlcount_len);

    /* Get the count, allocate buffer for XML data */
    if (odbc_fetch(*db, statement))
      {
	log_debug(ZONE,"after getting count (xmlcount): %d",xmlcount);

	if (xmlcount <= 0) {
	  log_error(ZONE,"[xdbodbc_get] no xmlcount results, server, user not found");
	  odbc_free(*db, statement);
	  return NULL;
	}
	buffer = (char*)malloc(xmlcount * XML_DATA_CHUNK_SIZE);
	if (!buffer) {
	  log_error(ZONE,"[xdbodbc_get] unable to allocate buffer");
	  odbc_free(*db, statement);
	  return NULL;
	}
      }
    else
      {
	log_error(ZONE,"[xdbodbc_get] no xmlcount results, server, user not found: '%s','%s'", server, user);
	odbc_free(*db, statement);
	return NULL;
      }

    odbc_free(*db, statement);

    /* Get the XML data chunks */
    query_xml = xdbodbc_query_get("get-storage-xml");
    if (!query_xml)
    { /* no query - eep! */
        log_error(ZONE,"--!!-- ERROR - get-storage-xml query not found?");
	free(buffer);
        return NULL;
    }

    qd = xdbodbc_querydef_init(query_xml);
    if (!qd) {
        log_error(ZONE,"--!!-- ERROR - get-storage-xml query def failed");
	free(buffer);
        return NULL;
    }

    xdbodbc_querydef_setvar(qd,"server_jid", server);
    xdbodbc_querydef_setvar(qd,"username", user);

    /* Execute the query! */
    qrc = odbc_query(*db, &statement, xdbodbc_querydef_finalize(qd));
    xdbodbc_querydef_free(qd);
    if (!qrc)
    { /* the query failed - bail out! */
        log_error(ZONE,"[xdbodbc_get] query_xml query failed");
	free(buffer);
        return NULL;
    }

    /* Get result column numbers to bind them */
    map = xdbodbc_colmap_init(query_xml);
    ndx_nxml = xdbodbc_colmap_index(map, "nxml");
    ndx_xml = xdbodbc_colmap_index(map, "xml");
    xdbodbc_colmap_free(map);

    odbc_bindcol(*db, statement, ndx_nxml + 1, &nxml, sizeof(int), &nxml_len);
    odbc_bindcol(*db, statement, ndx_xml + 1, xml_data, XML_DATA_CHUNK_SIZE, &xml_len);

    /* Fetch XML data chunks */ 
    *buffer = '\0';
    ncount = 0;
    while (odbc_fetch(*db, statement))
      {
	ncount++;
	if (ncount != nxml)
	  {
	    log_error(ZONE,"[xdbodbc_get] XML data chunk is out of sequence, is %d, should be %d",
		      nxml, ncount);
	    odbc_free(*db, statement);
	    free(buffer);
	    return NULL;
	  }

	strcat(buffer, xml_data);
      }

    odbc_free(*db, statement);

    if (ncount != xmlcount)
      {
	log_error(ZONE,"[xdbodbc_get] number of XML data chunks doesn't match, is %d, should be %d",
		  ncount, xmlcount);
	free(buffer);
	return NULL;
      }

    /* Return the XML data as xmlnode (build from string) */
    log_debug(ZONE,"Xml buffer: %s", buffer);
    xmldata = xmlnode_str(buffer, strlen(buffer));

    free(buffer);

    return xmldata;
}

int xdbodbc_set(char *server, char *user, xmlnode node)
{
    xmlnode query_xml,
      query_purge_xml,
      query_count;                     /* query nodes */
    HDBC *db;                          /* the database connection */
    HSTMT statement;                   /* query statement */
    query_def qd;                      /* the query definition */
    int qrc;                           /* return code from query */
    char *node_str = NULL;             /* XML data node string */
    int node_len;                      /* length of XML data node string */
    int num_chunks;                    /* number of XML data chunks */
    int i;
    int len;
    char nxml_data[50];                /* buffer of XML data ordinal and count */
    char xml_data[XML_DATA_CHUNK_SIZE+1];/* buffer for XML data chunk */

    if (!server || !user || !node)
    { /* the server & user was not specified - we have to bug off */
      log_error(ZONE,"[xdbodbc_set] server and/or user and/or node not specified");
      return -1;
    }
    
    /* Get node string */
    node_str = xmlnode2str(node);
    if (!node_str)
      {
	log_error(ZONE,"[xdbodbc_set] unable to convert node to string");
	return -1;
      }
    node_len = strlen(node_str);
    if (node_len == 0)
      {
	log_error(ZONE,"[xdbodbc_set] node string is empty");
	return -1;
      }

    num_chunks = node_len / XML_DATA_CHUNK_SIZE + 1;

    /* Get the query definitions */
    query_xml = xdbodbc_query_get("set-storage-xml");
    if (!query_xml)
    { /* no query - eep! */
        log_error(ZONE,"--!!-- ERROR - set-storage-xml query not found?");
        return -1;
    }
    query_purge_xml = xdbodbc_query_get("purge-storage-xml");
    if (!query_purge_xml)
    { /* no query - eep! */
        log_error(ZONE,"--!!-- ERROR - purge-storage-xml query not found?");
        return -1;
    }
    query_count = xdbodbc_query_get("set-storage-count");
    if (!query_count)
    { /* no query - eep! */
        log_error(ZONE,"--!!-- ERROR - set-storage-count query not found?");
        return -1;
    }

    /* Open the database. */
    db = xdbodbc_database();
    if (!db)
    { /* the database is unavailable - we are probably f**ked no matter what we do */
        log_error(ZONE,"[xdbodbc_set] database not initialized");
        return -1;
    }

    /* Purge the current data */
    qd = xdbodbc_querydef_init(query_purge_xml);
    if (!qd) {
        log_error(ZONE,"--!!-- ERROR - purge-storage-xml query def failed");
        return -1;
    }

    xdbodbc_querydef_setvar(qd, "server_jid", server);
    xdbodbc_querydef_setvar(qd, "username", user);
    /* Execute the query! */
    qrc = odbc_query(*db, &statement, xdbodbc_querydef_finalize(qd));
    xdbodbc_querydef_free(qd);
    odbc_free(*db, statement);

    /* Store XML chunks */
    for (i = 0; i < num_chunks; i++)
      {
	qd = xdbodbc_querydef_init(query_xml);
	if (!qd) {
	  log_error(ZONE,"--!!-- ERROR - set-storage-xml query def failed");
	  return -1;
	}

	xdbodbc_querydef_setvar(qd, "server_jid", server);
	xdbodbc_querydef_setvar(qd, "username", user);
	sprintf(nxml_data, "%d", i + 1);
	xdbodbc_querydef_setvar(qd, "nxml", nxml_data);
	len = (i < num_chunks - 1) ? XML_DATA_CHUNK_SIZE : strlen(node_str + XML_DATA_CHUNK_SIZE * i);
	strncpy(xml_data, node_str + XML_DATA_CHUNK_SIZE * i, len);
	xml_data[len] = '\0';
	xdbodbc_querydef_setvar(qd, "xml", xml_data);
 
	/* Execute the query! */
	qrc = odbc_query(*db, &statement, xdbodbc_querydef_finalize(qd));
	xdbodbc_querydef_free(qd);
	if (!qrc)
	  { /* the query failed - bail out! */
	    log_error(ZONE,"[xdbodbc_set] query_xml query failed");
	    odbc_free(*db, statement);
	    return -1;
	  }
      }

    odbc_free(*db, statement);

    /* Store XML chunks count */
    qd = xdbodbc_querydef_init(query_count);
    if (!qd) {
        log_error(ZONE,"--!!-- ERROR - set-storage-count query def failed");
        return -1;
    }

    xdbodbc_querydef_setvar(qd, "server_jid", server);
    xdbodbc_querydef_setvar(qd, "username", user);
    sprintf(nxml_data, "%d", num_chunks);
    xdbodbc_querydef_setvar(qd, "xmlcount", nxml_data);
    
    /* Execute the query! */
    qrc = odbc_query(*db, &statement, xdbodbc_querydef_finalize(qd));
    xdbodbc_querydef_free(qd);
    if (!qrc)
      { /* the query failed - bail out! */
	log_error(ZONE,"[xdbodbc_set] query_count query failed");
      }
    
    odbc_free(*db, statement);

    return 0;
}
