/* $Id: xdb_odbc_helper.c,v 1.3 2001/03/08 18:18:00 lubos Exp $
 * ODBC function wrappers to make life easier.
 *
 * Most of this code was taken lock, stock, and barrel
 * from the CyberSites Database Caching Layer.
 *
 * This code was written by Chuck Youse and Jon Simantov.
 *
 * 11/3/00 - Modified by LP (ABM), port to jabberd 1.2
 * 2/28/01 - Modified by LP (ABM), port to jabberd 1.4
 */

#include <stdio.h>
#include "jabberd.h"
#include "xdb_odbc.h"
#include "xdb_odbc_helper.h"

#define INIT_OK 0
#define INIT_ERROR -1

/* #define MAX_ODBC_DATA 256 */
/* static char	odbc_source[MAX_ODBC_DATA+1]="",  */
/* 		odbc_user[MAX_ODBC_DATA+1]="",  */
/* 		odbc_auth[MAX_ODBC_DATA+1]="";	 */
static HENV	henv;

static int dbref = 0;		/* number of other modules referencing us */

/*
 * Perform a "safe" ODBC operation.  In other words, check for error
 * conditions, and report them.  All unsuccessful database operations are
 * considered fatal if 'fatal' is non-zero.  
 */

int odbc_safe(HDBC hdbc, HSTMT hstmt, RETCODE rc, int fatal)
{
	UCHAR	sqlstate[6];
	SDWORD	native_error;
	UCHAR	error_msg[SQL_MAX_MESSAGE_LENGTH + 1];
	SWORD	error_len;
	RETCODE brc;

	if ((rc != SQL_SUCCESS) && (rc != SQL_SUCCESS_WITH_INFO)
 	    && (rc != SQL_NEED_DATA)) 
	  {
		for (;;) {
			brc = rc;
			rc = SQLError(henv, hdbc, hstmt, sqlstate, 
				&native_error, error_msg, sizeof(error_msg), 
				&error_len);
 			if (rc != SQL_SUCCESS) break; 
			log_error(ZONE, 
				   "SQL ERROR (rc=%d): %s", brc, error_msg);
		} 

		if (fatal) {
			log_error(ZONE, "exiting due to fatal SQL errors");
			exit(1);
		} else
		  return 0;
	}

	if (rc == SQL_NEED_DATA) return 2;
	return 1;
}

int odbc_init()
{
  dbref++;
  
  if (dbref > 1)		/* already connected */
    return INIT_OK;

  if (SQLAllocEnv(&henv) != SQL_SUCCESS) {
    log_error(ZONE, "fatal error: SQLAllocEnv() failure");
    return INIT_ERROR;
  }

  return INIT_OK;
}


/*
 * Connect the ODBC subsystem.  Call this before calling any other
 * routines.  This will abort with a fatal error if there is a trouble
 * connecting to the data source.
 */

HDBC odbc_connect(const char *odbc_source, const char *odbc_user, const char *odbc_auth)
{
        HDBC	hdbc;
	
	log_debug(ZONE,"connecting to ODBC (src,user,auth): %s,%s,%s",
		  odbc_source,odbc_user,odbc_auth);

	if (SQLAllocConnect(henv, &hdbc) != SQL_SUCCESS) {
		log_error(ZONE, "fatal error: SQLAllocConnect() failure");
		return INIT_ERROR;
	}

	if (odbc_safe(hdbc, SQL_NULL_HSTMT, 
		      SQLConnect(hdbc, 
				 (char *)odbc_source, strlen(odbc_source),
				 (char *)odbc_user, strlen(odbc_user),
				 (char *)odbc_auth, strlen(odbc_auth)
				 ), 0) == 0)
	  {
	    log_error(ZONE, "fatal error: Couldn't connect!");
	    return INIT_ERROR;
	  }
	
	return hdbc;
}

/*
 * Shut down the ODBC system, disconnecting from the database
 * and freeing any extra used memory.
 */

int odbc_disconnect(HDBC hdbc)
{
	if (SQLDisconnect(hdbc) != SQL_SUCCESS) {
		log_error(ZONE, "Error: SQLDisconnect() failure");
	}

	if (SQLFreeConnect(hdbc) != SQL_SUCCESS) {
		log_error(ZONE, "Error: SQLFreeConnect() failure");
	}

	return INIT_OK;
}

/*
 * Shut down the ODBC system, disconnecting from the database
 * and freeing any extra used memory.
 */

int odbc_deinit()
{
	if (dbref > 0) dbref--;

	if (dbref > 0)
	  return INIT_OK;

	if (SQLFreeEnv(henv) != SQL_SUCCESS) {
		log_error(ZONE, "Error: SQLFreeEnv() failure");
	}

	return INIT_OK;
}

/*
 * Allocate a statement handle and initialize it as a prepared statement.
 */

void odbc_mkstmt(HDBC hdbc, HSTMT *hstmt, const char *stmt)
{
  log_debug(ZONE, "Preparing SQL query: %s", stmt);
  if (odbc_safe(hdbc, SQL_NULL_HSTMT, SQLAllocStmt(hdbc, hstmt), 0)==0) {
    log_error(ZONE,"Preparing SQL query failed!");
    return;
  }
  if (odbc_safe(hdbc, *hstmt, SQLPrepare(*hstmt, (char *) stmt, SQL_NTS), 0) == 0) {
    log_error(ZONE,"Preparing SQL query failed!");
    return;
  }
}

int odbc_query(HDBC hdbc, HSTMT *hstmt, const char *stmt)
{
  int rc;
  log_debug(ZONE, "Executing SQL query: %s, %d, %d", stmt, hdbc, hstmt);
  if (!hdbc || !hstmt || !stmt) return 0;
  rc = odbc_safe(hdbc, SQL_NULL_HSTMT, SQLAllocStmt(hdbc, hstmt), 0);
  if (rc)
    return odbc_safe(hdbc, *hstmt, SQLExecDirect(*hstmt, (char *) stmt, strlen(stmt)), 0);
  else return rc;
}

/*
 * Bind an output argument to a statement parameter (for procedure calls).
 */

void odbc_bindoparam(HDBC hdbc, HSTMT hstmt, int param, int type, void *buf, int *length)
{
  if (odbc_safe(hdbc, hstmt,
		SQLBindParameter(hstmt, param, SQL_PARAM_OUTPUT,
				 SQL_C_DEFAULT, type, 256, 0, buf, 0,
				 (SDWORD *) length), 0)==0) {
    log_error(ZONE,"bindoparam failed.");
  }
}


/*
 * Bind an input argument to a statement parameter.
 */

void odbc_bindparam(HDBC hdbc, HSTMT hstmt, int param, int type, void *buf, int *length)
{
  if (odbc_safe(hdbc, hstmt,
		SQLBindParameter(hstmt, param, SQL_PARAM_INPUT, SQL_C_DEFAULT,
				 type, 256, 0, buf, 0, (SDWORD *) length), 0)==0) {
    log_error(ZONE,"bindparam failed.");
  }
}

/*
 * Bind an input argument to a statement parameter, to be
 * sent with SQLPutData.
 */

void odbc_bindbigparam(HDBC hdbc, HSTMT hstmt, int param, int type, void *buf, int *length, 
		       int maxdatasize)
{
  if (odbc_safe(hdbc, hstmt,
		SQLBindParameter(hstmt, param, SQL_PARAM_INPUT, SQL_C_DEFAULT,
				 type, maxdatasize, 0, buf, 0, (SDWORD *) length),
		0) == 0) {
    log_error(ZONE,"bindbigparam failed.");
  }
}


/*
 * Get an input parameter to use odbc_putdata
 */

RETCODE odbc_paramdata(HDBC hdbc, HSTMT hstmt, void *param)
{
	    return SQLParamData(hstmt, param);
}

/*
 * Bind a buffer to a result column.
 */

void odbc_bindcol(HDBC hdbc, HSTMT hstmt, int col, void *buf, int length, int *plen)
{
  if (odbc_safe(hdbc, hstmt, 	
		SQLBindCol(hstmt, col, SQL_C_DEFAULT, 
			   buf, length, (SDWORD *) plen), 0)==0) {
    log_error(ZONE,"bindcol failed.");
  }
}


/*
 * Execute a prepared statement.
 */


int odbc_exec(HDBC hdbc, HSTMT hstmt, int fatal)
{
	return odbc_safe(hdbc, hstmt, SQLExecute(hstmt), 0);
}

/*
 * Fetch a row for the given statement. 
 * Return non-zero if successful, 0 if no more rows.
 */

int odbc_fetch(HDBC hdbc, HSTMT hstmt)
{
	RETCODE rc;

	rc = SQLFetch(hstmt);
	if (rc == SQL_NO_DATA_FOUND) return 0;

	if(odbc_safe(hdbc, hstmt, rc, 0)==0) return 0;
	return 1;
}

/*
 * Fetch data for a row/column entry for the given statement. 
 * Return non-zero if successful, 0 if no more rows.
 */

int odbc_getdata(HDBC hdbc, HSTMT hstmt, int col, void *buf, int length, int *plen)
{
	RETCODE rc;

	rc = SQLGetData(hstmt, col, SQL_C_BINARY, 
			buf, length, (SDWORD *)plen);
	if (rc == SQL_NO_DATA_FOUND) return 0;

	if (odbc_safe(hdbc, hstmt, rc, 0)==0) return 0;
	return 1;
}

/*
 * Fetch data for a row/column entry for the given statement. 
 * Return non-zero if successful, 0 if no more rows.
 */

int odbc_putdata(HDBC hdbc, HSTMT hstmt, void *buf, int length)
{
	RETCODE rc;

	rc = SQLPutData(hstmt, buf, length);
	
	if (odbc_safe(hdbc, hstmt, rc, 0)==0) return 0;
	return 1;
}


/*
 * Fetch a row for the given statement. Shouldn't be called more
 * than once at a time, as won't get successive rows--use odbc_fetch
 * for that. 
 *
 * Return non-zero if successful, 0 if no more rows.
 */

int odbc_fetchonce(HDBC hdbc, HSTMT hstmt)
{
	RETCODE rc;

	rc = SQLFetch(hstmt);
	if (rc == SQL_NO_DATA_FOUND) return 0;

	return 1;
}

/*
 * Reset a statement handle. 
 */

void odbc_reset(HDBC hdbc, HSTMT hstmt)
{
  if(odbc_safe(hdbc, hstmt, SQLFreeStmt(hstmt, SQL_RESET_PARAMS), 0)==0) {
    log_error(ZONE,"reset failed.");
    return;
  }
  if(odbc_safe(hdbc, hstmt, SQLFreeStmt(hstmt, SQL_UNBIND), 0)==0) {
    log_error(ZONE,"reset failed.");
    return;
  }
  if(odbc_safe(hdbc, hstmt, SQLFreeStmt(hstmt, SQL_CLOSE), 0)==0) {
    log_error(ZONE,"reset failed.");
    return;
  }
}

/*
 * Destroy a statement handle. 
 */

void odbc_free(HDBC hdbc, HSTMT hstmt)
{
	SQLFreeStmt(hstmt, SQL_UNBIND);
	SQLFreeStmt(hstmt, SQL_DROP);
}

