/* $Id: xdb_odbc_querydef.c,v 1.2 2001/03/01 01:12:45 lubos Exp $
 *
 * xdb_odbc_querydef.c
 *
 * This code was written by Chuck Youse and Jon Simantov and Ben Kahn.
 *
 * 11/3/00 - Modified by LP (ABM), port to jabberd 1.2
 * 2/28/01 - Modified by LP (ABM), port to jabberd 1.4
 */
#include "jabberd.h"
#include "xdb_odbc.h"

query_def xdbodbc_querydef_init(xmlnode qd_root)
{
    pool p;                     /* new memory pool for querydef */
    query_def qd;               /* querydef pointer */
    query_def_bind_var var;     /* new bind variable structure */
    xmlnode tmp;                /* for traversing all <querydef> children */

    /* allocate pool and querydef structure */
    p = pool_new();
    qd = (query_def)pmalloc(p,sizeof(struct query_def_struct));
    qd->p = p;
    qd->query_text = NULL;
    qd->first_var = qd->last_var = NULL;

    for (tmp=xmlnode_get_firstchild(qd_root); tmp; tmp=xmlnode_get_nextsibling(tmp))
    { /* look for <text> and <bindvar> tags among the children */

        if (j_strcmp(xmlnode_get_name(tmp),"text")==0)
            qd->query_text = (char *)xmlnode_get_data(xmlnode_get_firstchild(tmp));
        else if (j_strcmp(xmlnode_get_name(tmp),"bindvar")==0)
        { /* copy the <bindvar> tag to the list of the bind values */
            var = (query_def_bind_var)pmalloc(p,sizeof(struct query_def_bind_var_struct));
            var->name = (const char *)xmlnode_get_attrib(tmp,"name");
            var->replace_text = (const char *)xmlnode_get_data(xmlnode_get_firstchild(tmp));
            var->default_value = (const char *)xmlnode_get_attrib(tmp,"default");
            var->need_escape = (is_false(xmlnode_get_attrib(tmp,"escape")) ? 0 : 1);
            var->next = NULL;
            var->prev = qd->last_var;
            if (qd->last_var)
                qd->last_var->next = var;
            else
                qd->first_var = var;
            qd->last_var = var;

        } /* end else if */

    } /* end for */

    if (qd->query_text == NULL)
      return NULL;

    return qd;

} /* end xdbodbc_querydef_init */

static void do_replace(
    query_def qd,                /* query def buffer containing string to be morphed */
    query_def_bind_var var,      /* variable definition to replace */
    const char *value)           /* value to replace variable with - NULL to delete */
{
    const char *repl_value = NULL;  /* replacement text (properly escaped) */
    spool s;                 /* builds the new query string */
    int var_len, head_len;   /* length of the replacement variable and string "head" */
    char *esc_buffer=NULL;   /* used to SQL-escape the replacement text */
    int size, maxsize;		/* contains max possible length of SQL-escaped string */
    char *p;
    const char *q;		/* p and q are temp variables for string copying */
    char *buffer;            /* used to make the "head" into a proper string */
    char *point, *next;      /* used in searching the original query string */

    /* first, make sure that we actually DO have a variable to replace! */
    next = strstr(qd->query_text,var->replace_text);
    if (!next)
        return;

    if (value)
    { /* may need to escape the value */
      maxsize = 2*strlen(value);
      size = strlen(value);
      esc_buffer = malloc(maxsize);
      if (!esc_buffer)
	{
	  log_error(ZONE, "Emergency! Cannot allocate a measly %d bytes! Bailing out...", maxsize);
	  exit(1);
	}
      for (p = esc_buffer, q = value; q <= value + size;)
	{
	  switch(*q)
	    {
	    case '\n':
	      *p++ = '\\';
	      *p++ = 'n';
	      q++;
	      break;
	    case '\'':
	    case '\\':
	      *p++ = *q;
	    default:
	      *p++ = *q++;
	    }
	}
      value = esc_buffer;
      repl_value = value;
    } /* end if */

    /* initialize other variables needed during the replacement loop */
    point = qd->query_text;
    var_len = strlen(var->replace_text);
    /* buffer is allocated as the maximum possible size of any "head" string */
    buffer = (char *)alloca((strlen(qd->query_text) - var_len + 1) * sizeof(char));
    s = spool_new(qd->p);

    while (next)
    { /* figure out how much we have to copy before the replacement text */
        head_len = (int)(next - point);
        if (head_len>0)
        { /* copy the "head" of the buffer and add it */
            memcpy(buffer,point,head_len * sizeof(char));
            buffer[head_len] = '\0';
            spool_add(s,buffer);

        } /* end if */

        /* add the replacement value */
        spool_add(s,(char *)repl_value);

        /* look for the next instance */
        point = next + var_len;
        next = strstr(point,var->replace_text);

    } /* end while */

    /* add the rest of the "tail" and save the resulting string */
    spool_add(s,point);
    qd->query_text = spool_print(s);

    if (esc_buffer) free(esc_buffer);
} /* end do_replace */

void xdbodbc_querydef_setvar(query_def qd, const char *name, const char *value)
{
    query_def_bind_var var;  /* the variable we're replacing */

    /* look for a matching variable definition */
    for (var=qd->first_var; var; var=var->next)
        if (j_strcmp(name,var->name)==0)
            break;

    if (!var)
        return;  /* no suitable variable definition found */

    /* take this variable definition out of the list */
    if (var->next)
        var->next->prev = var->prev;
    else
        qd->last_var = var->prev;
    if (var->prev)
        var->prev->next = var->next;
    else
        qd->first_var = var->next;

    /* Perform the variable replacement... */
    do_replace(qd,var,value);

} /* end xdbodbc_querydef_setvar */

char *xdbodbc_querydef_finalize(query_def qd)
{
    query_def_bind_var var;  /* used to default out all unspecified values */

    while (qd->first_var)
    { /* take the first variable off the list */
        var = qd->first_var;
        qd->first_var = var->next;
        if (qd->first_var)
            qd->first_var->prev = NULL;

        /* replace its default value into the text */
        do_replace(qd,var,var->default_value);

    } /* end while */

    return qd->query_text;

} /* end xdbodbc_querydef_finalize */

void xdbodbc_querydef_free(query_def qd)
{
    pool_free(qd->p);

} /* end xdbodbc_querydef_free */

col_map xdbodbc_colmap_init(xmlnode qd_root)
{
    pool p;                     /* new memory pool for colmap */
    col_map cm;                 /* new colmap to return */
    xmlnode tmp;                /* for traversing all <querydef> children */
    col_map_entry ent;          /* new colmap entry */

    p = pool_new();
    cm = (col_map)pmalloc(p,sizeof(struct col_map_struct));
    cm->p = p;
    cm->first = cm->last = NULL;

    for (tmp=xmlnode_get_firstchild(qd_root); tmp; tmp=xmlnode_get_nextsibling(tmp))
    { /* look for all <bindcol> tags under the <querydef> */
        if (j_strcmp(xmlnode_get_name(tmp),"bindcol")==0)
        { /* allocate a column map entry for this column */
            ent = (col_map_entry)pmalloc(p,sizeof(struct col_map_entry_struct));
            ent->next = NULL;
            ent->name = (const char *)xmlnode_get_attrib(tmp,"name");
            ent->index = atoi(xmlnode_get_attrib(tmp,"offset"));
            if (cm->last)
                cm->last->next = ent;
            else
                cm->first = ent;
            cm->last = ent;

        } /* end if */

    } /* end for */

    return cm;

} /* end xdbodbc_colmap_init */

void xdbodbc_colmap_free(col_map cm)
{
    pool_free(cm->p);

} /* end xdbodbc_colmap_free */

int xdbodbc_colmap_index(col_map cm, const char *name)
{
    col_map_entry ent;
    for (ent=cm->first; ent; ent=ent->next)
        if (j_strcmp(ent->name,name)==0)
            return ent->index;

    return -1;

} /* end xdbodbc_colmap_index */
