package com.yci.seb.portal.adapterportletcontainer;

//Adapter Portlet Api support
import com.yci.seb.portal.adapterportlet.*;
import com.yci.seb.portal.adapterportletcontainer.information.*;
import com.yci.seb.portal.services.resources.SEBPortalResources;

//YCI ECCommon
import com.yci.eccommon.util.StringUtil;

//Jetspeed
import org.apache.jetspeed.services.Profiler;
import org.apache.jetspeed.services.profiler.*;
import org.apache.jetspeed.om.profile.Profile;
import org.apache.turbine.om.security.Role;
import org.apache.turbine.om.security.Group;

//Turbine
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.DynamicURI;
import org.apache.turbine.services.*;
import org.apache.turbine.services.resources.ResourceService;

//java
import javax.servlet.http.*;
import java.util.*;
import java.net.URLEncoder;

/**
 * A <CODE>AdapterPortletURI</CODE> represents a URI to a specific portlet
 * function.  A URI is created through the <CODE>AdapterPortletResponse</CODE>
 * for a specific portlet mode.  Then additional parameters can be
 * added to the URI.  The complete URI can be converted to a string
 * which is ready for embedding into markup.
 * <P>
 * On top of the parameters, it is possible to add actions to a
 * portlet URI. Actions are portlet-specific activities that need to
 * be performed as result of the incoming request, but before the
 * <CODE>service()</CODE> method of the portlet is called. For example,
 * the PERSONALIZE mode of the portlet is likely to have a "Save"
 * button at the end of its dialog. The "Save" button has to bring
 * the user back to the DEFAULT mode of the portlet, but to save the
 * personalized portlet data, the portlet needs to be able to process
 * the posted data bfore the next markup is generated. This can be
 * achieved by adding a "Save" action to the URI that represents the
 * "Save" button. The respective listener is attached the respective
 * action listener to the portlet response. This listener will be
 * called when the next request comes and one of the portlet URIs
 * where the reason for the request. If more than one URI were part
 * of the response, the listener need to the check the action content.
 * This depends on the definition of the actual action which is the
 * responsibility of the portlet developer.
 * </P>
 * 
 * @version Modificaitons:
 * 	<p>02/15/2002	-	Marcus Mosttler	-	
 *		Created from jetspeed portlet api branch.
 *	<p>03/292002	-	Marcus Mosttler	-
 *		Updated to use DynamicURI to create path info, query data.
 *	<p>06/13/2002	-	Marcus Mosttler	-
 *		Added setState to allow for the window state to be set on the portlet uri.
 *	<p>06/19/2002	-	Marcus Mosttler	-
 *		Added method to append a special controller path.  This was necessary to
 *		include the Tab Controller tab portions of the URI.
 *	<p>06/24/2002	-	Marcus Mosttler	-
 *		Added setPortlet to set the PIID for this URI.  The PIID will be added to
 *		the path info during rendering of the uri string.
 *	<p>08/08/2002	-	Marcus Mostler	-
 *		Override add(...) to not lowercase the names.
 *	<p>08/14/2002	-	Marcus Mostler	-
 *		Updated setState to support Normal(Restore) and added code for minimize.
 *	<p>08/16/2002	-	Marcus Mosttler	-
 *		Added method to set the uri to be ssl.
 */
public class AdapterPortletURIImpl extends DynamicURI implements AdapterPortletURI
{
    //Profile configuration keys
    public final static String CONFIG_RESOURCE_DEFAULT = "resource.default";
    public final static String CONFIG_RESOURCE_EXT     = "resource.ext";

    //Profile default configuration values
    public final static String DEFAULT_CONFIG_RESOURCE_DEFAULT = "default";
    public final static String DEFAULT_CONFIG_RESOURCE_EXT = ".psml";

    private String piid = null;
    private String pid = null;
    private AdapterPortlet.Mode mode = null;
    private AdapterPortletWindow.State state = null;
    private HttpSession session = null;
    private PortalInformationProvider provider = null;
    private Hashtable parameters = new Hashtable();
    private String actionreference = null;

    private String uriHost = null;
    private String uriPath = null;
    private String uriParameters = null;
    private String uriSession = null;

    /**
	 * Buffer used to store the full string representation of this class.
	 */
	private StringBuffer fullBuffer;
	
    /**
	 * Buffer used to store the current URI string.
	 * This is returned if isDirty=false.
	 */
    private StringBuffer currentURI;
	
	/**
	 * Buffer used to store the current Query string.
	 * This is returned if isDirty=false.
	 */
    private StringBuffer currentQueryString;

    /**
	 * Flag used to keep from reprocessing to render same URI over and over
	 * when no changes have been made.
	 */
    private boolean isDirty = true;
    private boolean isPathInfoDirty = true;
    private boolean isQueryStringDirty = true;
    
	/**
	 * Flag used to keep from adding multiple profiles to a single URI.
	 */
    private boolean profileAdded = false;
	/**
	 * Flag used to keep from adding multiple portlets to a single URI.
	 */
    private boolean portletAdded = false;
	/**
	 * Flag used to keep from adding multiple controller paths to a single URI.
	 */
    private boolean controllerPathAdded = false;
    private StringBuffer controllerPathBuffer = null;
	/**
	 * Constructor. Creates a portlet uri.
	 */
	public AdapterPortletURIImpl(String piid, String pid, 
		AdapterPortlet.Mode mode, AdapterPortletWindow.State state,
	    HttpSession session, PortalInformationProvider provider)
	{
		super(((PortalInformationProviderImpl)provider).getRunData());

		if ((state == AdapterPortletWindow.State.MOVING)
	        || (state == AdapterPortletWindow.State.RESIZING))
	    {
	        throw new IllegalArgumentException(
		        "Invalid PortletWindow.State. No -ING states are allowed.");
	    }
		this.piid = piid;
	    this.pid = pid;
	    this.mode = mode;
	    this.state = state;
	    this.session = session;
	    this.provider = provider;
	    this.isDirty = true;
	    this.isPathInfoDirty = true;
	    this.isQueryStringDirty = true;
	}
	/**
	 * Constructor. Creates a portlet uri.
	 */
	public AdapterPortletURIImpl(String piid, String pid,
		AdapterPortlet.Mode mode, HttpSession session,
		PortalInformationProvider provider)
	{
	    this(piid, pid, mode, null, session, provider);
	}
	/**
	 * Constructor. Creates a portlet uri.
	 */
	public AdapterPortletURIImpl(RunData rundata)
	{
		super(rundata);
		this.provider = new PortalInformationProviderImpl(rundata);
	}
    /**
     * <p>If the type is P (0), then add name/value to the pathInfo
     * hashtable.
     *
     * <p>If the type is Q (1), then add name/value to the queryData
     * hashtable.
     *
     * @param type Type (P or Q) of insertion.
     * @param name A String with the name to add.
     * @param value A String with the value to add.
     */
    protected void add (int type, String name, String value )
    {
        Object[] tmp = new Object[2];
        tmp[0] = (Object) name.trim();
        tmp[1] = (Object) value;
        switch (type)
        {
        case PATH_INFO:
            this.pathInfo.addElement( tmp );
            this.hasPathInfo = true;
            break;
        case QUERY_DATA:
            this.queryData.addElement( tmp );
            this.hasQueryData = true;
            break;
        }
    }
	public void addAction(AdapterPortletAction action)
	{
	//Not currently implemented
	//  actionreference =
	//		PortletActionManager.storePortletAction(action, piid, provider, session);
    //    isDirty = true;
    }
/**
 * @see com.yci.seb.portal.adapterportlet.AdapterPortletURI
 */
public void addAction(String action) 
{
	setAction(action);
}
    public void addParameter(String name, String value)
    {
		addQueryData(name, value);
        isDirty = true;
        isQueryStringDirty = true;
    }
    /**
     * If a portlet value is supplied then it is added to the
     * path info.
     *
     * @see AdapterPortletURI#addPortlet(String portlet)
     */
	public void addPortlet(String portlet)
	{
		if((portlet != null && portlet.length() > 0) && portletAdded == false)
		{
			addPathInfo("", portlet);
			piid = portlet;
			portletAdded = true;
	        isDirty = true;
	        isPathInfoDirty = true;
		}
	}
	/**
	 * Appends the provided profile to the path info using page param.
	 *
     * @see AdapterPortletURI#addProfile(String profile)
     */
	public void addProfile(String profile)
    {
	    if(profile != null && profile.length() > 0)
		    addProfile(Profiler.PARAM_PAGE, profile);
    }
	/**
	 * Appends the provided profile type and name to the path info.
	 *
     * @see AdapterPortletURI#addProfile(String profileParam, String profileName)
     */
	public void addProfile(String profileParam, String profileName)
    {
	    if((profileParam != null && profileName != null) && profileAdded == false)
	    {
		    addPathInfo(profileParam, profileName);
		    profileAdded = true;
	        isDirty = true;
	        isPathInfoDirty = true;
		}
    }
/**
 * Retrieves the special controller path for the current request.
 * This is pulled from the JspPortletController via static method.
 * The path is then added to the path info.
 * 
 * Creation date: (6/19/2002 11:44:53 AM)
 */
public void appendControllerPath() 
{
	if(!controllerPathAdded)
	{
		if(controllerPathBuffer == null)
		{
			controllerPathBuffer = new StringBuffer(
				com.yci.seb.portal.controllers.JspPortletController.getControllerPath(
					((PortalInformationProviderImpl)provider).getRunData().getRequest() ));
		}
		if(controllerPathBuffer.length() > 0)
			addPathInfo("", controllerPathBuffer.toString());
			
		controllerPathAdded = true;
	}
}
/**
 * Adds the special controller path for the current request.
 * 
 * Creation date: (6/19/2002 11:44:53 AM)
 *
 * @param name java.lang.String
 * @param value int
 */
public void appendControllerPath(String name, int value)
{
	appendControllerPath(name, Integer.toString(value));
}
/**
 * Adds the special controller path for the current request.
 * 
 * Creation date: (6/19/2002 11:44:53 AM)
 *
 * @param name java.lang.String
 * @param value java.lang.String
 */
public void appendControllerPath(String name, String value)
{
	if(!controllerPathAdded)
	{
		if(value != null && value.length() > 0)
		{
			controllerPathBuffer = new StringBuffer(((name != null) ? name : ""));
			if(controllerPathBuffer.length() > 0)
				controllerPathBuffer.append("/");
			controllerPathBuffer.append(value);
//			addPathInfo( ((name != null) ? name : ""), value);
		}
			
//		controllerPathAdded = true;
	}
}
	/**
   	 * Gets the current request profile and adds it to the path info.
   	 */
	public void appendProfile() 
	{
		if(profileAdded == false)
		{
			try
			{
				Profile profile = provider.getProfile();
					
	//We don't currently want to add these due to the use of Role Fallback.
	//        	Group group = profile.getGroup();
	//        	if(group != null)
	//	        	appendProfile(buffer, Profiler.PARAM_GROUP, group.getName());
	//        	else 
	//        	{
	//            	Role role = profile.getRole();
	//            	if(role != null)
	//		        	appendProfile(buffer, Profiler.PARAM_ROLE, role.getName());
	//        	}

				//FIXME: need to determine if addition of the profile is necessary.		
				//It is only necessary when the profile is not default following 
				//may be the best solution.

		        // get configuration parameters from Jetspeed Resources
		        ResourceService serviceConf = 
		        	SEBPortalResources.getServiceResources(ProfilerService.SERVICE_NAME);

		        StringBuffer resourceDefault = new StringBuffer(
			        serviceConf.getString(CONFIG_RESOURCE_DEFAULT, DEFAULT_CONFIG_RESOURCE_DEFAULT));
		        
		        String resourceExt = serviceConf.getString(CONFIG_RESOURCE_EXT, 
			        DEFAULT_CONFIG_RESOURCE_EXT);
		        if(resourceExt.indexOf('.') == -1)
		            resourceDefault.append('.');
		        resourceDefault.append(resourceExt);
		        
		        String page = profile.getName(); 
		        if(page != null && !page.equalsIgnoreCase(resourceDefault.toString()))
			        addProfile(Profiler.PARAM_PAGE, page.substring(0, page.indexOf(resourceExt)));
			}
			catch(NullPointerException npe)
			{				
				//if we get a null pointer exception then we couldn't get the profile, so continue.
			}
		}
	}
/**
 * @see com.yci.seb.portal.adapterportlet.AdapterPortletURI
 */
public String getPathInfo() 
{
	if(isPathInfoDirty)
	{
	    //Create URI Parts to get extra portlet information.
	    //Can't do split if any of the following are null.
		if(pid != null && piid != null && mode != null && state != null)
		    splitURI();

		//Add Portlet URI Host
	    currentURI = new StringBuffer(getUriHost());
	        
	    //Add servlet and servlet context.
	    String scriptName = getScriptName();
	    currentURI.append(scriptName);

	    //FIXME: Add Root URI Path may be needed to get the actions
	    //and customization URI path info. Now just get what is after
	    //script name (if present in uriPath)
	    if(uriPath != null) 
	       	currentURI.append(uriPath.substring(uriPath.indexOf(scriptName)+scriptName.length()));

		//If profile or portlet added from outside this method then
		//the following corresponding method will add nothing.
		
	  	//Add Profile Document name to reference non default user PSML.
	    appendProfile();
	    //Add any special controller path (ex. for cardcontroller - pane0/Tab 1) 
	    appendControllerPath(); 
		//Add Portlet ID.
	   	addPortlet(piid);
		
		//Add path info collection (including the profile and portlet added above.
	    if(this.hasPathInfo)
		    currentURI.append(renderPathInfo(this.pathInfo));

		isPathInfoDirty = false;
	}	
	return currentURI.toString();
}
    /**
     * Currently returns string representation.  The toString()
     * will be expanded to meet a greater purpose so if we just want the
     * root uri for the portlet use this method, which will be modified to
     * only return the portlet root when toString is expanded.
     * 
     * @see AdapterPortletURI#getPortletRootUri()
     */
    public String getPortletRootUri()
    {
        return this.toString();
    }
/**
 * @see com.yci.seb.portal.adapterportlet.AdapterPortletURI
 */
public String getQueryString()
{
	if(isQueryStringDirty)
	{
		currentQueryString = new StringBuffer();
	   	//Add any query parameters from current URI.
	   	if(uriParameters != null) 
	      	currentQueryString.append(uriParameters);
	        	
	    if(this.hasQueryData)
	    {
			if(uriParameters == null) 
	  			currentQueryString.append('?');
	        currentQueryString.append(renderQueryString(this.queryData));
	    }
	    isQueryStringDirty = false;
	}
	return currentQueryString.toString();
}
    /**
     * @see AdapterPortletURI#getUriHost()
     */
	public String getUriHost()
	{
		if(uriHost == null)
		{
			StringBuffer output = new StringBuffer(getServerScheme());
	        output.append("://");
	        output.append(getServerName());
	        if( (getServerScheme().equals(HTTP) && getServerPort() != 80)
	        	|| (getServerScheme().equals(HTTPS) && getServerPort() != 443) )
	        {
	            output.append(":");
	            output.append(getServerPort());
	        }
	        uriHost=output.toString();
		}
		return uriHost;
	}
    /**
     * Override parent to make more efficient the adding
     * of path info from vector to path.
     *
     * @see DynamicURI#renderPathInfo(Vector data)
     */
    protected String renderPathInfo(Vector data)
    {
        String key = null;
        String value = null;
        StringBuffer out = new StringBuffer("");
        Iterator dataIter = data.iterator();
        while(dataIter.hasNext())
        {
            Object[] entry = (Object[])dataIter.next();
            key = URLEncoder.encode(entry[0].toString());
            value = entry[1].toString();
            
            if(key != null)
            {
	            if(!out.toString().endsWith("/"))
		            out.append("/");
                out.append(key);
            }
            if(value != null)
            {
	            if(!out.toString().endsWith("/"))
		            out.append("/");
		        if(value.indexOf("/") >= 0)
		        {
		        	out.append( StringUtil.replaceSubString(
			        	URLEncoder.encode(value), "%2F", "/") );
		        }
		        else
		        	out.append(URLEncoder.encode(value));
            }
        }
        return out.toString();
    }
    /**
     * Override parent to make more efficient the adding
     * of path info from vector to path.
     *
     * @see DynamicURI#renderQueryString(Vector data)
     */
    protected String renderQueryString(Vector data)
    {
        String key = null;
        String value = null;
        StringBuffer out = new StringBuffer("");
        Iterator dataIter = data.iterator();
        while(dataIter.hasNext())
        {
            Object[] entry = (Object[])dataIter.next();
            key = URLEncoder.encode(entry[0].toString());
            value = entry[1].toString();
            
            if( (key != null || key.length() == 0) 
	            || (value != null || value.length() == 0))
            {
	            if( (out.length() > 0)
		            && (!out.toString().endsWith("&")) 
		            && (!out.toString().endsWith("?")) )
	           	{ //assume someone else will add ? if not there
		            out.append("&");
	           	}
                out.append(key);
                out.append("=");
		        out.append(URLEncoder.encode(value));
            }
        }
        return out.toString();
    }
    /**
     * If a portlet value is supplied then it is added to the
     * path info.
     *
     * @see AdapterPortletURI#setPortlet(String portlet)
     */
	public void setPortlet(String portlet)
	{
//		if((portlet != null && portlet.length() > 0) && portletAdded == false)
//		{
//			addPathInfo("", portlet);
			piid = portlet;
//			portletAdded = true;
	        isDirty = true;
	        isPathInfoDirty = true;
//		}
	}
    /**
     * Method to specify that a URI should use SSL.
     *
     * @param enable boolean Specifies to enable or disable.
     */
    public void setSSL(boolean enable)
    {
	    if(enable)
		    super.setSecure();
		else if(!getServerScheme().equals(HTTP))
		{
			setServerScheme(HTTP);
			setServerPort(80);
		}
	}
/**
 * Set the window state and add as an action to the URI path info.
 *
 * Creation date: (6/13/2002 11:39:39 AM)
 * 
 * @param newState com.yci.seb.portal.adapterportlet.AdapterPortletWindow.State
 */
public void setState(AdapterPortletWindow.State newState) 
{
	state = newState;
	//FIXME:  we are only supporting the Maximize state at this time.
	// we need a better method of relating a window state to a control action.
	if(state.equals(AdapterPortletWindow.State.MAXIMIZED))
	{
		addQueryData("action", "controls.SEBMaximize");
		addQueryData("portlet", piid);
		isDirty = true;
		isQueryStringDirty = true;
	}
	else if(state.equals(AdapterPortletWindow.State.MINIMIZED))
		;
	else if(state.equals(AdapterPortletWindow.State.CLOSED))
	{
		//addQueryData("action", "controls.SEBMaximize");
		//addQueryData("portlet", piid);
		//isDirty = true;
		//isQueryStringDirty = true;
	}
	else if(state.equals(AdapterPortletWindow.State.NORMAL))
	{
		addQueryData("action", "controls.Restore");
		isDirty = true;
		isQueryStringDirty = true;
	}
	else
		;
}
	private void splitURI()
	{
	    // get base URI of Portal pointing to the given portlet, mode and state
	    String completeURI =
	        provider.getPortletURI(piid, pid, mode, state, actionreference);
		isDirty = true;
		isPathInfoDirty = true;

	    // find protocol
	    int idx = completeURI.indexOf("//");
	    // find begin of path
	    int pathIdx = completeURI.indexOf('/', idx + 2);
	    // find begin of parameters
	    int parametersIdx = completeURI.indexOf('?', idx + 2);
	    // find begin of session id
	    int sessionIdx = completeURI.indexOf(';', idx + 2);

	    // get host
	    if((pathIdx != -1) && (parametersIdx != -1) && (pathIdx > parametersIdx))
	        pathIdx = parametersIdx;
	    else if ((pathIdx == -1) && (parametersIdx != -1))
	        pathIdx = parametersIdx;
	    if((pathIdx != -1) && (sessionIdx != -1) && (pathIdx > sessionIdx))
	        pathIdx = sessionIdx;
	    else if ((pathIdx == -1) && (sessionIdx != -1))
	        pathIdx = sessionIdx;
	    if(pathIdx == -1)
	    {
	        uriHost = completeURI;
	        pathIdx = completeURI.length();
	    }
	    else
	        uriHost = completeURI.substring(0, pathIdx);

	    // get other parts of uri
	    if ((parametersIdx == -1) && (sessionIdx == -1))
	    { // no parameters AND no session id
	        uriPath = completeURI.substring(pathIdx);
	    }
	    else if (parametersIdx == -1)
	    { // no parameters BUT session id
	        uriPath = completeURI.substring(pathIdx, sessionIdx);
	        uriSession = completeURI.substring(sessionIdx);
	    }
	    else if (sessionIdx == -1)
	    { // no session id BUT parameters
	        uriPath = completeURI.substring(pathIdx, parametersIdx);
	        uriParameters = completeURI.substring(parametersIdx);
	        isQueryStringDirty = true;
	    }
	    else
	    { 	// there is both, the session id and the parameters
	        // check if session id occurs before parameters
	        if (sessionIdx < parametersIdx)
	        {
	            uriPath = completeURI.substring(pathIdx, sessionIdx);
	            uriSession = completeURI.substring(sessionIdx, parametersIdx);
	            uriParameters = completeURI.substring(parametersIdx);
	            isQueryStringDirty = true;
	        }
	        else
	        {
	            // else, the parameters occur before the session id
	            uriPath = completeURI.substring(pathIdx, parametersIdx);
	            uriParameters = completeURI.substring(parametersIdx, sessionIdx);
	            uriSession = completeURI.substring(sessionIdx);
	            isQueryStringDirty = true;
	        }
	    }		
	}
	/**
	 * Returns a String Representation of the URI for the current Portlet and Request.
	 *
	 * @return java.lang.String
	 */
	public String toString()
    {
        if(isDirty)
        {
	        fullBuffer = new StringBuffer(getPathInfo());
	        fullBuffer.append(getQueryString());
	        
	        //If session information as URL rewrite then add to URI.
		    if (uriSession!=null) 
   				fullBuffer.append(uriSession);
	
			isDirty = false;
	    }		
        return fullBuffer.toString();
    }
}
