package com.yci.seb.portal.controllers;

//Jetspeed support
import org.apache.jetspeed.portal.*;
import org.apache.jetspeed.portal.controllers.*;
import org.apache.jetspeed.services.*;
import org.apache.jetspeed.services.template.*;
import org.apache.jetspeed.util.Config;

//Turbine support
import org.apache.turbine.util.RunData;
import org.apache.turbine.modules.ActionLoader;
import org.apache.turbine.services.jsp.JspService;
import org.apache.turbine.services.TurbineServices;

//ECS support
import org.apache.ecs.ConcreteElement;
import org.apache.ecs.StringElement;

//Adapter Portlet Api support
import com.yci.seb.portal.adapterportlet.*;
import com.yci.seb.portal.adapterportletcontainer.*;

/**
 * Enables rendering of a collection of portlets from a PSML.
 * The layout format of the presentation is defined by this class
 * and a corresponding JSP used to define the specific layout
 * properties (such as columns, multi-columns, rows, etc).
 *
 * @version Modificaitons:
 * 	<p>01/25/2002	-	Marcus Mosttler	-	Initial Creation.
 * 	<p>06/18/2002	-	Daniel Joseph	-
 *		The current instance of JspPortletController is	added to Request
 *		in the getHTMLContent method
 * 	<p>06/19/2002	-	Marcus Mosttler	-	
 *		Added getControllerPath and setControllerPath to allow for
 * 		the controller to set a special path for the controller.  This is
 *		used by the AdapterPortletURI to append the controller path to the
 *		URI being generated (main purpose is for the paned controller).
 * 	<p>08/15/2002	-	Marcus Mosttler	-
 *		Added static method to check for useSessionState parameter to
 *		determine if controller, control, and portlet should use their
 *		session state or not.
 */
public class JspPortletController extends AbstractPortletController 
{
    public static final String NAVIGATION_PARAMETER = "pane";
    
    /**
	 * Used to provide logging capabilities via Log4J.
	 */
	protected static org.apache.log4j.Logger logger;

	protected static final String ATTR_CONTROLLER_PATH = "controllerPath";
	protected static final String ATTR_USE_SESSION_STATE = "useSessionState";
	
	public JspPortletController()
	{
		super();
		logger = org.apache.log4j.Logger.getLogger(this.getClass().getName());
	}
/**
 * Returns the special path for the controller from the request.
 * This is done so the AdapterPortletURI can pull the
 * controller path and add it to any URI paths is creates.
 * 
 * Creation date: (6/19/2002 11:32:19 AM)
 * 
 * @param request javax.servlet.http.HttpServletRequest
 * @return java.lang.String Controller Path
 */
public static String getControllerPath(javax.servlet.http.HttpServletRequest request) 
{
	Object controllerPath = request.getAttribute(ATTR_CONTROLLER_PATH);
	if(controllerPath == null)
		return "";
	else
		return controllerPath.toString();
}
	/**
	 *
	 * @param portlets PortletSet
	 * 		A collection of portlets to be rendered by this controller.
	 * @param rundata RunData
	 * 		The RunData representing this request created by the Turbine Servlet.
	 * @return ConcreteElement
	 * 		This is not used but is required as a legacy hold over of ECS.
	 */
	protected ConcreteElement getHTMLContent(PortletSet portlets, RunData rundata)
	{
		if(logger.isDebugEnabled())
			logger.debug("In getHTMLContent");
			
		//place data on request needed by the controller view template.
		rundata.getRequest().setAttribute("controllerPortlets", portlets.getPortlets());
		rundata.getRequest().setAttribute("controllerSkin",
			portlets.getPortletConfig().getPortletSkin());
		rundata.getRequest().setAttribute("controllerConfig", getConfig());
		rundata.getRequest().setAttribute("controller", this);

        if(logger.isDebugEnabled())
        {
			logger.debug("Using Controller Portlet Skin [ " 
				+ portlets.getPortletConfig().getPortletSkin().getName() + " ]");
        }
		
	    String actionName = getConfig().getInitParameter("action");	
	    if(actionName != null)
	    {   // if there is an action with the same name in modules/actions/portlets exec it
	        try
	        {
	            ActionLoader.getInstance().exec(rundata, actionName);
	        }
	        catch(Exception e)
	        {
	            logger.error(e);
	        }
	    }
	
	    // either the action selected the template, or use the default template 
	    // defined in the registry
	    String template = getConfig().getInitParameter("template");
	
	    // generate the content
	    String errorString = "";
	    try
	    {
	        if(-1 == template.indexOf(".jsp"))
	            template = template + ".jsp";
	
	        //we use the template locator service to translate the template
	        String locatedTemplate =
	            TemplateLocator.locateControllerTemplate(rundata, template);
	
	        JspService service =
	            (JspService) TurbineServices.getInstance().getService(JspService.SERVICE_NAME);
	        //this is only necessary if we don't run in a JSP page environment
	        //but better be safe than sorry...
	        service.addDefaultObjects(rundata);
	
	        //handle request
	        service.handleRequest(rundata, locatedTemplate);
	        
	        //Clean up the request attributes.
	        rundata.getRequest().removeAttribute("controllerPortlets");
			rundata.getRequest().removeAttribute("controllerSkin");
			rundata.getRequest().removeAttribute("controllerConfig");
			rundata.getRequest().removeAttribute("controller");
	    }
	    catch(Exception e)
	    {
	        errorString = e.toString();
	    }
	    return new StringElement(errorString);
	}
/**
 * Adds a special path for the controller to the request.
 * This is done so the AdapterPortletURI can pull the
 * controller path and add it to any URI paths is creates.
 * 
 * Creation date: (6/19/2002 11:32:19 AM)
 * 
 * @param request javax.servlet.http.HttpServletRequest
 * @param controllerPath java.lang.String
 */
protected void setControllerPath(javax.servlet.http.HttpServletRequest request, 
	String controllerPath) 
{
	request.setAttribute(ATTR_CONTROLLER_PATH, controllerPath);
}
    /**
 	 * Check for useSessionState parameter to determine if 
 	 * controller, control, and portlet should use their session state or not.
     *
     * @param config Config The controller, control or portlet config parameters.
     * @param request AdapterPortletRequest The Request Session Data.
     *
     * @return boolean 
     *		Value from request parameter or if not present the value from 
     *		xreg/psml or true if not found or accessible.
     */
	public static boolean useSessionState(Config config, AdapterPortletRequest request)
	{
		String result;
		//check portlet request
		result = request.getParameter(ATTR_USE_SESSION_STATE);
		if(result == null)
		{	//now check http request incase it is a global request parameter.
			result = ((AdapterPortletRequestImpl)request).getServletRequest()
				.getParameter(ATTR_USE_SESSION_STATE);
			if(result == null)
			{	//now check xreg/psml
				result = config.getInitParameter(ATTR_USE_SESSION_STATE, Boolean.TRUE.toString());
			}
		}
			
        return Boolean.valueOf(result).booleanValue();
	}
    /**
 	 * Check for useSessionState parameter to determine if 
 	 * controller, control, and portlet should use their session state or not.
     *
     * @param config Config The controller, control or portlet config parameters.
     * @param rundata RunData The Request Session Data.
     *
     * @return boolean 
     *		Value from request parameter or if not present the value from 
     *		xreg/psml or true if not found or accessible.
     */
	public static boolean useSessionState(Config config, RunData rundata)
	{
		String result;

		if(rundata.getParameters().containsKey(ATTR_USE_SESSION_STATE))
			result = rundata.getParameters().getString(ATTR_USE_SESSION_STATE);
		else
			result = config.getInitParameter(ATTR_USE_SESSION_STATE, Boolean.TRUE.toString());
			
        return Boolean.valueOf(result).booleanValue();
	}
}
