package com.jcraft.jsch;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;

import junit.framework.Assert;

import org.junit.Before;
import org.junit.Test;

import java.util.concurrent.ExecutionException;

public class SeesionParallelTest {

	private static final String host1 = "host1";
	private static final String user1 = "user1";
	private static final String pass1 = "password1";
	private static final String command1 = "pwd";
	private static final String expectedOutput1 = "/home/user1";

	private static final String host2 = "host2";
	private static final String user2 = "user2";
	private static final String pass2 = "password2";
	private static final String command2 = "pwd";
	private static final String expectedOutput2 = "/home/user2";

	private static final String host3 = "host3";
	private static final String user3 = "user3";
	private static final String pass3 = "password3";
	private static final String command3 = "pwd";
	private static final String expectedOutput3 = "/home/user3";

	@Before
	public void setUp() throws Exception {
		Session.setUnauthenticatedLimit(Session.UNAUTH_NO_LIMIT);
	}

	/**
	 * Verifies that changing the limit from one number to a different valid limit will not
	 * cause any Exception to be thrown.
	 * 
	 */
	public void testChangingLimits() throws Exception {
		Session.setUnauthenticatedLimit(10);
		Session.setUnauthenticatedLimit(5);
	}

	/**
	 * Tests making multiple, concurrent requests with no unauthenticated limit.
	 * Expects to get JSchException.
	 * <p>
	 * If no exception is thrown, we are working against a server that allows
	 * unlimited unauthenticated connections.
	 * 
	 * @throws Exception
	 */
	@Test(expected = com.jcraft.jsch.JSchException.class)
	public void testBaseline() throws Throwable {
		try{
			makeParallelConnections(50);
		}finally{
			//allow any existng connection attempts time to finish
			Thread.sleep(1500);
		}
	}

	/**
	 * Tests setting limit to 0 throws exception
	 * 
	 * @throws Exception
	 */
	@Test(expected = com.jcraft.jsch.JSchException.class)
	public void testInvalidLimit1() throws Throwable {
		Session.setUnauthenticatedLimit(0);
	}

	/**
	 * Tests setting limit to negative number throws exception
	 * 
	 * @throws Exception
	 */
	@Test(expected = com.jcraft.jsch.JSchException.class)
	public void testInvalidLimit2() throws Throwable {
		Session.setUnauthenticatedLimit(-3);
	}

	
	/**
	 * Tests changing the limit to the same number does not cause any errors.
	 */
	@Test
	public void testChangeLimitToSameNumber() throws Exception {
		Session.setUnauthenticatedLimit(10);
		Session.setUnauthenticatedLimit(10);
	}

	/**
	 * Tests changing the limit to the same number does not cause any errors.
	 */
	@Test
	public void testChangeLimitToDifferentNumber() throws Exception {
		Session.setUnauthenticatedLimit(10);
		Session.setUnauthenticatedLimit(12);
		Session.setUnauthenticatedLimit(4);
	}

	/**
	 * Tests changing the limit to the same number does not cause any errors.
	 */
	@Test
	public void testLowerTheLimitDuringRuntime() throws Throwable {
		Session.setUnauthenticatedLimit(10);
		HashMap<Connector, Future<Integer>> firstResults = makeParallelConnections(100,false);

		Session.setUnauthenticatedLimit(5);
		HashMap<Connector, Future<Integer>> secondResults = makeParallelConnections(50,false);

		checkFutures(firstResults.values());
		checkFutures(secondResults.values());

		checkConnectors(firstResults.keySet());
		checkConnectors(secondResults.keySet());
	}

	/**
	 * Test simple case of connection to single machine in parallel after setting the unauthenticated connection limit.
	 * 
	 * @throws Throwable
	 */
	@Test
	public void testUnauthLimit() throws Throwable {
		Session.setUnauthenticatedLimit(10);
		Collection<Connector> connectors = makeParallelConnections(100).keySet();

		for (Connector c : connectors) {
			Assert.assertTrue(c.getSucceeded());
		}
	}

	/** 
	 * Tests connection to multiple machines in parallel while changing the unauthenticated connection limit.
	 * @throws Throwable
	 */
	@Test
	public void testConcurrentConnectionsToMulipleMachines() throws Throwable {
		
		Assert.assertNotSame("Test requires different hosts", host1, host2);
		Assert.assertNotSame("Test requires different hosts", host1, host3);
		Assert.assertNotSame("Test requires different hosts", host2, host3);
		
		Session.setUnauthenticatedLimit(10);
		HashMap<Connector, Future<Integer>> firstResults = makeParallelConnections(user1, host1, pass1, command1, expectedOutput1, 100, false);

		Session.setUnauthenticatedLimit(5);
		HashMap<Connector, Future<Integer>> secondResults = makeParallelConnections(user2, host2, pass2, command2, expectedOutput2, 100, false);

		Session.setUnauthenticatedLimit(8);
		HashMap<Connector, Future<Integer>> thirdResults = makeParallelConnections(user3, host3, pass3, command3, expectedOutput3, 100, false);

		checkFutures(firstResults.values());
		checkFutures(secondResults.values());
		checkFutures(thirdResults.values());

		checkConnectors(firstResults.keySet());
		checkConnectors(secondResults.keySet());
		checkConnectors(thirdResults.keySet());
	}

	
	
	/**
	 * calls makeParallelConnections(count, checkResults) with checkResults set to true.
	 * 
	 * @param count number of processes to run in parallel
	 * @return map.  the key is the Connector, the value is the corresponding Future
	 * @throws Throwable will re-throw any exception encountered.
	 */
	protected HashMap<Connector, Future<Integer>> makeParallelConnections(int count) throws Throwable {
		return makeParallelConnections(count, true);
	}

	
	/**
	 * Runs <code>count</code> connections in parallel.  If <code>checkResults</code> is true, it will wait for completion
	 * and check that no exceptions were thrown and that each Connector succeeded.
	 * <p>
	 * This method calls makeParallelConnections(String, String, String, String, String, count, checkResults).
	 * 
	 * @param count number of processes to run in parallel
	 * @param checkResults whether to check the results
	 * @return map.  the key is the Connector, the value is the corresponding Future
	 * @throws Throwable if checkResults was true, will re-throw any exception encountered.
	 */
	protected HashMap<Connector, Future<Integer>> makeParallelConnections(int count, boolean checkResults) throws Throwable {
		return makeParallelConnections(user1, host1, pass1, command1, expectedOutput1, count, checkResults);
	}
	
	/**
	 * Runs <code>count</code> connections in parallel.  If <code>checkResults</code> is true, it will wait for completion
	 * and check that no exceptions were thrown and that each Connector succeeded.
	 * <p>
	 * Connects with given parameters
	 * 
	 * @param user user to connect as
	 * @param host host to connect to
	 * @param pass password to use
	 * @param command command to execute on remote side
	 * @param expectedOutput expected output from the command
	 * @param count number of connections to run in parallel
	 * @param checkResults whether to wait for all processes to finish and verify no exceptions thrown and expected output matches actual output
	 * @return map.  the key is the Connector, the value is the corresponding Future
	 * @throws Throwable if checkResults was true, will re-throw any exception encountered.
	 */
	protected HashMap<Connector, Future<Integer>> makeParallelConnections(String user, String host, String pass, String command, String expectedOutput, int count, boolean checkResults) throws Throwable {
		HashMap<Connector, Future<Integer>> results = new HashMap<Connector, Future<Integer>>();
		
		ExecutorService pool = Executors.newCachedThreadPool();

		for (int i = 0; i < count; i++) {
			Connector connector = new Connector(user, host, pass, command, expectedOutput, i);
			Future<Integer> future = pool.submit(connector);

			results.put(connector, future);
		}

		pool.shutdown();

		if (checkResults) {
			// Check to see if any exceptions where thrown.
			// If so, rethrow them.
			
			checkFutures(results.values());
			checkConnectors(results.keySet());
		}

		return results;
	}

	
	/**
	 * Checks the given collection of Futures to verify no exception was thrown.
	 * 
	 * @param futures list of futures to check
	 * @throws Throwable re-throws the first exception it finds (if any)
	 */
	protected void checkFutures(Collection<Future<Integer>> futures) throws Throwable{
		for (Future<Integer> f : futures) {
			try {
				f.get();
			} catch (ExecutionException e) {
				throw e.getCause();
			}
		}
	}


	/**
	 * Checks the list of connectors to verify each succeeded
	 * 
	 * @param connectors list of connectors to check
	 */
	protected void checkConnectors(Collection<Connector> connectors) {
		for (Connector connector : connectors) {
			Assert.assertTrue(connector.getSucceeded());
		}
	}

}
