unit EasyDockSite;
(* Docking manager by DoDi, imported from Delphi.
To be added or ported:
  - message handling
  - header painting (header should become a control, based on splitter)
  - anchoring
  - splitters
  - field and method argument names

More issues:
  The Controls.TDockZone class deserves modifications, when it shall become usable.
  Then the visibility of some members has to be changed, when new docking managers
  shall use that class.
  IMO every dock tree should have its private TDockZone class, with full access
  to all members, as required. Remove this class from Controls.

  The Orientation handling sucks :-(
  A new SiblingsVertical method/property should replace all references to
  TDockZone.Orientation.
  TControl.DockOrientation is obsolete, can be removed. The orientation has to
  be reflected in the alignment and anchoring, and can be retrieved from there.

  Lazarus does not handle docking manager as interfaces.
  I suspect a memory leak, when a docking manager is exchanged.
  In a docking manger, ported from Delphi, the methods inherited from the
  TDockManager class must be marked as override.
*)

{$MODE Delphi}

{.$DEFINE WndProc} //using WndProc?

interface

uses
{$IFDEF fpc}
  LCLIntf, //TRect
  LCLType, //HDC
{$ELSE}
  //LCLIntf, //TRect
  Messages, //TMessage...
{$ENDIF}
  Classes, //TStream
  Forms,
  ExtCtrls, //splitter - not yet working
  Controls;

type
  TEasyTree = class; //forward declaration
  TChildControl = TControl; //experimental: TForm

  TEasyZone = class //(TDockZone)
  private
    _Tree: TEasyTree;
    _ChildControl: TChildControl;
    _FirstChild,
    _NextSibling, _PrevSibling, _Parent: TEasyZone;
    _Orientation: TDockOrientation;
    procedure SetControl(Control: TChildControl);
  {$IFDEF old}
    procedure SetFirstChild(zone: TEasyZone);
    procedure SetNextSibling(zone: TEasyZone);
    procedure SetPrevSibling(zone: TEasyZone);
    procedure SetOrientation(NewOrientation: TDockOrientation);
    procedure SetBounds(TLBR: TRect; NewOrientation: TDockOrientation); //unlink if parent changes
    function GetHeight: Integer;
    function GetWidth: Integer;
  {$ELSE}
    procedure SetBounds(TLBR: TRect);
  {$ENDIF}
    function GetLeft: Integer;
    function GetTop: Integer;
  {$IFDEF old}
    function GetBottom: Integer;
    function GetBottomOrRight(fBottom: boolean): integer;
    function GetRight: Integer;
  {$ELSE}
  {$ENDIF}
    function GetTopOrLeft(fTop: boolean): Integer;
  private //very basic liniking
  {$IFDEF old}
    procedure AddAsFirstChild(zone: TEasyZone);
    function  RemoveChild(zone: TEasyZone): TEasyZone;
      //return non-empty zone (to refresh)
    procedure SetLimit(AValue: integer; AOrientation: TDockOrientation);
      //propagate into control. Orientation??? (should be automatic!)
  {$ELSE}
    procedure InsertAfter(LinkAfter, NewZone: TEasyZone);
      //do not handle orientation!
    procedure SetParent(zone: TEasyZone);
      //unlink
  {$ENDIF}
  protected
    function GetBounds: TRect;
    function GetHeaderSize: integer; virtual;
    function GetVisible: boolean;
    function GetVisibleControl: TChildControl;
  public
    constructor Create(ATree: TEasyTree);
    destructor Destroy; override;
    procedure Clear;
    function  DockSite: TWinControl;
    procedure AddSibling(NewZone: TEasyZone; where: TAlign);
    procedure ReplaceChild(OldChild, NewChild: TEasyZone);
    //function ScaleTo(rOld, rNew: TRect): integer;
    procedure ScaleTo(const ptOld, ptNew, ptOuter: TPoint);
  public //instead of properties
  {$IFDEF old}
    Limit: integer;
  {$ELSE}
    BR: TPoint;
  {$ENDIF}
    property ChildControl: TChildControl read _ChildControl write SetControl;
    property FirstChild: TEasyZone read _FirstChild;  // write SetFirstChild;
    property NextSibling: TEasyZone read _NextSibling;  // write SetNextSibling;
    property PrevSibling: TEasyZone read _PrevSibling;  // write SetPrevSibling;
    property Parent: TEasyZone read _Parent write SetParent;
    property Orientation: TDockOrientation read _Orientation write _Orientation; // write SetOrientation;
    property Visible: boolean read GetVisible;

    property Bottom: integer read BR.Y;  // GetBottom;
    property Left: integer read GetLeft;
    property Right: integer read BR.X;  // GetRight;
    property Top: integer read GetTop;

  {$IFDEF old}
    property Height: integer read GetHeight;
    property Width: integer read GetWidth;
  {$ELSE}
  {$ENDIF}
  end;

  TEasyTree = class(TDockTree)
  //TEasyTree = class(TInterfacedObject, IDockManager)
  private
    FDockSite: TWinControl;
    //FReplacingControl: TControl;
    FTopZone: TEasyZone;
    FSiteRect: TRect; //to detect changed site extent
    FSplitter: TSplitter;
    FUpdateCount: integer;
    procedure UpdateTree;
  protected //interface
  (* Unlike in Delphi, all inherited methods must be marked "override"!
  *)
    procedure BeginUpdate; override;
    procedure EndUpdate; override;
    procedure GetControlBounds(Control: TControl; out CtlBounds: TRect);  override;
    procedure InsertControl(Control: TControl; InsertAt: TAlign;
      DropCtl: TControl);  override;
    procedure PositionDockRect(Client, DropCtl: TControl; DropAlign: TAlign;
      var DockRect: TRect);  override;
    procedure RemoveControl(Control: TControl);  override;
    procedure ResetBounds(Force: Boolean);  override; //site resized
    procedure SetReplacingControl(Control: TControl); override; //unused
    procedure LoadFromStream(Stream: TStream);  override; //virtual;
    procedure SaveToStream(Stream: TStream);  override; //virtual;
  protected //added
    procedure BuildDockLayout(zone: TEasyZone); //unused/nop
    function  DockHeaderSize: integer; virtual;
    function  FindControlZone(zone: TEasyZone; Control: TControl): TEasyZone;
    procedure RemoveZone(Zone: TEasyZone);
    //procedure UpdateTree;
  {$IFDEF wndproc}
  protected //message handling
    FOldWndProc: TWndMethod;
    function  InternalHitTest(const MousePos: TPoint; out HTFlag: Integer): TEasyZone;
    procedure WindowProc(var Message: TMessage);
  {$ELSE}
  {$ENDIF}
  public
    constructor Create(DockSite: TWinControl);  override; //virtual;
    destructor Destroy; override;
    procedure AdjustDockRect(Control: TControl; var ARect: TRect); override; //virtual;
    procedure PaintSite(DC: HDC); override; //virtual;
  end;

var
  DropOn: TControl;

implementation

uses SysUtils, Types;

{ TEasyTree }

constructor TEasyTree.Create(DockSite: TWinControl);
begin
  inherited Create(DockSite);
  FDockSite := DockSite;
  FSiteRect := DockSite.ClientRect;
  FTopZone := TEasyZone.Create(self);
  FTopZone.SetBounds(FSiteRect);
{$IFDEF wndproc}
  if not (csDesigning in DockSite.ComponentState) then begin
    FOldWndProc := FDockSite.WindowProc;
    FDockSite.WindowProc := WindowProc;
    FSplitter := TSplitter.Create(nil);
    FSplitter.Parent := DockSite;
  end;
{$ELSE}
{$ENDIF}
end;

destructor TEasyTree.Destroy;
begin
  FreeAndNil(FTopZone);
  FreeAndNil(FSplitter);
  inherited;
end;

procedure TEasyTree.BeginUpdate;
begin
  inc(FUpdateCount);
end;

procedure TEasyTree.EndUpdate;
begin
  dec(FUpdateCount);
  if (FUpdateCount = 0) and (FTopZone.FirstChild <> nil) then begin
    BuildDockLayout(FTopZone);
  end;
end;

function TEasyTree.DockHeaderSize: integer;
begin
  Result := 10;
end;

procedure TEasyTree.AdjustDockRect(Control: TControl; var ARect: TRect);
begin
  //test!
{
  if Control.DockOrientation = doVertical then
    inc(ARect.Top, DockHeaderSize)
  else
    inc(ARect.Left, DockHeaderSize);
}
end;

function TEasyTree.FindControlZone(zone: TEasyZone; Control: TControl): TEasyZone;
begin
  Result := zone;
  if Result = nil then
    exit;
  if Result.ChildControl = Control then
    exit;
  zone := zone.FirstChild;
  Result := nil;
  while (zone <> nil) and (Result = nil) do begin
    Result := FindControlZone(zone, Control);
    zone := zone.NextSibling;
  end;
end;

procedure TEasyTree.GetControlBounds(Control: TControl;
  out CtlBounds: TRect);
var
  zone: TEasyZone;
begin
//zone in client TLBR
  zone := FindControlZone(FTopZone, Control);
  if zone = nil then
    CtlBounds := Rect(0,0,0,0)
  else begin
    CtlBounds := Control.BoundsRect;
  end;
end;

procedure TEasyTree.InsertControl(Control: TControl; InsertAt: TAlign;
  DropCtl: TControl);
var
  DropZone, OldZone, NewZone, OldParent, NewParent: TEasyZone;
  r: TRect;
(* special cases:
  1) first child in top zone - no orientation
  2) second child in top zone - determines orientation
In all other cases all zones have an orientation:
  3) insert isogonal
  4) insert orthogonal

Cases 2 and 3 can be merged, with an additional or redundant setting of the orientation.
*)

begin
//some checks
  if DropCtl = nil then
    DropCtl := FDockSite; //the dock site
  if (Control = nil) or (not Control.Visible) or (DropCtl = Control) then
    exit; //nothing changed
  if not (Control is TWinControl) then
    exit; //for now, could wrap control into a form here

  DropZone := FindControlZone(FTopZone, DropCtl);
  //if DropZone = nil then exit; //not here!?

  NewZone := TEasyZone.Create(self);
  NewZone.ChildControl := Control as TChildControl;
//redock (assume control has been undocked before?)
  //Control.Parent := FDockSite;

//special case: in root zone (empty dock site)
  if FTopZone.FirstChild = nil then begin
    FTopZone.InsertAfter(nil, NewZone);
    NewZone.SetBounds(FDockSite.ClientRect);
  end else begin
  //more checks
    OldZone := FindControlZone(FTopZone, Control);
      //check after placing the control
    r := DropZone.GetBounds; //for later adjustment
  //get requested orientation, adjust align
    case InsertAt of
    alTop, alBottom: Control.DockOrientation := doVertical;
    alLeft, alRight: Control.DockOrientation := doHorizontal;
    else //unhandled or unspecific
      if DropCtl.DockOrientation = doNoOrient then
        DropCtl.DockOrientation := doHorizontal; //assume
      Control.DockOrientation := DropCtl.DockOrientation;
    //fix alignment
      if Control.DockOrientation = doVertical then
        InsertAt := alBottom
      else
        InsertAt := alRight;
    end;
    (* Now Control.DockOrientation is the insert orientation,
      DropCtl.DockOrientation is the zone orientation,
      InsertAt is one of alLeft/Right/Top/Bottom
    *)

  //check orientation - control orientation cannot be doNone!
    OldParent := DropZone.Parent;
    (* One special case remains: top zone without orientation
    *)
    if (OldParent.Orientation = doNoOrient) then begin
      assert(OldParent = FTopZone, '???');
      FTopZone.Orientation := Control.DockOrientation; //easy
    end;
  //iso or orthogonal insert?
    if (OldParent.Orientation <> Control.DockOrientation) then begin
    //need intermediate zone
      NewParent := TEasyZone.Create(self);
      NewParent.Orientation := Control.DockOrientation;
      NewParent.BR := r.BottomRight;
      OldParent.ReplaceChild(DropZone, NewParent); //unlink DropZone
    //orthogonal orientation
      //DropZone.SetBounds(r);
      NewParent.InsertAfter(nil, DropZone);
    end;
  //set control orientation
    DropZone.AddSibling(NewZone, InsertAt); //could check and handle parent insertion!

  //clear eventually moved zone
    if OldZone <> nil then begin
    //must rebuild both zones!
      RemoveZone(OldZone); //must NOT modify moved control!
      OldZone.Free;
      BuildDockLayout(FTopZone);
    end else //adjust extents
      BuildDockLayout(DropZone.Parent); //new zone inserted here
  end;
  FDockSite.Invalidate;
end;

procedure TEasyTree.LoadFromStream(Stream: TStream);
begin
  //todo
end;

procedure TEasyTree.PaintSite(DC: HDC);
begin
  //nop, for now
end;

procedure TEasyTree.PositionDockRect(Client, DropCtl: TControl;
  DropAlign: TAlign; var DockRect: TRect);
begin
//debug!
  DropOn := DropCtl;

  if (DropCtl = nil) or (DropCtl = FTopZone.ChildControl) then
    exit;

  DockRect := DropCtl.BoundsRect;
  DockRect.TopLeft := FDockSite.ClientToScreen(DockRect.TopLeft);
  DockRect.BottomRight := FDockSite.ClientToScreen(DockRect.BottomRight);

  case DropAlign of
  alTop:    DockRect.Bottom := (DockRect.Top + DockRect.Bottom) div 2;
  alBottom: DockRect.Top := (DockRect.Top + DockRect.Bottom) div 2;
  alLeft:   DockRect.Right := (DockRect.Left + DockRect.Right) div 2;
  alRight:  DockRect.Left := (DockRect.Left + DockRect.Right) div 2;
  end;
end;

procedure TEasyTree.RemoveControl(Control: TControl);
var
  zone: TEasyZone;
begin
//propagate changes into parent zones
  zone := FindControlZone(FTopZone, Control);
  if zone <> nil then begin
    RemoveZone(zone);
  end;
end;

procedure TEasyTree.ResetBounds(Force: Boolean);
var
  rNew: TRect;
begin
//drop site resized
  if (csLoading in FDockSite.ComponentState) then
    exit; //not the right time to do anything
  if FTopZone.FirstChild = nil then
    exit; //zone is empty, nothing to do
//how to determine old bounds?
  rNew := FDockSite.ClientRect;
  if not CompareMem(@rNew, @FSiteRect, sizeof(rNew)) then
    Force := True;  //something has changed
  if not Force then
    exit;
  FTopZone.ScaleTo(FSiteRect.BottomRight, rNew.BottomRight, rNew.BottomRight);
  FSiteRect := rNew;
end;

procedure TEasyTree.SaveToStream(Stream: TStream);
var
  r: TRect;
  s: string;
const
  eol: string = #13#10; //to be replaced by system constant?
  OrientString: array[TDockOrientation] of char = ('N','H','V'
    {$IFDEF FPC} ,'P' {$ENDIF} );

  procedure WriteZone(zone: TEasyZone; level: integer);
  var
    ind: string;
    ctl: TChildControl;
  begin
    ind := StringOfChar(' ', level*2);
  //zone
    r := zone.GetBounds;
    s := Format('%s%s (%d,%d)-(%d,%d)%s', [ind, OrientString[zone.orientation], //zone.Limit,
      r.Top, r.Left, r.Bottom, r.Right, eol]);
    Stream.Write(s[1], length(s));
  //control
    ctl := zone.ChildControl;
    if ctl <> nil then begin
      r := ctl.BoundsRect;
      s := Format('%s%s (%d,%d)-(%d,%d)%s', [ind, ctl.Name,
        r.Top, r.Left, r.Bottom, r.Right, eol]);
      Stream.Write(s[1], length(s));
      if ctl.Visible then
        s := 'visible'
      else
        s := 'hidden';
      s := ind + s + eol;
      Stream.Write(s[1], length(s));
    end;
    zone := zone.FirstChild;
    while zone <> nil do begin
      WriteZone(zone, level+1);
      zone := zone.NextSibling;
    end;
    Stream.Write(eol[1], length(eol));
  end;

begin
//for now: dump tree
{$IFDEF wndproc}
//splitter
  if FSplitter.Visible then begin
    r := FSplitter.BoundsRect;
    s := Format('Splitter (%d,%d)-(%d,%d)%s', [
      r.Top, r.Left, r.Bottom, r.Right, eol]);
    Stream.Write(s[1], length(s));
  end;
{$ELSE}
{$ENDIF}
  WriteZone(FTopZone, 0);
end;

procedure TEasyTree.SetReplacingControl(Control: TControl);
begin
  //FReplacingControl := Control;
end;

procedure TEasyTree.UpdateTree;
begin
  //todo
  //FDockSite.Invalidate;
end;

procedure TEasyTree.BuildDockLayout(zone: TEasyZone);
begin
  //nothing to do?
end;

procedure TEasyTree.RemoveZone(Zone: TEasyZone);
var
  p, ch: TEasyZone;

  procedure InlineKids(z: TEasyZone);
  var
    pp, p, ch: TEasyZone;
  begin
  (* move all children of z into z.parent.parent, in place of z.parent
  *)
    p := z.Parent;
    pp := p.Parent;
  //link first child
    ch := z.FirstChild;
    ch._PrevSibling := p._PrevSibling;
    if ch.PrevSibling <> nil then
      ch.PrevSibling._NextSibling := ch
    else
      pp._FirstChild := ch;
    ch._Parent := pp;
  //link last child
    while ch.NextSibling <> nil do begin
      ch := ch.NextSibling;
      ch._Parent := pp;
    end;
    ch._NextSibling := p.NextSibling;
    if ch.NextSibling <> nil then
      ch.NextSibling._PrevSibling := ch;
  //delete zones
    z._FirstChild := nil;
    p._Parent := nil; //don't unlink!
    p.Free;
  end;

var
  r: TRect;
begin
//propagate changes into parents!
//parents only can have zones, no controls
  repeat
    p := zone.Parent;
    if p = nil then
      exit; //reached top zone!
    zone.Free; //unlink
    zone := p;
  until zone.FirstChild <> nil;
(* cases:
  zone without parent (top) - exit
  zone with 0 children - excluded before
  zone with 1 child (ch),
    child is leaf -> looses orientation, move up
    child contains 1 child (cc) -> orientation fits, move child.child up
  zone with more children - exit
*)
  while (zone.Parent <> nil) do begin
    p := zone.Parent;
    ch := zone.FirstChild;
    if ch.NextSibling <> nil then begin
    //more than 1 child - check next level
    end else if ch.FirstChild = nil then begin
    //contains control, move up
      ch.ChildControl.DockOrientation := zone.Parent.Orientation;
      p.ReplaceChild(zone, ch); //move control up
      zone.Free;
    end else if ch.FirstChild.NextSibling = nil then begin
    //contains zone with 1 child: orientation fits, move up
      InlineKids(ch.FirstChild);
    end;
    zone := p;
  end;
//update parent zone, to close the gap
  Zone := p.FirstChild;
  while Zone <> nil do begin
    if Zone.NextSibling = nil then
      Zone.BR := p.BR; //resize last zone
    if Zone.ChildControl <> nil then begin
      r := Zone.GetBounds;
      AdjustDockRect(Zone.ChildControl, r);
      Zone.ChildControl.BoundsRect := r;
    end;
    zone := Zone.NextSibling;
  end;
end;

{$IFDEF wndproc}
procedure TEasyTree.WindowProc(var Message: TMessage);
var
  p: TPoint;
  zone: TEasyZone;
  HitTestValue: integer;
  r: TRect;
begin
  case Message.Msg of
  WM_MOUSEMOVE:
    begin
      p := SmallPointToPoint(TWMMouse(Message).Pos);
      Zone := InternalHitTest(P, HitTestValue);
      if (zone <> nil) and (HitTestValue = htborder) then begin
      end;
    end;
  WM_LBUTTONDOWN: //allow to undock. Resize???
    begin
      P := SmallPointToPoint(TWMMouse(Message).Pos);
      Zone := InternalHitTest(P, HitTestValue);
      if zone <> nil then begin
      {$IFDEF splitter}
        if (HitTestValue = htborder) and (zone.PrevSibling <> nil)
        and (zone.PrevSibling.ChildControl <> nil) then begin
          if FSplitter <> nil then begin
            zone := zone.PrevSibling;
            r := zone.GetBounds;
            if zone.ChildControl.DockOrientation = doVertical then begin
            //splitter below PrevSibling
              FSplitter.Align := alTop;
              r.Top := r.Bottom;
              inc(r.Bottom, 4); //width?
            end else begin
            //splitter right of sibling
              FSplitter.Align := alLeft;
              r.Left := r.Right;
              inc(r.Right, 4);
            end;
            FSplitter.BoundsRect := r;
            FSplitter.ResizeStyle := rsLine;
            FSplitter.Visible := true; //false where???
          end;
        end else
      {$ELSE}
      {$ENDIF}
        if zone.ChildControl <> nil then begin
          if (zone.ChildControl.DragKind = dkDock)
          and (zone.ChildControl.DragMode = dmAutomatic) then begin
            zone.ChildControl.BeginDrag(False);
          end;
        end; //zone<>nil
        exit;
      end;
    end;
  WM_SETCURSOR: //over splitter?
    begin
    {
      Zone := InternalHitTest(P, HitTestValue);
      if (zone <> nil) and (zone.PrevSibling <> nil) and (HitTestValue = htborder) then begin
      end;
    }
    end;
  end;
  if assigned(FOldWndProc) then
    FOldWndProc(Message);
end;

function TEasyTree.InternalHitTest(const MousePos: TPoint;
  out HTFlag: Integer): TEasyZone;
begin
  Result := FTopZone;
  while Result <> nil do begin
    if (MousePos.X > Result.Right) or (MousePos.Y > Result.Bottom) then
      Result := Result.NextSibling
    else if Result.FirstChild <> nil then
      Result := Result.FirstChild
    else begin
      break; //here?
    end;
  end;
//todo: exact zone part
  if Result = nil then
    HTFlag := HTNOWHERE
  else begin
    HTFlag := HTCAPTION; //control never detected???
{
  //detect splitter - affects PrevSibling!
    if (Result.PrevSibling <> nil) then begin
      if Result.ChildControl.DockOrientation = doVertical then begin
      //splitter near top
        if (Result.PrevSibling.Bottom - MousePos.Y) < 4 then
          HTFlag := HTBORDER; //if on splitter
      end else begin //splitter near left
      //splitter near left
        if (Result.PrevSibling.Right - MousePos.x) < 4 then
          HTFlag := HTBORDER; //if on splitter
      end;
      //HTFlag := HTBORDER; //if on splitter
    end;
}
  end;
end;
{$ELSE}
  //todo: Lazarus handler
{$ENDIF}

{ TEasyZone }

constructor TEasyZone.Create(ATree: TEasyTree);
begin
  _Tree := ATree;
end;

destructor TEasyZone.Destroy;
begin
  Clear;
  Parent := nil; //unlink
  inherited;
end;

procedure TEasyZone.Clear;
begin
//let parent care for updating its chain?
  while FirstChild <> nil do
    FirstChild.Free;
end;

function TEasyZone.DockSite: TWinControl;
begin
  Result := _Tree.FDockSite;
end;

function TEasyZone.GetHeaderSize: integer;
begin
  Result := 10;  //override in TLazDockZone
end;

function TEasyZone.GetVisible: boolean;
begin
  Result := GetVisibleControl <> nil;
end;

function TEasyZone.GetVisibleControl: TChildControl;
var
  zone: TEasyZone;
begin
  Result := ChildControl;
  if (ChildControl <> nil) then begin
    if not ChildControl.Visible then
      Result := nil;
    exit;
  end;
//search children
  zone := FirstChild;
  while zone <> nil do begin
    Result := zone.GetVisibleControl;
    if Result <> nil then
      exit;
    zone := zone.NextSibling;
  end;
end;

//-------------- basic linking ----------------

procedure TEasyZone.SetParent(zone: TEasyZone);
begin
//for public use! unlink if parent changes.
  if zone <> Parent then begin
  //unlink, handle FirstChild
    if PrevSibling <> nil then
      PrevSibling._NextSibling := NextSibling
    else if Parent <> nil then
      Parent._FirstChild := NextSibling;
    if NextSibling <> nil then
      NextSibling._PrevSibling := PrevSibling;
    _Parent := zone;
    _NextSibling := nil;
    _PrevSibling := nil;
  end;
end;

procedure TEasyZone.InsertAfter(LinkAfter, NewZone: TEasyZone);
begin
//LinkAfter=nil for insert as first child.
  NewZone.Parent := self; //unlink if required
  if LinkAfter = nil then begin //as first child
    NewZone._NextSibling := FirstChild;
    if FirstChild <> nil then
      FirstChild._PrevSibling := NewZone;
    _FirstChild := NewZone;
  end else begin
    NewZone._PrevSibling := LinkAfter;
    NewZone._NextSibling := LinkAfter.NextSibling;
    if LinkAfter.NextSibling <> nil then
      LinkAfter.NextSibling._PrevSibling := NewZone;
    LinkAfter._NextSibling := NewZone;
  end;
  //NewZone.Orientation := LinkAfter.Orientation;
end;

procedure TEasyZone.AddSibling(NewZone: TEasyZone; where: TAlign);
var
  LinkAfter: TEasyZone;
  r, r2: TRect;
  NewOrientation: TDockOrientation;
begin
//orientation is NOT checked!
  r := GetBounds; //valid old values
  case where of
  alLeft, alTop:  LinkAfter := PrevSibling;
  alRight, alBottom: LinkAfter := self;
  else assert(False, 'unhandled insertion');
    LinkAfter := nil; //must never happen!
  end;
  Parent.InsertAfter(LinkAfter, NewZone);
//resize?
  r2 := r;
  case where of
  alLeft:
    begin
      NewOrientation := doHorizontal;
      r.Left := (r.Left+r.Right) div 2;
      r2.Right := r.Left;
    end;
  alRight:
    begin
      NewOrientation := doHorizontal;
      r.Right := (r.Left+r.Right) div 2;
      r2.Left := r.Right;
    end;
  alTop:
    begin
      NewOrientation := doVertical;
      r.Top := (r.Bottom+r.Top) div 2;
      r2.Bottom := r.Top;
    end;
  alBottom:
    begin
      NewOrientation := doVertical;
      r.Bottom := (r.Bottom+r.Top) div 2;
      r2.Top := r.Bottom;
    end;
  else //keep compiler happy
    NewOrientation := doNoOrient;
  end;
//parent orientation? (if in rootzone)
  //if parent.Orientation = doNoOrient then
    Parent.Orientation := NewOrientation;
  if ChildControl <> nil then
    ChildControl.DockOrientation := NewOrientation;
  if NewZone.ChildControl <> nil then
    NewZone.ChildControl.DockOrientation := NewOrientation;
  SetBounds(r);
  NewZone.SetBounds(r2);
end;

procedure TEasyZone.ReplaceChild(OldChild, NewChild: TEasyZone);
begin
//usage: insert intermediate parent zone
  NewChild.Parent := nil; //unlink
  NewChild.Parent := Self;
  NewChild._NextSibling := OldChild.NextSibling;
  NewChild._PrevSibling := OldChild.PrevSibling;
  OldChild.Parent := nil; //unlink
  if NewChild.NextSibling <> nil then
    NewChild.NextSibling._PrevSibling := NewChild;
//handle FirstChild
  if NewChild.PrevSibling <> nil then
    NewChild.PrevSibling._NextSibling := NewChild
  else
    _FirstChild := NewChild;
//init size
  NewChild.Orientation := OldChild.Orientation; //propagate?
  NewChild.BR := OldChild.BR;
  //if NewChild.ChildControl <> nil then NewChild.ChildControl.DockOrientation := Orientation;
end;

//function TEasyZone.ScaleTo(rOld, rNew: TRect): integer;
procedure TEasyZone.ScaleTo(const ptOld, ptNew, ptOuter: TPoint);
var
  r: TRect;
  ch: TEasyZone;
begin
(* change all coordinates from old to new extent.
  Set exact values for last sibling.
  Must use floating point division, to keep rounding errors acceptable!
*)
{
  br.X := br.X * ptNew.X div ptOld.X;
  br.Y := br.Y * ptNew.Y div ptOld.Y;
  if NextSibling = nil then begin
}
  //exact boundary in parent orientation
  if (Parent = nil) or (Parent.Orientation = doNoOrient) then
    br := ptOuter
  else if Parent.Orientation = doVertical then begin
    br.X := ptOuter.X;
    if NextSibling = nil then
      br.Y := ptOuter.Y
    else
      br.Y := round(br.Y * ptNew.Y / ptOld.Y);
  end else begin
    br.Y := ptOuter.Y;
    if NextSibling = nil then
      br.X := ptOuter.X
    else
      br.X := round(br.X * ptNew.X / ptOld.X);
  end;

  if ChildControl <> nil then begin
    r := GetBounds;
    _Tree.AdjustDockRect(ChildControl, r);
    ChildControl.BoundsRect := r;
  end else begin
    ch := FirstChild;
    while ch <> nil do begin
      ch.ScaleTo(ptOld, ptNew, BR);
      ch := ch.NextSibling;
    end;
  end;
end;

procedure TEasyZone.SetControl(Control: TChildControl);
begin
//propagate orientation into control
  _ChildControl := Control;
  if Parent <> nil then begin
    //this test is obsolete now, no zone has the dock site as its control.

    Control.DockOrientation := Parent.Orientation; //also obsolete
    Control.Parent := _Tree.FDockSite; //unless already done by TControl?
  //what's required to allow for undocking???
  {$IFDEF WinChild}
    Control.BorderStyle := bsSizeToolWin; //experimental: TChildControl=TForm
  {$ELSE}
  {$ENDIF}
  {$IFDEF fpc}
    //why are DragMode and DragKind *protected* properties???
  {$ELSE}
    Control.DragKind := dkDock;
    Control.DragMode := dmAutomatic;
  {$ENDIF}
    Control.Enabled := True;
  end //else it's the dock site - don't touch! (see above)
end;

//----------- prev sibling based coordinates ----------

function TEasyZone.GetTopOrLeft(fTop: boolean): Integer;
var
  zone, prev: TEasyZone;
begin
// In a parent zone of vertical orientation the zone.PrevSibling.Right is zone.Left.
  zone := self;
  while zone.Parent <> nil do begin
    if (fTop = (zone.Parent.Orientation = doVertical)) then begin
      prev := zone.PrevSibling;
      while prev <> nil do begin
        if prev.Visible then begin
          if fTop then
            Result := prev.Bottom
          else
            Result := prev.Right;
          exit;
        end;
        prev := prev.PrevSibling;
      end;
    end;
    zone := zone.Parent;
  end;
//reached top zone
  Result := 0;
end;

function TEasyZone.GetLeft: Integer;
begin
  Result := GetTopOrLeft(False);
end;

function TEasyZone.GetTop: Integer;
begin
  Result := GetTopOrLeft(True);
end;

{$IFDEF old}
function TEasyZone.GetWidth: Integer;
begin
//shared width should not return zero?
  if Parent = nil then //root zone
    Result := Self.docksite.ClientWidth
  else if Parent.Orientation = doVertical then //shared extent
    Result := Parent.Right - Parent.Left //right=limit
  else if not Visible then
    Result := 0
  else
    Result := GetRight - GetLeft;
end;

function TEasyZone.GetHeight: Integer;
begin
//shared height should not return zero?
  if Parent = nil then //root zone
    Result := docksite.ClientHeight
  else if Parent.Orientation <> doVertical then //shared extent
    Result := Parent.Bottom - Parent.Top //bottom=limit
  else if not Visible then
    Result := 0
  else
    Result := Bottom - Top;
end;
{$ELSE}
{$ENDIF}

function TEasyZone.GetBounds: TRect;
begin
//return defined extent
  Result.Top := Top;
  Result.Left := Left;
  Result.BottomRight := BR;
end;

procedure TEasyZone.SetBounds(TLBR: TRect);
{
var
  n: integer;
  zone: TEasyZone;
}
begin
(* Zone cannot be the root zone. If so, ignore?
  Recurse into child zones.
*)
  BR := TLBR.BottomRight;
  if ChildControl <> nil then begin
    //if Parent <> nil then begin //exclude root zone!
      _Tree.AdjustDockRect(ChildControl, TLBR);
    {$IFDEF ChildForm}
      ChildControl.BorderStyle := bsSizeToolWin;
    {$ELSE}
    {$ENDIF}
      ChildControl.BoundsRect := TLBR;
      exit; //all done - unless we use controls for child zones!
    //end; //else root zone, continue
  end; //else empty root zone?

{$IFDEF new}
//propagate into child zones
  n := VisibleChildCount;
  if n = 0 then
    exit;

//the following should become a virtual method? AlignChildren(...)?

//here: space kids evenly
  NewLimit := NewLimit div n; //differs for last child!
  //zone := FirstVisibleChild;
  zone := FirstChild;
  while n > 1 do begin
    if Orientation = doHorizontal then
      TLBR.Right := TLBR.Left + NewLimit
    else
      TLBR.Bottom := TLBR.Top + NewLimit;
    if zone.Visible then begin
      zone.SetBounds(TLBR);
      if Orientation = doHorizontal then
        TLBR.Left := TLBR.Right
      else
        TLBR.Top := TLBR.Bottom;
      dec(n);
    end else if Orientation = doHorizontal then
      zone.FLimit := TLBR.Left
    else
      zone.FLimit := TLBR.Top;
    zone := zone.NextSibling;
  end;
//last child extends to our Limit
  if Orientation = doHorizontal then
    TLBR.Right := FLimit
  else
    TLBR.Bottom := FLimit;
  zone.SetBounds(TLBR);
//more invisible siblings?
  zone := zone.NextSibling;
  while zone <> nil do begin
    zone.FLimit := FLimit;
    zone := zone.NextSibling;
  end;
{$ELSE}
{$ENDIF}
end;

end.

