unit jQueryElements;

{ jQuery-elements for the fcl-web jQuery interface

  Copyright (C) 2009 Joost van der Sluis/CNOC

  This library is free software; you can redistribute it and/or modify it
  under the terms of the GNU Library General Public License as published by
  the Free Software Foundation; either version 2 of the License, or (at your
  option) any later version with the following modification:

  As a special exception, the copyright holders of this library give you
  permission to link this library with independent modules to produce an
  executable, regardless of the license terms of these independent modules,and
  to copy and distribute the resulting executable under terms of your choice,
  provided that you also meet, for each linked independent module, the terms
  and conditions of the license of that module. An independent module is a
  module which is not derived from or based on this library. If you modify
  this library, you may extend this exception to your version of the library,
  but you are not obligated to do so. If you do not wish to do so, delete this
  exception statement from your version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
  for more details.

  You should have received a copy of the GNU Library General Public License
  along with this library; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}


{$mode objfpc}{$H+}

{ $DEFINE CGIDEBUG}

interface

uses
  Classes, SysUtils, fpdatasetform, htmlwriter, htmlelements, fphtml, httpdefs, db,
  dom, xmlwrite, contnrs;

type

  { TColumn }

  TColumn = class(TCollectionItem)
  private
    FFieldName: String;
    FName: String;
    FReadOnly: boolean;
  published
    property Name : String read FName write FName;
    property FieldName : String read FFieldName write FFieldName;
    property ReadOnly : boolean read FReadOnly write FReadOnly;
  end;

  TOnCellSubmitEvent = procedure(AID : String; AColumn : TColumn; ANewValue : String) of object;

  { TColumns }

  TColumns = class (TCollection)
  private
    function GetItem(index : integer): TColumn;
    procedure SetItem(index : integer; const AValue: TColumn);
  public
    function FindColumnByName(AName : TComponentName) : TColumn;
    property Items [index : integer] : TColumn read GetItem write SetItem;
  end;

  { TjQueryPage }

  TjQueryGrid = class;

  TjQueryPage = class (TComponent)
  private
    FAjaxURL: String;
    FHeader: THTML_head;
    FInitScript: TStringList;
    FjQueryElementList : TFPObjectList;
  protected
    function FindjQueryElement(AName : TComponentName) : TjQueryGrid;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure AddjQueryElement(AnElement: THTMLContentProducer);
    procedure RemovejQueryElement(AnElement: THTMLContentProducer);
    procedure InitialiseHeader(AHeader: THTML_head);
    procedure FinalizeHeader;
    procedure HandleRequest(ARequest: TRequest; AResponse: TResponse; var Handled: boolean);
    property InitScript: TStringList read FInitScript;
    property Header: THTML_head read FHeader;
  published
    property AjaxURL: String read FAjaxURL write FAjaxUrl;
  end;

  { TjQueryGrid }

  TjQueryGrid = class (THTMLContentProducer)
  private
    FCaption: string;
    FColumns: TColumns;
    FDatasource: TDatasource;
    FOnCellSubmit: TOnCellSubmitEvent;
    FQueryPage: TjQueryPage;
    FReadOnly: boolean;
    FRecordsPerPage: integer;
    procedure SetColumns(const AValue: TColumns);
    procedure SetQueryPage(const AValue: TjQueryPage);
  protected
    function StartForm (aWriter : THTMLWriter) : THTMLCustomElement; virtual;
    procedure EndForm (aWriter : THTMLWriter); virtual;
    procedure SetName(const NewName: TComponentName); override;
  public
    constructor Create(AOwner : TComponent); override;
    destructor Destroy; override;
    function WriteContent (aWriter : THTMLWriter) : THTMLCustomElement; override;
    function HandleAjaxRequest(ARequest : TRequest; AResponse : TResponse) : Boolean; virtual;
  published
    property Columns : TColumns read FColumns write SetColumns;
    property DataSource : TDatasource read FDatasource write FDatasource;
    property RecordsPerPage : integer read FRecordsPerPage write FRecordsPerPage default 20;
    property ReadOnly : boolean read FReadOnly write FReadOnly;
    property Caption: string read FCaption write FCaption;
    property OnCellSubmit : TOnCellSubmitEvent read FOnCellSubmit write FOnCellSubmit;
    property jQueryPage : TjQueryPage read FQueryPage write SetQueryPage;
  end;

implementation

uses
{$ifdef CGIDEBUG}
     dbugintf,
{$endif}
     strutils, bufdataset,sqldb;

{ TjQueryGrid }

procedure TjQueryGrid.SetQueryPage(const AValue: TjQueryPage);
begin
  if FQueryPage=AValue then exit;
  if assigned(FQueryPage) then
    FQueryPage.RemovejQueryElement(self);
  if assigned(AValue) then
    AValue.AddjQueryElement(self);
  FQueryPage:=AValue;
end;

procedure TjQueryGrid.SetColumns(const AValue: TColumns);
begin
  FColumns.Assign(AValue);
end;

function TjQueryGrid.StartForm(aWriter: THTMLWriter): THTMLCustomElement;
var ATable : THTML_table;
    ADiv : THTML_div;
begin
  ATable := aWriter.Starttable;
  ATable.ID := self.Name;
  ATable.elementclass:= 'scroll';
  awriter.Endtable;

  ADiv := aWriter.StartDiv;
  ADiv.ID := self.Name+'_pager';
  ADiv.elementclass:= 'scroll';
  awriter.EndDiv;
end;

procedure TjQueryGrid.EndForm(aWriter: THTMLWriter);
var
    AQXBasics : TjQueryPage;
    i : integer;

const TableInitScript =
 'jQuery("#%compname%").jqGrid({' + LineEnding +
 '  datatype: ''xml'',' + LineEnding +
 '  mtype: ''GET'',' + LineEnding +
// '  colNames:[''Inv No'',''Date'', ''Amount'',''Tax'',''Total'',''Notes''],' + LineEnding +
 '  colModel :[' + LineEnding +
// '    {name:''invid'', index:''invid'', width:55},' + LineEnding +
// '    {name:''invdate'', index:''invdate'', width:90},' + LineEnding +
// '    {name:''amount'', index:''amount'', width:80, align:''right''},' + LineEnding +
// '    {name:''tax'', index:''tax'', width:80, align:''right''},' + LineEnding +
// '    {name:''total'', index:''total'', width:80, align:''right''},' + LineEnding +
// '    {name:''note'', index:''note'', width:150, sortable:false} ],' + LineEnding +
 '%columnsdefinition%'+
 '  ],'+ LineEnding +
 '  pager: jQuery(''#%compname%_pager''),' + LineEnding +
 '  rowList:[10,20,30],' + LineEnding +
 '  sortname: ''id'',' + LineEnding +
 '  sortorder: "desc",' + LineEnding +
 '  viewrecords: true,' + LineEnding +
 '  imgpath: ''themes/basic/images'',';


var ColumnsDefinition : TStringList;

begin
  ColumnsDefinition:=TSTringList.Create;
  try
    for i := 0 to Columns.Count -1 do with Columns.Items[i] do
      begin
      ColumnsDefinition.add('  { ');
      if FieldName<>'' then
        ColumnsDefinition.Add('    index:'+QuotedStr(FieldName)+',');
      if not ReadOnly then
        ColumnsDefinition.Add('    editable: true,');
      ColumnsDefinition.add('    name:'+QuotedStr(Name));
      if i<Columns.Count-1 then
        ColumnsDefinition.add('  },')
      else
        ColumnsDefinition.add('  }')
      end;

    AQXBasics:= nil;
    for i := 0 to owner.ComponentCount-1 do
      if owner.Components[i] is TjQueryPage then
        begin
        AQXBasics := TjQueryPage(Owner.Components[i]);
        Break;
        end;
    if assigned(AQXBasics) then
      begin
      AQXBasics.InitScript.add(StringsReplace(TableInitScript,['%compname%','%columnsdefinition%'],[self.Name,ColumnsDefinition.Text],[rfIgnoreCase,rfReplaceAll]));
      if not ReadOnly then
        begin
        AQXBasics.InitScript.add('  cellEdit: true,');
//        AQXBasics.InitScript.add('  cellsubmit: ''clientArray'',');
        AQXBasics.InitScript.add('  cellsubmit: ''remote'',');
        AQXBasics.InitScript.add('  cellurl: ' + QuotedStr(FQueryPage.AjaxURL + '&compname=' + Self.Name + '&cmd=cellsubmit') +',');
        end;
      AQXBasics.InitScript.add('  rowNum:'+IntToStr(FRecordsPerPage)+',');
      AQXBasics.InitScript.add('  url:'+ QuotedStr(FQueryPage.AjaxURL + '&compname=' + Self.Name)+',');
      AQXBasics.InitScript.add('  caption: ' +QuotedStr(FCaption));

      AQXBasics.InitScript.add('});');
      end;
  finally
    ColumnsDefinition.Free;
  end;
end;

procedure TjQueryGrid.SetName(const NewName: TComponentName);
begin
  inherited SetName(NewName);
  if FCaption='' then FCaption:=Name;
end;

constructor TjQueryGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FColumns := TColumns.Create(TColumn);
  FRecordsPerPage:=20;
end;

destructor TjQueryGrid.Destroy;
begin
  FColumns.Destroy;
  inherited Destroy;
end;

function TjQueryGrid.WriteContent(aWriter: THTMLWriter): THTMLCustomElement;
begin
  Result:=StartForm(aWriter);
  EndForm(aWriter);
end;

function TjQueryGrid.HandleAjaxRequest(ARequest: TRequest; AResponse: TResponse): Boolean;
var XmlDoc : TXMLDocument;
    rowsNode, rownode,tempnode : TDomNode;
    ColNr : integer;
    AField : TField;
    AColumn : TColumn;
    Command : String;
    id : string;
    FieldValue : String;
    i : integer;
begin
  Result := True;
  Command := ARequest.QueryFields.Values['cmd'];
  if SameText(Command,'cellsubmit') then
    begin
    for i := 0 to ARequest.ContentFields.Count-1 do
      begin
      if sametext(ARequest.ContentFields.Names[i],'id') then
        id := ARequest.ContentFields.ValueFromIndex[i]
      else
        begin
        AColumn := Columns.FindColumnByName(ARequest.ContentFields.Names[i]);
        if assigned(AColumn) then
          begin
          if AColumn.FieldName<>'' then
            begin
            AField := FDatasource.DataSet.FindField(AColumn.FieldName);
            end;
          FieldValue := ARequest.ContentFields.ValueFromIndex[i];
          end;
        end
      end;
    assert(id<>'');
    assert(assigned(AColumn));
    if assigned(OnCellSubmit) then
      OnCellSubmit(id,AColumn, FieldValue);
    end
  else if assigned(FDatasource) then
    begin
    AResponse.ContentType := 'text/xml';
    XmlDoc := TXMLDocument.Create;
    try
      rowsNode := XmlDoc.CreateElement('rows');
      XmlDoc.AppendChild(rowsNode);

      FDatasource.DataSet.First;
      while not FDatasource.DataSet.EOF do
        begin
        rownode := XmlDoc.CreateElement('row');
        TDomElement(rownode).SetAttribute('id',IntToStr(FDatasource.DataSet.RecNo));
        for ColNr:=0 to Columns.Count-1 do
          begin
          tempnode := XmlDoc.CreateElement('cell');
          if Columns.items[ColNr].FieldName <> '' then
            begin
            AField := FDatasource.DataSet.FieldByName(Columns.items[ColNr].FieldName);
            assert(assigned(AField));
            tempnode.AppendChild(XmlDoc.CreateTextNode(AField.asstring));
            end;
          rownode.AppendChild(tempnode);
          end;

        rowsNode.AppendChild(rownode);
        FDatasource.DataSet.Next;
        end;

      tempnode := XmlDoc.CreateElement('page');
      tempnode.AppendChild(XmlDoc.CreateTextNode('1'));
      rowsNode.AppendChild(tempnode);

      tempnode := XmlDoc.CreateElement('total');
      tempnode.AppendChild(XmlDoc.CreateTextNode(IntToStr(Trunc(FDatasource.DataSet.RecordCount/FRecordsPerPage)+1)));
      rowsNode.AppendChild(tempnode);

      tempnode := XmlDoc.CreateElement('records');
      tempnode.AppendChild(XmlDoc.CreateTextNode(IntToStr(FDatasource.DataSet.RecordCount)));
      rowsNode.AppendChild(tempnode);

      AResponse.ContentStream := TMemoryStream.Create;
      WriteXML(XmlDoc,AResponse.ContentStream);
    finally
      XmlDoc.Free;
    end;
    result:= true;
    end;
end;

{ TjQueryPage }

function TjQueryPage.FindjQueryElement(AName: TComponentName): TjQueryGrid;
var i : integer;
begin
  Result := nil;
  for i := 0 to FjQueryElementList.Count-1 do
    if SameText(TComponent(FjQueryElementList.items[i]).Name,AName) then
      begin
      result := tjQueryGrid(FjQueryElementList.Items[i]);
      break;
      end
end;

constructor TjQueryPage.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FjQueryElementList := TFPObjectList.Create(False);
  FInitScript := TStringList.Create;
end;

destructor TjQueryPage.Destroy;
var i : integer;
begin
  for i := FjQueryElementList.Count-1 downto 0 do
    if assigned(FjQueryElementList.Items[i]) then
      TjQueryGrid(FjQueryElementList.Items[i]).jQueryPage := nil;
  FjQueryElementList.Free;
  FInitScript.Free;
  inherited Destroy;
end;

procedure TjQueryPage.AddjQueryElement(AnElement: THTMLContentProducer);
begin
  assert(FjQueryElementList.IndexOf(AnElement)=-1);
  FjQueryElementList.Add(AnElement);
end;

procedure TjQueryPage.RemovejQueryElement(AnElement: THTMLContentProducer);
begin
  assert(FjQueryElementList.IndexOf(AnElement)>-1);
  FjQueryElementList.Remove(AnElement);
end;

procedure TjQueryPage.InitialiseHeader(AHeader: THTML_head);
var AWriter : THTMLwriter;
begin
  assert(assigned(AHeader));
  FHeader := AHeader;
  AWriter:= THTMLwriter.create(AHeader.OwnerDocument as THTMLDocument);
  try
    AWriter.CurrentElement := AHeader;
    aWriter.Link('stylesheet','themes/basic/grid.css','text/css','screen');
    aWriter.Link('stylesheet','themes/jqModal.css','text/css','screen');
    awriter.Script('','text/javascript','jquery-1.3.2.js');
    awriter.Script('','text/javascript','jquery.jqGrid.js');
    awriter.Script('','text/javascript','js/jqModal.js');
    awriter.Script('','text/javascript','js/jqDnR.js');
    awriter.Script('','text/javascript','js/grid.celledit.js');
  finally
    AWriter.Destroy;
  end;
end;

procedure TjQueryPage.FinalizeHeader;
var AWriter : THTMLwriter;
    ScriptElement : THTML_script;
begin
  assert(assigned(FHeader));
  InitScript.Insert(0,'jQuery(document).ready(function(){');
  InitScript.Append('});');

  AWriter:= THTMLwriter.create(FHeader.OwnerDocument as THTMLDocument);
  try
    AWriter.CurrentElement := FHeader;
    ScriptElement := THTML_script.create(aWriter.Document);
    ScriptElement.AppendChild(THTML_Text.Create(awriter.Document)).NodeValue:=InitScript.Text;
    ScriptElement.thetype:='text/javascript';
  finally
    AWriter.Destroy;
  end;
  FHeader.AppendChild(ScriptElement);
end;

procedure TjQueryPage.HandleRequest(ARequest: TRequest; AResponse: TResponse; var Handled: boolean);
var AGrid : TjQueryGrid;
begin
  if handled then exit;
  AGrid := FindjQueryElement(ARequest.QueryFields.Values['compname']);
  if assigned(AGrid) then
    Handled := AGrid.HandleAjaxRequest(ARequest, AResponse);
end;

{ TColumns }

function TColumns.GetItem(index: integer): TColumn;
begin
  Result := TColumn(inherited GetItem(index));
end;

procedure TColumns.SetItem(index: integer; const AValue: TColumn);
begin
  Inherited SetItem(index,AValue);
end;

function TColumns.FindColumnByName(AName: TComponentName): TColumn;
var i : integer;
begin
  Result := Nil;
  for i := 0 to count -1 do if sametext(Items[i].Name,AName) then
    begin
    result := items[i];
    break;
    end;
end;

end.

