{ Win32 OLE drop source for DND of files to other applications

  Copyright (C) 2012 Bernd Kreuss <prof7bit@gmail.com>

  This library is free software; you can redistribute it and/or modify it
  under the terms of the GNU Library General Public License as published
  by the Free Software Foundation; either version 2 of the License, or (at
  your option) any later version with the following modification:

  As a special exception, the copyright holders of this library give you
  permission to link this library with independent modules to produce an
  executable, regardless of the license terms of these independent
  modules,and to copy and distribute the resulting executable under terms
  of your choice, provided that you also meet, for each linked independent
  module, the terms and conditions of the license of that module. An
  independent module is a module which is not derived from or based on
  this library. If you modify this library, you may extend this exception
  to your version of the library, but you are not obligated to do so. If
  you do not wish to do so, delete this exception statement from your
  version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library
  General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

{ On Windows we need to initiate the dragging manually. Also the drag
  source is not connected to any particular control or window as far as
  windows is concerned. Therefore the LCL component that wishes to act
  as a source needs to detect the initiating mouse moves on its own
  and then call StartDragManually(). This function will then set up the
  needed data structures and call DoDragDrop() which will enter a modal
  event loop and not return until drag/drop operation is finished.
}
unit DragDropWin32;

{$mode objfpc}{$H+}

interface
uses
  FileDragSource;

function DragStartsAutomatically: Boolean;
procedure StartDragManually(Src: TFileDragSource);


implementation
uses
  Classes,
  Windows,
  ActiveX,
  shlobj;

const
  MyFileDragFormat: FORMATETC = (
    CfFormat : CF_HDROP;
    Ptd      : nil;
    dwAspect : DVASPECT_CONTENT;
    lindex   : -1;
    tymed    : TYMED_HGLOBAL;
  );

type

  { TDropSource }

  TDropSource = class(TInterfacedObject, IDropSource)
  private
    FSrc: TFileDragSource;
  public
    constructor Create(Src: TFileDragSource);
    function QueryContinueDrag(fEscapePressed: BOOL; grfKeyState: DWORD): HResult; StdCall;
    function GiveFeedback(dwEffect: DWORD): HResult; StdCall;
  end;

  { TDataObject }

  TDataObject = class(TInterfacedObject, IDataObject)
  private
    FSrc: TFileDragSource;
  public
    constructor Create(Src: TFileDragSource);
    function GetData(const formatetcIn: FORMATETC; out medium: STGMEDIUM): HRESULT; STDCALL;
    function GetDataHere(const pformatetc: FormatETC; out medium: STGMEDIUM): HRESULT; STDCALL;
    function QueryGetData(const pformatetc: FORMATETC): HRESULT; STDCALL;
    function GetCanonicalFormatEtc(const pformatetcIn: FORMATETC; out pformatetcOut: FORMATETC): HResult; STDCALl;
    function SetData(const pformatetc: FORMATETC; const medium: STGMEDIUM; FRelease: BOOL): HRESULT; StdCall;
    function EnumFormatEtc(dwDirection: DWord; out enumformatetcpara: IENUMFORMATETC): HRESULT; StdCall;
    function DAdvise(const formatetc: FORMATETC; advf: DWORD; const AdvSink: IAdviseSink; out dwConnection: DWORD): HRESULT; StdCall;
    function DUnadvise(dwconnection: DWord): HRESULT; StdCall;
    function EnumDAdvise(out enumAdvise: IEnumStatData): HResult; StdCall;
  private
    function HaveThisFormat(const f: TFORMATETC): Boolean;
  end;

{$note SHCreateStdEnumFmtEtc() definition in shlobj is wrong, report this bug}
function SHCreateStdEnumFmtEtc(cfmt:UINT; afmt: PFORMATETC; var ppenumFormatEtc:IEnumFORMATETC):HRESULT;StdCall;external 'shell32' name 'SHCreateStdEnumFmtEtc';

function DragStartsAutomatically: Boolean;
begin
  Result := False;
end;

procedure StartDragManually(Src: TFileDragSource);
var
  DataObject: IDataObject;
  DropSource: IDropSource;
  DWEffect: DWord;
begin
  Src.CallOnDragBegin;
  DataObject := TDataObject.Create(Src);
  DropSource := TDropSource.Create(Src);
  DoDragDrop(DataObject, DropSource, DROPEFFECT_COPY, @DWEffect);
  Src.CallOnDragEnd;
end;

{ TDataObject }

constructor TDataObject.Create(Src: TFileDragSource);
begin
  FSrc := Src;
  inherited Create;
end;

function TDataObject.GetData(const formatetcIn: FORMATETC; out medium: STGMEDIUM): HRESULT; STDCALL;
var
  FileList: TStringList;
  FileName: String;
  sFileList: WideString;
  BufLen: PtrInt;
  hgDropFiles: THANDLE;
  pgDropFiles: PDROPFILES;

begin
  // This method may be called multiple times amd also even when no
  // drop happens at all because I am using CF_HDROP which means: files
  // exist in the file system and so windows thinks it is ok to access
  // the dragged data immediately after dragging has begun!

  if HaveThisFormat(formatetcIn) then begin

    FileList := TStringList.Create;
    FSrc.CallOnDragGetData(FileList);

    // First we need a widestring #0 sepatated and #0#0 at the end.
    sFileList := '';
    for FileName in FileList do begin
      sFileList += UTF8Decode(FileName) + #0;
    end;
    sFileList += #0;
    FileList.Free;

    // now we need to allocate memory for the DROPFILES structure
    // we need room for that structure plus the above widestring
    BufLen := SizeOf(DROPFILES) + 2*Length(sFileList); //widestring!
    hgDropFiles := GlobalAlloc(GMEM_MOVEABLE or GMEM_DDESHARE or GMEM_ZEROINIT, BufLen);

    // populate the DROPFILES structure,
    // it has the string of filenames appended directly at the end
    pgDropFiles := GlobalLock(hgDropFiles);
    pgDropFiles^.pFiles := SizeOf(DROPFILES); // offset of the file list
    pgDropFiles^.fWide := True; // contains widestring!
    Move(sFileList[1], pgDropFiles[1], 2*Length(sFileList)); // widestring!
    GlobalUnlock(hgDropFiles);

    // populate the STGMEDIUM structore;
    medium.Tymed := TYMED_HGLOBAL;
    medium.HGLOBAL := hgDropFiles;
    medium.PUnkForRelease := nil;

    Result := S_OK;
  end
  else
    Result := DV_E_FORMATETC;
end;

function TDataObject.GetDataHere(const pformatetc: FormatETC; out medium: STGMEDIUM): HRESULT; STDCALL;
begin
end;

function TDataObject.QueryGetData(const pformatetc: FORMATETC): HRESULT; STDCALL;
begin
  if HaveThisFormat(pformatetc) then
    Result := S_OK
  else
    Result := DV_E_FORMATETC;
end;

function TDataObject.GetCanonicalFormatEtc(const pformatetcIn: FORMATETC; out pformatetcOut: FORMATETC): HResult; STDCALl;
begin
end;

function TDataObject.SetData(const pformatetc: FORMATETC; const medium: STGMEDIUM; FRelease: BOOL): HRESULT; StdCall;
begin
end;

function TDataObject.EnumFormatEtc(dwDirection: DWord; out enumformatetcpara: IENUMFORMATETC): HRESULT; StdCall;
begin
  if dwDirection = DATADIR_GET then
    Result := SHCreateStdEnumFmtEtc(1, @MyFileDragFormat, enumformatetcpara)
  else
    Result := E_NOTIMPL;
end;

function TDataObject.DAdvise(const formatetc: FORMATETC; advf: DWORD; const AdvSink: IAdviseSink; out dwConnection: DWORD): HRESULT; StdCall;
begin
  Result := OLE_E_ADVISENOTSUPPORTED;
end;

function TDataObject.DUnadvise(dwconnection: DWord): HRESULT; StdCall;
begin
  Result := OLE_E_ADVISENOTSUPPORTED;
end;

function TDataObject.EnumDAdvise(out enumAdvise: IEnumStatData): HResult; StdCall;
begin
  Result := OLE_E_ADVISENOTSUPPORTED;
end;

function TDataObject.HaveThisFormat(const f: TFORMATETC): Boolean;
begin
  if (f.tymed = MyFileDragFormat.tymed)
  and (f.CfFormat = MyFileDragFormat.CfFormat)
  and (f.dwAspect = MyFileDragFormat.dwAspect) then
    Result := True
  else
    Result := False;
end;

{ TDragSource }

constructor TDropSource.Create(Src: TFileDragSource);
begin
  FSrc := Src;
  inherited Create;
end;

function TDropSource.QueryContinueDrag(fEscapePressed: BOOL; grfKeyState: DWORD): HResult; StdCall;
begin
  // if the Escape key has been pressed since the last call, cancel the drop
  if fEscapePressed = True then
    exit(DRAGDROP_S_CANCEL);

  // if the LeftMouse button has been released, then do the drop!
  if (grfKeyState and MK_LBUTTON) = 0 then
    exit(DRAGDROP_S_DROP);

  // continue with the drag-drop
  Result := S_OK;
end;

function TDropSource.GiveFeedback(dwEffect: DWORD): HResult; StdCall;
begin
  Result := DRAGDROP_S_USEDEFAULTCURSORS;
end;

initialization
  OleInitialize(nil);
finalization
  OleUninitialize();
end.

