{ Component creates a drag source to drag files to other applications.

  Copyright (C) 2012 Bernd Kreuss <prof7bit@gmail.com>

  This library is free software; you can redistribute it and/or modify it
  under the terms of the GNU Library General Public License as published
  by the Free Software Foundation; either version 2 of the License, or (at
  your option) any later version with the following modification:

  As a special exception, the copyright holders of this library give you
  permission to link this library with independent modules to produce an
  executable, regardless of the license terms of these independent
  modules,and to copy and distribute the resulting executable under terms
  of your choice, provided that you also meet, for each linked independent
  module, the terms and conditions of the license of that module. An
  independent module is a module which is not derived from or based on
  this library. If you modify this library, you may extend this exception
  to your version of the library, but you are not obligated to do so. If
  you do not wish to do so, delete this exception statement from your
  version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library
  General Public License for more details.

  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

{ Create an instance of this component, set the control you want to
  act as drag source, set the event method OnDragGetData. If DragGetData
  is called you must add file names of existing files to the stringlist
  with absolute paths and then the other application will attempt to
  access these files. Note that this function might get called multiple
  times during the drag process, even if the drag is not yet completed.

  It does only work if you set the control that you want to act as source
  to DragMode=dmManual, the internal LCL Drag mechanism is incompatible.

  This is not yet complete, features are still missing and also it works
  only with GTK2 and Win32 at the moment. I made this as a quick and
  pragmatic hack to make my application work. Ideally the methods and
  properties of this component should be made a part of the LCL TWinControl
  class itself and the widgetset specific code should go to lcl/interfaces/
}
unit FileDragSrc;

{$mode objfpc}{$H+}

interface
uses
  Classes,
  Controls;

type
  TDataEvent = procedure(Sender: TObject; DragData: TStringList) of Object;

  { TFileDragSource }

  TFileDragSource = class(TComponent)
    constructor Create(AOwner: TComponent); override;
  protected
    FControl: TWinControl;
    FDataEvent: TDataEvent;
    FBeginEvent: TNotifyEvent;
    FEndEvent: TNotifyEvent;
    FOldMouseDown: TMouseEvent;
    FOldMouseMove: TMouseMoveEvent;
    FButtonDownPos: TPoint;
    FIsDragging: Boolean;
    procedure UnsetControl;
    procedure SetControl(AControl: TWinControl);
    procedure MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure MouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
    procedure InstallMouseRedirection;
    procedure UninstallMouseRedirection;
  public
    InternalData: TObject; // may store some widgetset dependent state here
    procedure CallOnDragBegin;
    procedure CallOnDragGetData(Data: TStringList);
    procedure CallOnDragEnd;
    property IsDragging: Boolean read FIsDragging;
  published
    property Control: TWinControl read FControl write SetControl;
    property OnDragGetData: TDataEvent read FDataEvent write FDataEvent;
    property OnDragBegin: TNotifyEvent read FBeginEvent write FBeginEvent;
    property OnDragEnd: TNotifyEvent read FEndEvent write FEndEvent;
  end;

procedure Register;

implementation
uses
  LResources,
  typinfo,
  DragDropDummy,
  {$ifdef LCLGTK2}DragDropGtk2;{$endif}
  {$ifdef WIN32}DragDropWin32;{$endif}

{ TFileDragSource }

constructor TFileDragSource.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FControl := nil;
  FDataEvent := nil;
  FBeginEvent := nil;
  FEndEvent := nil;
  FButtonDownPos := Point(-1, -1);
  FIsDragging := False;
end;

procedure TFileDragSource.UnsetControl;
begin
  if FControl = nil then
    exit;
  UninstallMouseRedirection;
  UnmakeDragSource(self);
  FControl := nil;
end;

procedure TFileDragSource.SetControl(AControl: TWinControl);
begin
  if FControl = AControl then
    exit;
  if Assigned(FControl) then
    UnsetControl;
  FControl := AControl;
  MakeDragSource(Self);
  InstallMouseRedirection;
end;

procedure TFileDragSource.MouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Button = mbLeft then
    FButtonDownPos := Point(X, Y)
  else
    FButtonDownPos := Point(-1, -1);
  if Assigned(FOldMouseDown) then
    FOldMouseDown(Sender, Button, Shift, X, Y);
end;

procedure TFileDragSource.MouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
var
  DragDist: Double;
begin
  if (FButtonDownPos.Y <> -1) and (FButtonDownPos.Y <> -1) then begin
    if ssLeft in Shift then begin
      DragDist := sqrt(sqr(X - FButtonDownPos.X) + sqr(Y - FButtonDownPos.Y));
      if DragDist > 5 then begin
        FButtonDownPos := Point(-1, -1);
        StartDrag(Self);
      end;
    end
    else
      FButtonDownPos := Point(-1, -1);
  end;
  if Assigned(FOldMouseMove) then
    FOldMouseMove(Sender, Shift, X, Y);
end;

procedure TFileDragSource.InstallMouseRedirection;
begin
  FOldMouseDown := TMouseEvent(GetMethodProp(Control, 'OnMouseDown'));
  FOldMouseMove := TMouseMoveEvent(GetMethodProp(Control, 'OnMouseMove'));
  SetMethodProp(Control, 'OnMouseMove', TMethod(@MouseMove));
  SetMethodProp(Control, 'OnMouseDown', TMethod(@MouseDown));
end;

procedure TFileDragSource.UninstallMouseRedirection;
begin
  SetMethodProp(Control, 'OnMouseMove', TMethod(FOldMouseMove));
  SetMethodProp(Control, 'OnMouseDown', TMethod(FOldMouseDown));
end;

procedure TFileDragSource.CallOnDragBegin;
begin
  FIsDragging := True;
  if Assigned(OnDragBegin) then
    OnDragBegin(Control);
end;

procedure TFileDragSource.CallOnDragGetData(Data: TStringList);
begin
  if Assigned(OnDragGetData) then
    OnDragGetData(Control, Data);
end;

procedure TFileDragSource.CallOnDragEnd;
begin
  FIsDragging := False;
  if Assigned(OnDragEnd) then
    OnDragEnd(Control);
end;

procedure Register;
begin
  RegisterComponents('System', [TFileDragSource]);
end;

initialization
  {$i filedragsrc.lrs}
end.

