unit uSimplelogger;
{
Logger en fichier texte.
Protg par un TEvent pour permettre le log en multithread
La procdure AddLog n'est pas protge vis--vis du multi-thread mais elle-mme se protge pour les
  autres threads de l'application
L'utilisation normale est:
   LockForWrite
   Add(...)
   Unlock

TODO: bloquer l'criture dans le log de l'extrieur, mais sans bloquer le log des lignes
  les mettre en attendant dans un STringList
  lors du dblocage, vider la stringlist dans le log avant de dbloquer l'accs
  metre un Timeout minimal de 10 secondes pour permettre cette opration

TODO: envoyer les lignes crites via un event  SyncConf qui serait en mode monitoring
}
interface

uses
  Classes,
{$IFDEF LINUX}
  unix,
  {$IFDEF UseCThreads}
  CThreads,
  {$ENDIF}
{$ENDIF}
  Sysutils,
  syncobjs;

type

  TErrorSeverity = (esInfo, esWarning, esError, esUnknown);
  TWriterMode = (wmAppend, wmCreate, wmRecycle, wmUnknown);
  TLogEventEx = procedure (const Level: Integer; const Mess: String; const Severity: TErrorSeverity) of object;

  { TSitaLogger }

  TSitaLogger = class (TObject)
  private
    FFilename: string;
    FCurrentTailleLog: Integer;
    FLogLevel: Integer;
    FLogFileSize: Integer;
    FLogSource: string;
    FEvent: TEvent;
    FCriticalSection: TCriticalSection;
    FLogTimes: Boolean;
    FConsoleTee: Boolean;
    FWriterMode: TWriterMode;
    //FLogToMonitor: TLogEventEx;

    procedure SetLogSource(const Value: string);
    procedure SetFilename(const Value: string);
  protected
    procedure AddToFile(const Text: string);
    procedure InternalAddLog(const ALevel: Integer; const Text: string; const Severity: TErrorSeverity);
    procedure DoRecycle;
  public
    constructor Create(AFileName: string);
    destructor Destroy; override;

    procedure Recycle;
    procedure Add(const ALevel: Integer; const Text: string; const Severity: TErrorSeverity=esInfo);
    function LockForWrite(const TimeoutMs: Integer): TWaitResult;
    procedure Unlock;
    procedure Clear;
    procedure AddStrings(const AList: TStrings);
    procedure SetSize(const ASize: string);

    property ConsoleTee: Boolean read FConsoleTee write FConsoleTee;
    property LogFileName: string read FFilename write SetFilename;
    property LogLevel: Integer read FLogLevel write FLogLevel;
    property LogFileSize: Integer read FLogFileSize write FLogFileSize;
    property LogTimes: Boolean read FLogTimes write FLogTimes;
    property Source: string read FLogSource write SetLogSource;
    property WriterMode: TWriterMode read FWriterMode write FWriterMode;

    //property LogToMonitor: TLogEventEx read FLogToMonitor write FLogToMonitor;
  end;

procedure AddLog(const ALevel: Integer; const Text: string; const Severity: TErrorSeverity; const TimeOut: Integer=5000);

var
  SitaLogger: TSitaLogger = nil;
{$ifdef FPC}
  CS: TRTLCriticalSection;
{$else}
  CS: TCriticalSection;
{$endif}

implementation

{ TSitaLogger }
function DecodeTaille(const ASource: string; out AMultiplicateur: string; out AMultiple: Int64): Integer;
var
  P: Integer;
  sTaille: string;
begin
  Result := 256;
  if ASource = EmptyStr then Begin
    AMultiplicateur := 'kb';
    AMultiple := 1024;
    Exit
  end;
  sTaille := EmptyStr;
  P := 1;
  // scan de la taille
  while (P <= Length(ASource)) do begin
    if (UpperCase(ASource[P]) >= '0') and (UpperCase(ASource[P]) <= '9') then begin
      sTaille := sTaille + ASource[P];
      Inc(P)
    end else begin
      // fin des chiffrres on prend le multiplicateur
      AMultiplicateur := Copy(ASource, P, Length(ASource) - P + 1);
      Break;
    end
  end;
  Result := StrToIntDef(sTaille, -1);
  if Result = -1 then begin
    AMultiplicateur := 'kb';
    AMultiple := int64(1024);
    Result := 256;
    Exit
  end;
  // choisir le multiplicateur dans la box
  AMultiplicateur := TrimLeft(AMultiplicateur);
  if UpperCase(AMultiplicateur) = EmptyStr then
    AMultiple := int64(1)
  else
  if (UpperCase(AMultiplicateur) = 'KB') or (UpperCase(AMultiplicateur) = 'K') then
    AMultiple := int64(1024)
  else
  if (UpperCase(AMultiplicateur) = 'MB') or (UpperCase(AMultiplicateur) = 'M') then
    AMultiple := int64(1024) * int64(1024)
  else
  if (UpperCase(AMultiplicateur) = 'GB') or (UpperCase(AMultiplicateur) = 'G') then
    AMultiple := int64(1024) * int64(1024) * int64(1024)
  else
    AMultiple := 1;
end;

procedure AddLog(const ALevel: Integer; const Text: string; const Severity: TErrorSeverity; const TimeOut: Integer=5000);
var
  Msg: string;
{$ifdef FPC}
  Retries: Integer=10;
{$else}
  Retries: Integer;
{$endif}
begin
  // cette procdure ne peut tre utilise que par le thread principal de l'application
  // les autres threads doivent avoir chacun la leur
{$ifndef FPC}
  Retries := 10;
{$endif}
  if not Assigned(SitaLogger) then
    Exit;
{$ifdef FPC}
  EnterCriticalsection(CS);
{$else}
  CS.Enter;
{$endif}
  try
    while Retries > 0 do begin
      if SitaLogger.LockForWrite(TimeOut) = wrSignaled then
        try
          SitaLogger.Add(ALevel, Text, Severity);
          Retries := 0
        finally
          SitaLogger.Unlock
        end
      else begin
        Dec(Retries)
        //Msg := 'Unable to lock service log file for writing';
        //raise Exception.CreateFmt('%s Error %d:%s', ['Log', 23, Msg]);
      end
    end
  finally
{$ifdef FPC}
    LeaveCriticalsection(CS)
{$else}
    CS.Leave
{$endif}
  end
end;

constructor TSitaLogger.Create(AFileName: string);
begin
  Inherited Create;
  FEvent := TEvent.Create(nil, True, True, '');
  FEvent.SetEvent;
  FCriticalSection := TCriticalSection.Create;
  FFileName := AFileName;
  FLogFileSize := 10 * 1024 * 1024;
  FLogSource := 'Lg';
  FLogLevel := 5;
  FLogTimes := True;
  FConsoleTee := False;
  FWriterMode := wmAppend;
end;

destructor TSitaLogger.Destroy;
begin
  FEvent.SetEvent;
  FreeAndNil(FEvent);
  FCriticalSection.Leave;
  FreeAndNil(FCriticalSection);
  inherited;
end;

procedure TSitaLogger.Add(const ALevel: Integer; const Text: string; const Severity: TErrorSeverity);
var
  wText: string;
  wSeverity: string;
begin
  if Source = EmptyStr then
    raise Exception.Create('No source given for Log thread');
  if FFilename = EmptyStr then
    Exit;
  if ALevel > FLogLevel then
    Exit;
  try
    try
      case Severity of
        esError: wSeverity := ' !Erreur! ';
        esWarning: wSeverity := ' !Attention! ';
        else wSeverity := ' ';
      end;
      if FLogTimes then
        wText := FormatDateTime('dd/mm/yyyy hh:nn:ss', now) + wSeverity + Text
      else
        wText := Text;
      AddToFile(wText);
    except
    end;
  finally
    Unlock
  end;
  //if Assigned(FLogToMonitor) then
  //  FLogToMonitor(ALevel, wText, Severity);
end;

procedure TSitaLogger.AddToFile(const Text: string);
var
  F: TextFile;
  Taille: Integer;
  StrIn: TFileStream;
  StrW: TMemoryStream;
begin
  AssignFile(F, FFilename);
  if not FileExists(FFilename) then begin
    Rewrite(F);
    FCurrentTailleLog := 0;
    CloseFile(F);
  end;
  try
    System.Reset(F);
    Append(F);
    try
      Writeln(F, Text);
      Flush(F);
      CloseFile(F);
      FCurrentTailleLog := FCurrentTailleLog + Length(Text) + 2;
      // voir si le fichier log est trop plein
      if FCurrentTailleLog > FLogFileSize then begin
        // limite dpasse, raccourcir; compter une limite dans un fichier texte et retirer le dbut
        if FWriterMode = wmRecycle then begin
          Recycle
        end else begin
          if FileExists(FFilename) then begin
            StrIn := TFileStream.Create(FFilename, fmOpenRead);
            try
              Taille := StrIn.Size;
              // enlever la moiti de la taille
              StrIn.Position := Taille div 2;
              // stream intermdiaire
              StrW := TMemoryStream.Create;
              StrW.Position := 0;
              StrW.CopyFrom(StrIn, Taille div 2);
              // fermer la source
              FreeAndNil(StrIn);
              // rouvrir en output
              StrIn := TFileStream.Create(FFilename, fmCreate);
              StrW.Position := 0;
              // on copie tout le reste
              StrIn.CopyFrom(StrW, 0);
              // nouvelle taille courante
              FCurrentTailleLog := StrIn.Size;
              FreeAndNil(StrIn);
              FreeAndNil(StrW)
            finally
              if Assigned(StrIn) then
                FreeAndNil(StrIn);
              if Assigned(StrW) then
                FreeAndNil(StrW)
            end;
          end
        end
      end
    except
      CloseFile(F);
      if FConsoleTee then
        writeln(Text);
    end;
  except
    CloseFile(F);
  end
end;

procedure TSitaLogger.SetLogSource(const Value: string);
begin
  FLogSource := Value;
end;

function TSitaLogger.LockForWrite(const TimeoutMs: Integer): TWaitResult;
var
  WaitMs: Integer;
begin
  FCriticalSection.Enter;
  try
    WaitMs := TimeoutMs;
    if WaitMs < 15000 then
      WaitMs := 15000;
    Result := FEvent.WaitFor(WaitMs);
    if Result = wrSignaled then
      FEvent.ResetEvent;
  finally
    FCriticalSection.Leave
  end
  // event bloqu on sort avec timeout ou autre chose
end;

procedure TSitaLogger.Unlock;
begin
  FEvent.SetEvent
end;

procedure TSitaLogger.Clear;
begin
  If FileExists(FFilename) then begin
    LockForWrite(5000);
    try
      Sysutils.DeleteFile(FFilename);
    finally
      Unlock
    end
  end;
end;

procedure TSitaLogger.SetFilename(const Value: string);
begin
  FFilename := Value;
end;

procedure TSitaLogger.AddStrings(const AList: TStrings);
var
  Index: Integer;
begin
  if AList.Count = 0 then
    Exit;
  LockForWrite(5000);
  try
    for Index := 0 to AList.Count - 1 do
      AddToFile(AList[Index]);
  finally
    Unlock
  end
end;

procedure TSitaLogger.SetSize(const ASize: string);
var
  Multiplicateur: string;
  Multi: int64;
  nTaille: Integer;
begin
  // metre la taille selon la forme nnnxx (10MB)
  InternalAddLog(2, Format('Setting log size to %s', [ASize]), esInfo);
  nTaille := DecodeTaille(ASize, Multiplicateur, Multi);
  FLogFileSize := Integer(nTaille * Multi);
  if FLogFileSize < 128 * 1024 then
    FLogFileSize := 128 * 1024;
end;

procedure TSitaLogger.InternalAddLog(const ALevel: Integer; const Text: string; const Severity: TErrorSeverity);
var
  Msg: string;
begin
  if LockForWrite(10000) = wrSignaled then
    try
      Add(ALevel, Text, Severity)
    finally
      Unlock
    end;
  //else begin
  //  Msg := 'Unable to lock service log file for writing';
  //  raise Exception.CreateFmt('%s Error %d:%s', ['Com', 23, Msg]);
  //end
end;

procedure TSitaLogger.Recycle;
begin
  DoRecycle
end;

procedure TSitaLogger.DoRecycle;
var
  LogFile: String;
  wLogFile: String;
  wLogFile2: String;
  LogVersions: Integer;
begin
  LogFile := FFilename;
  if FileExists(LogFile) then begin
    // touver le dernier
    LogFile := ChangeFileExt(LogFile, '');
    for LogVersions := 0 to 10 do begin
      wLogFile := LogFile + Format('%2.2d', [LogVersions]) + '.log';
      if not FileExists(wLogFile) then
        Break;
    end;
    if LogVersions >= 10 then
      DeleteFile(wLogFile);
    wLogFile2 := LogFile + Format('%2.2d', [0]) + '.log';
    while LogVersions > 0 do begin
      wLogFile2 := LogFile + Format('%2.2d', [LogVersions]) + '.log';
      Dec(LogVersions);
      wLogFile := LogFile + Format('%2.2d', [LogVersions]) + '.log';
      DeleteFile(wLogFile2);
      RenameFile(wLogFile, wLogFile2);
    end;
    // le dernier: 00 -->> 01
    wLogFile2 := wLogFile;
    wLogFile := ChangeFileExt(ParamStr(0), '.log');
    wLogFile := StringReplace(wLogFile, 'WINCE', '', [rfIgnoreCase]);
    RenameFile(wLogFile, wLogFile2);
    FFilename := wLogFile
  end;
end;

initialization
  SitaLogger := TSitaLogger.Create('SitaLogger.txt');
{$ifdef FPC}
  InitCriticalSection(CS);
{$else}
  CS := TCriticalSection.Create
{$endif}

finalization
  if Assigned(SitaLogger) then
    FreeAndNil(SitaLogger);
{$ifdef FPC}
  DoneCriticalsection(CS)
{$else}
  FreeAndNil(CS)
{$endif}

end.

