﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Runtime.InteropServices;
using System.Diagnostics;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using FFmpeg.AutoGen;

namespace player.cs.console
{
	class Program
	{
		static void Main(string[] args)
		{
			play(args);
		}

		enum FrameType
		{
			FRAME_TYPE_HEADER,
			FRAME_TYPE_STREAM,
			FRAME_TYPE_METADATA,
			FRAME_TYPE_NOFRAME,
		};

		[StructLayout(LayoutKind.Sequential, Pack = 1)]
		unsafe struct RecordFrame
		{
			public Int64 timestamp;
			public Int32 frameType;
			public Int32 frameSize;
			public Int16 motionLevel;
			public Int32 flags;
			private fixed byte extraData[20];
		};

		[StructLayout(LayoutKind.Sequential, Pack = 1)]
		struct RecordHeader
		{
			public Int16 width;
			public Int16 height;
			public Int32 pixelFormat;
			public Int32 codecId;
			public Int32 timebaseNum;
			public Int32 timebaseDen;
		};

		static unsafe int play(string[] args)
		{
			var fileContent = File.ReadAllBytes("7711.brf");
			RecordFrame* pRecordFrame;
			RecordHeader* pRecordHeader;
			var pos = 0;
			fixed (byte* buf = fileContent)
			{
				pRecordFrame = (RecordFrame*)&buf[pos];
				pos += sizeof(RecordFrame);
				Debug.Assert(pRecordFrame->frameType == (int)FrameType.FRAME_TYPE_HEADER);
				Debug.Assert(pRecordFrame->frameSize == sizeof(RecordHeader));
				pRecordHeader = (RecordHeader*)&buf[pos];
				pos += sizeof(RecordHeader);
				var len = fileContent.Length;
				var firstTimestamp = pRecordFrame->timestamp;
				// in previous version of ffmpeg, value for AV_CODEC_ID_H264 has been 28
				AVCodecID codecId = (pRecordHeader->codecId == 28) ? AVCodecID.AV_CODEC_ID_H264 : (AVCodecID)pRecordHeader->codecId;
				AVCodec* pCodec = FFmpegInvoke.avcodec_find_decoder(codecId);
				Debug.Assert(pCodec != null);
				AVCodecContext* pCodecContext = FFmpegInvoke.avcodec_alloc_context3(pCodec);
				Debug.Assert(pCodecContext != null);
				AVBufferRef* hw_device_ctx;
				Debug.Assert(FFmpegInvoke.av_hwdevice_ctx_create(
					             &hw_device_ctx,
					             FFmpeg.AutoGen.AVHWDeviceType.AV_HWDEVICE_TYPE_DXVA2,
					             null, null, 0) == 0);
				pCodecContext->hw_device_ctx = hw_device_ctx;
				pCodecContext->codec_id = codecId;
				pCodecContext->time_base.num = pRecordHeader->timebaseNum;
				pCodecContext->time_base.den = pRecordHeader->timebaseDen;
				if ((pCodec->capabilities & FFmpegInvoke.CODEC_CAP_TRUNCATED) != 0)
					pCodecContext->flags |= FFmpegInvoke.CODEC_FLAG_TRUNCATED;
				pCodecContext->width = pRecordHeader->width;
				pCodecContext->height = pRecordHeader->height;
				pCodecContext->pix_fmt = (AVPixelFormat)pRecordHeader->pixelFormat;
				Debug.Assert(FFmpegInvoke.avcodec_open2(pCodecContext, pCodec, null) == 0);

				AVPacket packet;
				FFmpegInvoke.av_init_packet(&packet);

				SwsContext* pConvertContext = FFmpegInvoke.sws_getContext(
					pCodecContext->width, pCodecContext->height, pCodecContext->pix_fmt,
					pCodecContext->width, pCodecContext->height, AVPixelFormat.PIX_FMT_RGBA,
					FFmpegInvoke.SWS_FAST_BILINEAR, null, null, null);

				AVFrame* pFrame = FFmpegInvoke.av_frame_alloc();
				Debug.Assert(pFrame != null);
				int frameSize = FFmpegInvoke.avpicture_get_size(
					pCodecContext->pix_fmt, pRecordHeader->width, pRecordHeader->height);
				byte* pFrameBuffer = (byte*)FFmpegInvoke.av_malloc((uint)frameSize);
				FFmpegInvoke.avpicture_fill(
					(AVPicture*)pFrame, pFrameBuffer,
					pCodecContext->pix_fmt, pRecordHeader->width, pRecordHeader->height);

				AVFrame* pConvertedFrame = FFmpegInvoke.av_frame_alloc();
				Debug.Assert(pConvertedFrame != null);
				int convertedFrameSize = FFmpegInvoke.avpicture_get_size(
					AVPixelFormat.PIX_FMT_RGBA, pRecordHeader->width, pRecordHeader->height);
				byte* pConvertedFrameBuffer = (byte*)FFmpegInvoke.av_malloc((uint)convertedFrameSize);
				FFmpegInvoke.avpicture_fill(
					(AVPicture*)pConvertedFrame, pConvertedFrameBuffer,
					AVPixelFormat.PIX_FMT_RGBA, pRecordHeader->width, pRecordHeader->height);

				while (true)
				{
					while (pos != len)
					{
						var rf = (RecordFrame*)(buf + pos);
						pos += sizeof(RecordFrame);
						Debug.Assert(rf->frameType == (int)FrameType.FRAME_TYPE_STREAM);
						packet.data = buf;
						packet.size = rf->frameSize;
						Debug.Assert(packet.size > 0);
						packet.flags = rf->flags;
						packet.dts = packet.pts = rf->timestamp - firstTimestamp;
						packet.stream_index = 0;
						var ret = FFmpegInvoke.avcodec_send_packet(pCodecContext, &packet);
						Debug.Assert(ret == 0);
						Debug.Assert(FFmpegInvoke.avcodec_receive_frame(pCodecContext, pFrame) == 0);
						ret = FFmpegInvoke.sws_scale(
							pConvertContext,
							&pFrame->data_0, pFrame->linesize,
							0, pFrame->height,
							&pConvertedFrame->data_0, pConvertedFrame->linesize);
						Debug.Assert(ret == pFrame->height);
						var bTest = true;
						if (bTest)
						{
							var frameBitmap = new Bitmap(pRecordHeader->width, pRecordHeader->height, pConvertedFrame->linesize[0], PixelFormat.Format32bppPArgb, new IntPtr(pConvertedFrame->data_0));
							frameBitmap.Save("first-frame.jpg", ImageFormat.Jpeg);
							frameBitmap.Dispose();
							Console.ReadKey();
							return 0;
						}
						pos += packet.size;
					}
					pos = sizeof(RecordFrame) + sizeof(RecordHeader);
				}
				FFmpegInvoke.av_frame_free(&pFrame);
			}
			return 0;
		}
	}
}
