/* pipe raw audio data to UI
 *
 * Copyright (C) 2013 Robin Gareus <robin@gareus.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define _GNU_SOURCE

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>

#include "lv2/lv2plug.in/ns/lv2core/lv2.h"
#include "./uris.h"

typedef struct {
  float* input;
  float* output;
  LV2_Atom_Sequence* notify;

  LV2_URID_Map* map;
  RapLV2URIs uris;
  LV2_Atom_Forge forge;
  LV2_Atom_Forge_Frame frame;
} AudioPipe;

typedef enum {
  RAP_NOTIFY   = 0,
  RAP_INPUT    = 1,
  RAP_OUTPUT   = 2,
} PortIndex;


static LV2_Handle
instantiate(const LV2_Descriptor*     descriptor,
            double                    rate,
            const char*               bundle_path,
            const LV2_Feature* const* features)
{
  (void) descriptor; /* unused variable */
  (void) bundle_path; /* unused variable */

  AudioPipe* self = (AudioPipe*)calloc(1, sizeof(AudioPipe));
  if(!self) {
    return NULL;
  }

  int i;
  for (i=0; features[i]; ++i) {
    if (!strcmp(features[i]->URI, LV2_URID__map)) {
      self->map = (LV2_URID_Map*)features[i]->data;
    }
  }

  if (!self->map) {
    fprintf(stderr, "AudioPipe.lv2 error: Host does not support urid:map\n");
    free(self);
    return NULL;
  }

  lv2_atom_forge_init(&self->forge, self->map);

  map_rap_uris(self->map, &self->uris);
  return (LV2_Handle)self;
}

static void
connect_port(LV2_Handle handle,
             uint32_t   port,
             void*      data)
{
  AudioPipe* self = (AudioPipe*)handle;

  switch ((PortIndex)port) {
    case RAP_INPUT:
      self->input = (float*) data;
      break;
    case RAP_OUTPUT:
      self->output = (float*) data;
      break;
    case RAP_NOTIFY:
      self->notify = (LV2_Atom_Sequence*)data;
      break;
  }
}

static void
run(LV2_Handle handle, uint32_t n_samples)
{
  AudioPipe* self = (AudioPipe*)handle;
  const size_t size = sizeof(float) * n_samples;

  const uint32_t capacity = self->notify->atom.size;
  assert(capacity >= size + sizeof(LV2_Atom) * 4);

  lv2_atom_forge_set_buffer(&self->forge, (uint8_t*)self->notify, capacity);
  lv2_atom_forge_sequence_head(&self->forge, &self->frame, 0);
  lv2_atom_forge_frame_time(&self->forge, 0);

  lv2_atom_forge_atom(&self->forge, sizeof(LV2_Atom) + size, self->uris.atom_Vector);
  lv2_atom_forge_atom(&self->forge, sizeof(float), self->uris.atom_Float);
  lv2_atom_forge_raw(&self->forge, self->input, size);
  lv2_atom_forge_pad(&self->forge, size);

  lv2_atom_forge_pop(&self->forge, &self->frame);

  /* if not processing in-place, forward audio */
  if (self->input != self->output) {
    memcpy(self->output, self->input, size);
  }
}

static void
cleanup(LV2_Handle handle)
{
  free(handle);
}

static const void*
extension_data(const char* uri)
{
  return NULL;
}

static const LV2_Descriptor descriptor = {
  RAP_URI,
  instantiate,
  connect_port,
  NULL,
  run,
  NULL,
  cleanup,
  extension_data
};

LV2_SYMBOL_EXPORT
const LV2_Descriptor*
lv2_descriptor(uint32_t index)
{
  switch (index) {
  case 0:
    return &descriptor;
  default:
    return NULL;
  }
}

/* vi:set ts=8 sts=2 sw=2: */
