/* simple audio-pipe // scope
 *
 * Copyright (C) 2013 Robin Gareus <robin@gareus.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>

#include <gtk/gtk.h>
#include <cairo.h>

#include "lv2/lv2plug.in/ns/extensions/ui/ui.h"
#include "./uris.h"

#define DAWIDTH 640
#define DAHEIGHT 200
#define CYPOS(X) ( (X) * 100.0 + 100.5)

typedef struct {
  LV2_URID_Map* map;

  LV2UI_Write_Function write;
  LV2UI_Controller     controller;

  RapLV2URIs uris;

  GtkWidget* box;
  GtkWidget* darea;

  GtkWidget* spinner;
  GtkAdjustment *spinner_adj;

  float data[DAWIDTH];
  int   didx;

  int   stride;
  int   remain;
} AudioPipeUI;


gboolean expose_event_callback (GtkWidget *widget, GdkEventExpose *event, gpointer data) {
  AudioPipeUI* ui = (AudioPipeUI*) data;
  cairo_t *cr;
  cr = gdk_cairo_create(ui->darea->window);

  cairo_set_source_rgba (cr, .0, .0, .0, 1.0);
  cairo_rectangle(cr, 0, 0, DAWIDTH, DAHEIGHT);
  cairo_fill(cr);

  cairo_set_source_rgba (cr, .0, 1.0, .0, 1.0);
  cairo_set_line_width(cr, 1.0);

  /* this runs in gtk's main thread
   * TODO: read from ringbuffer or blit cairo surface
   */

  cairo_move_to(cr, 0, CYPOS(ui->data[0]));
  for (int i=1 ; i < DAWIDTH; ++i) {
    cairo_line_to(cr, i+.5, CYPOS(ui->data[i]));
  }
  cairo_stroke (cr);

  if (ui->stride > 20) {
    cairo_set_line_width(cr, 2.0);
    cairo_set_source_rgba (cr, .9, .2, .2, .6);
    cairo_move_to(cr, ui->didx + .5, 0);
    cairo_line_to(cr, ui->didx + .5, DAHEIGHT);
    cairo_stroke (cr);
  }
  cairo_destroy (cr);
  return TRUE;
}

static LV2UI_Handle
instantiate(const LV2UI_Descriptor*   descriptor,
            const char*               plugin_uri,
            const char*               bundle_path,
            LV2UI_Write_Function      write_function,
            LV2UI_Controller          controller,
            LV2UI_Widget*             widget,
            const LV2_Feature* const* features)
{
  AudioPipeUI* ui = (AudioPipeUI*)malloc(sizeof(AudioPipeUI));
  ui->map        = NULL;
  ui->write      = write_function;
  ui->controller = controller;
  ui->box        = NULL;
  ui->darea      = NULL;
  ui->didx       = 0;
  ui->stride     = 25;
  ui->remain     = 0;

  *widget = NULL;

  for (int i = 0; features[i]; ++i) {
    if (!strcmp(features[i]->URI, LV2_URID_URI "#map")) {
      ui->map = (LV2_URID_Map*)features[i]->data;
    }
  }

  if (!ui->map) {
    fprintf(stderr, "UI: Host does not support urid:map\n");
    free(ui);
    return NULL;
  }

  map_rap_uris(ui->map, &ui->uris);

  ui->box = gtk_hbox_new(FALSE, 4);
  ui->darea = gtk_drawing_area_new ();
  gtk_widget_set_size_request (ui->darea, DAWIDTH, DAHEIGHT);

  ui->spinner_adj = (GtkAdjustment *) gtk_adjustment_new (25.0, 1.0, 100.0, 1.0, 5.0, 0.0);
  ui->spinner = gtk_spin_button_new (ui->spinner_adj, 1.0, 0);

  gtk_box_pack_start(GTK_BOX(ui->box), ui->darea, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(ui->box), ui->spinner, FALSE, FALSE, 0);

  g_signal_connect (G_OBJECT (ui->darea), "expose_event", G_CALLBACK (expose_event_callback), ui);

  *widget = ui->box;

  return ui;
}

static void
cleanup(LV2UI_Handle handle)
{
  AudioPipeUI* ui = (AudioPipeUI*)handle;
  gtk_widget_destroy(ui->darea);
  free(ui);
}

static void
port_event(LV2UI_Handle handle,
           uint32_t     port_index,
           uint32_t     buffer_size,
           uint32_t     format,
           const void*  buffer)
{
  AudioPipeUI* ui = (AudioPipeUI*)handle;
  if (format == ui->uris.atom_eventTransfer) {
    LV2_Atom* atom = (LV2_Atom*)buffer;
    if (atom->type == ui->uris.atom_Vector) {
      LV2_Atom_Vector* vof = (LV2_Atom_Vector*)LV2_ATOM_BODY(atom);
      if (vof->atom.type == ui->uris.atom_Float) {
	size_t nelem = (atom->size - sizeof(LV2_Atom)) / vof->atom.size;
	float *data = (float*) LV2_ATOM_BODY(&vof->atom);

	/* TODO process/filter data depending on speed || trigger
	 * write into ringbuffer (!) OR draw a cairo-surface here
	 *
	 * this callback runs in the "communication" thread of the LV2-host
	 * usually a g_timeout() at ~25fps
	 */
	ui->stride = gtk_spin_button_get_value (GTK_SPIN_BUTTON(ui->spinner));
	/* XXX quick dirty hack, take every Nth sample */
	for (int i=ui->remain; i < nelem; i+= ui->stride) {
	  ui->data[ui->didx] = data[i];
	  ui->didx = (ui->didx + 1) % DAWIDTH;
	}
	ui->remain = (ui->remain + nelem) % ui->stride;

	/* signal gtk's main thread to redraw the widget
	 * TODO use  gtk_widget_queue_draw_area(ui->darea, x, y, w, h)
	 * to minimize redraw-area
	 */
	gtk_widget_queue_draw(ui->darea);
      }
    }
  }
}

static const void*
extension_data(const char* uri)
{
  return NULL;
}

static const LV2UI_Descriptor descriptor = {
  RAP_URI "#ui",
  instantiate,
  cleanup,
  port_event,
  extension_data
};

LV2_SYMBOL_EXPORT
const LV2UI_Descriptor*
lv2ui_descriptor(uint32_t index)
{
  switch (index) {
  case 0:
    return &descriptor;
  default:
    return NULL;
  }
}

/* vi:set ts=8 sts=2 sw=2: */
